// ort.h  (OpenRTM)
#ifndef _ORT_H_
#define _ORT_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <ctype.h>
#include <assert.h>

#ifdef _OPENMP
#include <omp.h>
#endif

#ifdef MAIN
#define EXTERN
#else
#define EXTERN extern
#endif

#define PROGRAM "OpenRTM"
#define VERSION_MAJOR (1)
#define VERSION_MINOR (0)
#define VERSION_BUILD (1)

#define PI   (4.0*atan(1.0))
#define DTOR (PI/180)
#define C    (2.99792458e8)
#define MU0  (4*PI*1e-7)
#define EPS0 (1/(C*C*MU0))
#define ETA0 (C*MU0)
#define EPS  (1e-10)
#define EPS2 (1e-20)

#define ARRAY_INC (10000)
#define MAXCHAR (4096)
#define MAXVERTEX (500)
#define MAXTOKEN (3*MAXVERTEX+4)

#define MAX(x,y) ((x) > (y) ? (x) : (y))
#define MIN(x,y) ((x) < (y) ? (x) : (y))

typedef struct {double r, i;} d_complex_t;
typedef struct {float  r, i;} f_complex_t;

// 計算

typedef struct {
	int    mtype;                   // データ種別(=1/2/3)
	double epsr, sigma;             // 比誘電率、導電率[S/m]
	double thick;                   // 厚さ [m]
	int nangle;                     // 以下ファイル、角度数
	double *angle;                  // 角度[度]
	d_complex_t *rv, *rh;           // 反射係数
	d_complex_t *tv, *th;           // 透過係数
} material_t;                       // 物性値

typedef struct {
	double v[4][3];                 // 頂点座標 [m]
	double n[3];                    // 単位法線ベクトル
	int    m;                       // 物性値番号(=0,1,...)
} triangle_t;                       // 三角形

typedef struct {
	int    atype;                   // 1=iso, 2=dipole, 3=beam, 4=file
	int    pol;                     // 1=V, 2=H, 3=RHCP, 4=LHCP
	double theta, phi;              // 中心のθφ [deg]
	double bw;                      // ダイポール、ビーム幅 [deg]
	double bwtheta, bwphi;          // ビーム、θφビーム幅 [deg]
	int    ntheta, nphi;            // ファイル、θφデータ数
	d_complex_t **etheta, **ephi;   // ファイル、パターン
	double gain;                    // 利得
} antenna_t;                        // アンテナ

typedef struct {
	double pos[3];                  // 座標 [m]
	double power, phase;            // power [W] and phase [deg]
	int    antenna;                 // アンテナ番号(=0,1,...)
} tx_t;                             // 送信点

typedef struct {
	double pos[3];                  // 座標 [m]
	int    antenna;                 // アンテナ番号(=0,1,...)
} rx0d_t;                           // 観測点

typedef struct {
	double pos[2][3];               // 始点と終点の座標 [m]
	int    div;                     // 分割数
	int    antenna;                 // アンテナ番号(=0,1,...)
} rx1d_t;                           // 観測線

typedef struct {
	double pos[4][3];               // 4頂点の座標 [m]
	int    div[2];                  // 分割数
	int    antenna;                 // アンテナ番号(=0,1,...)
} rx2d_t;                           // 観測面

typedef struct {
	int    npos;                    // 頂点数 (>1)
	double (*pos)[3];               // 頂点座標 [m]
	int    *itri;                   // 三角形番号(=0,1,...)
	double (*vector)[3];            // 線分の単位方向ベクトル
	double *length;                 // 線分の長さ [m]
	double *cumlng;                 // 送信点からの長さ [m]
} ray_t;                            // レイ

typedef struct {
	int    npos;                    // 頂点数 (>1)
	double (*pos)[3];               // 頂点座標 [m]
	int    *itri;                   // 三角形番号(=0,1,...)
	int    *rdt;                    // =1/2/3 : R/D/T (反射/回折/透過)
} path_t;                           // 伝搬経路

typedef struct {
	double pos[3];                  // 座標 [m]
	int    antenna;                 // アンテナ番号(=0,1,...)
	int    npath;                   // 伝搬経路数
	path_t *path;                   // 伝搬経路
	d_complex_t *efield;            // 受信電界(複素数) [√W]
	double power[2];                // 受信電力(位相差あり/なし) [W]
	double delay[2];                // 平均遅延、遅延スプレッド [m]
} rx_t;                             // 受信

EXTERN int        NMaterial;        // 物性値数 (>1)
EXTERN material_t *Material;        // 物性値

EXTERN int        NAntenna;         // アンテナ数
EXTERN antenna_t  *Antenna;         // アンテナ

EXTERN int        NTriangle;        // 三角形数
EXTERN triangle_t *Triangle;        // 三角形
EXTERN int        NEdge;            // 稜線数
EXTERN double     (*Edge)[2][3];    // 稜線
EXTERN int        **LosTxEdge;      // 送信点を三角形の間はLOSであるか(計算時間短縮)

EXTERN char       Title[256];       // タイトル
EXTERN double     Frequency;        // 周波数 [Hz]

EXTERN int        MaxPath;          // 最大伝搬経路数
EXTERN int        MaxRef;           // 最大反射回数
EXTERN int        NDivLaunch;       // ローンチング法、緯度分割数
EXTERN int        NDivAntenna;      // アンテナパターン、緯度分割数
EXTERN int        Icomp[4];         // 直接法の計算成分
EXTERN int        Log[2];           // 数値出力
EXTERN int        Plot3d[3];        // 形状出力3D

EXTERN int        NTx;              // 送信点数
EXTERN tx_t       *Tx;              // 送信点

EXTERN int        NRx0d;            // 観測点数
EXTERN rx0d_t     *Rx0d;            // 観測点

EXTERN int        NRx1d;            // 観測線数
EXTERN rx1d_t     *Rx1d;            // 観測線

EXTERN int        NRx2d;            // 観測面数
EXTERN rx2d_t     *Rx2d;            // 観測面

EXTERN int        NRx[4];           // 受信点数(観測点/観測線/観測面/合計)
EXTERN rx_t       *Rx;              // 受信

EXTERN int        NRay;             // レイ数
EXTERN ray_t      *Ray;             // レイ

EXTERN double     Lbound;           // 計算対象の大きさ

EXTERN int        NThread;          // スレッド数
EXTERN int        HTML;             // 1:HTML出力

// ポスト処理

typedef struct {
	int    user;                    // =0/1 : 自動スケール/ユーザースケール
	int    db;                      // =0/1 = dB ?
	double min, max;                // 最小値、最大値
	int    div;                     // 分割数
} scale_t;                          // スケール

EXTERN int        Rx0d_path;
EXTERN int        Rx0d_profile;
EXTERN scale_t    Rx0d_profile_xscale, Rx0d_profile_yscale;

EXTERN int        Rx1d_path;
EXTERN int        Rx1d_profile;
EXTERN int        Rx1d_power, Rx1d_delay;
EXTERN scale_t    Rx1d_power_scale, Rx1d_delay_scale;

EXTERN int        Rx2d_path;
EXTERN int        Rx2d_power[2], Rx2d_delay[2];
EXTERN int        Rx2d_stat[4];
EXTERN int        Rx2d_decay[2];
EXTERN scale_t    Rx2d_power_scale, Rx2d_delay_scale, Rx2d_decay_scale;
EXTERN int        Rx2d_geometry;

EXTERN double     Width2d, Height2d, Fontsize2d;

#ifdef __cplusplus
}
#endif

#endif  // _ORT_H_
