/*
plot2dRx0d.c
観測点の図形出力(2D)
*/

#include "ort.h"
#include "complex.h"
#include "vector.h"
#include "ort_prototype.h"
#include "ev.h"


// 観測点の遅延プロファイルを図形表示する(2D)
void plot2dRx0d(void)
{
	// レイアウト
	const double x1 = 0.12 * Width2d;
	const double y1 = 0.1  * Height2d;
	const double x2 = 0.9  * Width2d;
	const double y2 = 0.86 * Height2d;
	const double h = Fontsize2d;

	unsigned char rgb[3];
	char str[BUFSIZ];

	if ((NRx0d <= 0) || (NRx[0] <= 0)) return;

	// 観測点に関するループ(1観測点=1ページ)
	for (int irx = 0; irx < NRx0d; irx++) {
		rx_t rx = Rx[irx];

		// 伝搬経路数
		const int npath = rx.npath;

		// X軸データ(伝搬時間[nsec])
		double *xdata = (double *)malloc(npath * sizeof(double));
		for (int ipath = 0; ipath < npath; ipath++) {
			double lng = 0;
			for (int ipos = 0; ipos < rx.path[ipath].npos - 1; ipos++) {
				lng += vector_distance(rx.path[ipath].pos[ipos + 0],
				                       rx.path[ipath].pos[ipos + 1]);
			}
			xdata[ipath] = lng / C * 1e9;  // m -> nsec
			//printf("%d %d %e\n", irx, ipath, xdata[ipath]);
		}

		// Y軸データ(受信電力[dBW])
		double *ydata = (double *)malloc(npath * sizeof(double));
		for (int ipath = 0; ipath < npath; ipath++) {
			ydata[ipath] = 20 * log10(MAX(d_abs(rx.efield[ipath]), EPS));
		}

		// X軸スケール
		double dxmax = 0;
		for (int ipath = 0; ipath < npath; ipath++) {
			dxmax = MAX(dxmax, xdata[ipath]);
		}
		double xmax, xmin;
		int    xdiv;
		getscale(dxmax, Rx0d_profile_xscale, &xmin, &xmax, &xdiv);
		//printf("%e %e %e %d\n", dxmax, xmin, xmax, xdiv);

		// Y軸スケール
		double dymax = -1000;
		for (int ipath = 0; ipath < npath; ipath++) {
			dymax = MAX(dymax, ydata[ipath]);
		}
		double ymax, ymin;
		int    ydiv;
		getscale(dymax, Rx0d_profile_yscale, &ymin, &ymax, &ydiv);
		//printf("%e %e %e %d\n", dymax, ymin, ymax, ydiv);

		// 図形表示開始
		ev2d_newPage();

		// グリッド
		ev2dlib_grid(x1, y1, x2, y2, xdiv, ydiv);

		// 遅延プロファイル
		int npath0 = 0;  // 描画されたパス数
		for (int ipath = 0; ipath < npath; ipath++) {
			//printf("%d %f %f\n", ipath, xdata[ipath], ydata[ipath]);
			double x = x1 + (x2 - x1) * (xdata[ipath] - xmin) / (xmax - xmin);
			double y = y1 + (y2 - y1) * (ydata[ipath] - ymin) / (ymax - ymin);
			if ((x < x1) || (x > x2)) continue;
			if (y < y1) continue;
			y = MIN(y, y2);
			getpathcolor(rx.path[ipath].npos - 1, rgb);
			ev2d_setColor(rgb[0], rgb[1], rgb[2]);
			ev2d_drawLine(x, y1, x, y);
			npath0++;
		}
		ev2d_setColor(0, 0, 0);

		// free
		free(xdata);
		free(ydata);

		// Y軸ラベル
		sprintf(str, "%.1f", ymax);
		ev2d_drawString(x1 - 3.5 * h, y2 - 0.3 * h, h, str);
		sprintf(str, "%.1f", ymin);
		ev2d_drawString(x1 - 3.5 * h, y1 - 0.3 * h, h, str);
		ev2d_drawString(x1 - 4.0 * h, y2 - 2.0 * h, h, "[dBW]");

		// X軸ラベル
		ev2d_drawString((x1 + x2) / 2 - 3.0 * h, y1 - 1.2 * h, h, "time [nsec]");
		ev2d_drawString((x1 + x2) / 2 - 3.5 * h, y1 - 2.5 * h, h, "(length [m])");

		// X軸最小値
		sprintf(str, "%.3f[nsec]", xmin);
		ev2d_drawString(x1 - 1.0 * h, y1 - 1.2 * h, h, str);
		sprintf(str, "(%.3e[m])", xmin * C * 1e-9);
		ev2d_drawString(x1 - 1.5 * h, y1 - 2.5 * h, h, str);

		// X軸最大値
		sprintf(str, "%.3f[nsec]", xmax);
		ev2d_drawString(x2 - 4.5 * h, y1 - 1.2 * h, h, str);
		sprintf(str, "(%.3e[m])", xmax * C * 1e-9);
		ev2d_drawString(x2 - 5.0 * h, y1 - 2.5 * h, h, str);

		// タイトル
		strcpy(str, "Rx delay profile");
		ev2d_drawString(x1, y2 + 2.9 * h, h, str);
		sprintf(str, "Rx point #%d", irx + 1);
		ev2d_drawString(x1, y2 + 1.7 * h, h, str);
		ev2d_drawString(x1, y2 + 0.5 * h, h, Title);

		// 伝搬経路数
		sprintf(str, "No. of paths = %d/%d", npath0, npath);
		ev2d_drawString(x2 - 15.0 * h, y2 + 2.9 * h, h, str);

		// 受信電力
		const double p0 = 10 * log10(MAX(rx.power[0], EPS2));
		const double p1 = 10 * log10(MAX(rx.power[1], EPS2));
		sprintf(str, "power = %.3f / %.3f [dBW]", p0, p1);
		ev2d_drawString(x2 - 15.0 * h, y2 + 1.7 * h, h, str);

		// 遅延時間
		const double t0 = rx.delay[0] / C * 1e9;
		const double t1 = rx.delay[1] / C * 1e9;
		sprintf(str, "delay = %.3f / %.3f [nsec]", t0, t1);
		ev2d_drawString(x2 - 15.0 * h, y2 + 0.5 * h, h, str);
	}

	ev2d_setColor(0, 0, 0);
}
