/*
plot2dRx1d.c
観測線の図形出力(2D)
*/

#include "ort.h"
#include "ort_prototype.h"
#include "ev.h"


// 観測線の分布図を図形出力する
// kind = 0/1 : power/delay
void plot2dRx1d(int kind)
{
	const double x1 = 0.12 * Width2d;
	const double y1 = 0.1  * Height2d;
	const double x2 = 0.9  * Width2d;
	const double y2 = 0.86 * Height2d;
	const double h = Fontsize2d;
	const int xdiv = 10;

	char str[BUFSIZ];

	assert((kind == 0) || (kind == 1));
	if ((NRx1d <= 0) || (NRx[1] <= 0)) return;

	// 受信点の通し番号
	int irx = NRx[0];

	// 1観測線=1ページ
	for (int n = 0; n < NRx1d; n++) {
		// 観測線の分割数
		const int ndiv = Rx1d[n].div;

		// X data
		double *xdata = (double *)malloc(ndiv * sizeof(double));
		for (int idiv = 0; idiv < ndiv; idiv++) {
			xdata[idiv] = x1 + (x2 - x1) * (idiv + 0.5) / ndiv;
		}

		// Y data
		double *ydata[2];
		for (int comp = 0; comp < 2; comp++) {
			ydata[comp] = (double *)malloc(ndiv * sizeof(double));
		}
		for (int idiv = 0; idiv < ndiv; idiv++) {
			for (int comp = 0; comp < 2; comp++) {
				ydata[comp][idiv] = (kind == 0)
					? Rx[irx].power[comp]
					: Rx[irx].delay[comp];
			}
			// 受信点の通し番号++
			irx++;
		}

		// 平均を計算する（図形表示のため）
		double average[2];
		for (int comp = 0; comp < 2; comp++) {
			double sum = 0;
			for (int idiv = 0; idiv < ndiv; idiv++) {
				sum += ydata[comp][idiv];
			}
			sum /= ndiv;
			average[comp] = (kind == 0)
				? 10 * log10(MAX(sum, EPS2))
				: sum * (1e9 / C);
		}

		// 受信電力はdBWに、遅延時間はnsecに変換する
		for (int comp = 0; comp < 2; comp++) {
			for (int idiv = 0; idiv < ndiv; idiv++) {
				ydata[comp][idiv] = (kind == 0)
					? 10 * log10(MAX(ydata[comp][idiv], EPS2))
					: ydata[comp][idiv] * (1e9 / C);
			}
		}

		// Y軸スケール
		double dmax = ydata[0][0];
		for (int div = 0; div < Rx1d[n].div; div++) {
			dmax = MAX(dmax, MAX(ydata[0][div], ydata[1][div]));
		}
		double ymax, ymin;
		int    ydiv;
		scale_t scale = (kind == 0) ? Rx1d_power_scale : Rx1d_delay_scale;
		getscale(dmax, scale, &ymin, &ymax, &ydiv);
		//printf("%d %e %e %d\n", kind, ymin, ymax, ydiv);

		// Y座標を正規化する
		const double fy = (fabs(ymax - ymin) > EPS) ? (y2 - y1) / (ymax - ymin) : 0;
		for (int comp = 0; comp < 2; comp++) {
			for (int idiv = 0; idiv < ndiv; idiv++) {
				const double y = MAX(ymin, MIN(ymax, ydata[comp][idiv]));
				//ydata[comp][idiv] = y1 + (y2 - y1) * (y - ymin) / (ymax - ymin);
				ydata[comp][idiv] = y1 + fy * (y - ymin);
			}
		}

		// 図形出力
		ev2d_newPage();

		// グリッド
		ev2dlib_grid(x1, y1, x2, y2, xdiv, ydiv);

		// 平均値
		sprintf(str, "average %s", (kind == 0) ? "[dBW]" : "[nsec]");
		ev2d_drawString(x2 - 8.0 * h, y2 + 2.9 * h, h, str);

		// 成分1(赤)
		ev2d_setColor(255, 0, 0);
		ev2d_drawPolyline(ndiv, xdata, ydata[0]);
		ev2d_drawLine(x2 - 14.0 * h, y2 + 2.0 * h, x2 - 12.0 * h, y2 + 2.0 * h);
		ev2d_drawString(x2 - 11.0 * h, y2 + 1.7 * h, h, (kind == 0) ? "with phase" : "mean delay");
		sprintf(str, "%.3f", average[0]);
		ev2d_drawString(x2 - 2.0 * h, y2 + 1.7 * h, h, str);

		// 成分2(青)
		ev2d_setColor(0, 0, 255);
		ev2d_drawPolyline(ndiv, xdata, ydata[1]);
		ev2d_drawLine(x2 - 14.0 * h, y2 + 0.8 * h, x2 - 12.0 * h, y2 + 0.8 * h);
		ev2d_drawString(x2 - 11.0 * h, y2 + 0.5 * h, h, (kind == 0) ? "without phase" : "delay spread");
		sprintf(str, "%.3f", average[1]);
		ev2d_drawString(x2 - 2.0 * h, y2 + 0.5 * h, h, str);

		// 色初期化
		ev2d_setColor(0, 0, 0);

		// free
		free(xdata);
		free(ydata[0]);
		free(ydata[1]);

		// タイトル
		strcpy(str, (kind == 0) ? "Rx power" : "Rx delay");
		ev2d_drawString(x1, y2 + 2.9 * h, h, str);
		sprintf(str, "Rx line #%d", n + 1);
		ev2d_drawString(x1, y2 + 1.7 * h, h, str);
		ev2d_drawString(x1, y2 + 0.5 * h, h, Title);

		// Y軸
		if (kind == 0) {
			sprintf(str, "%.1f", ymax);
			ev2d_drawString(x1 - 3.5 * h, y2 - 0.3 * h, h, str);
			sprintf(str, "%.1f", ymin);
			ev2d_drawString(x1 - 3.5 * h, y1 - 0.3 * h, h, str);
			ev2d_drawString(x1 - 4.0 * h, y2 - 2.0 * h, h, "[dBW]");
		}
		else {
			sprintf(str, "%.1e", ymax);
			ev2d_drawString(x1 - 4.5 * h, y2 - 0.3 * h, h, str);
			sprintf(str, "%.1e", ymin);
			ev2d_drawString(x1 - 4.5 * h, y1 - 0.3 * h, h, str);
			ev2d_drawString(x1 - 4.0 * h, y2 - 2.0 * h, h, "[nsec]");
		}

		// 始点座標
		sprintf(str, "X[m] = %.3e", Rx1d[n].pos[0][0]);
		ev2d_drawString(x1 - 2.0 * h, y1 - 1.0 * h, h, str);
		sprintf(str, "Y[m] = %.3e", Rx1d[n].pos[0][1]);
		ev2d_drawString(x1 - 2.0 * h, y1 - 1.9 * h, h, str);
		sprintf(str, "Z[m] = %.3e", Rx1d[n].pos[0][2]);
		ev2d_drawString(x1 - 2.0 * h, y1 - 2.8 * h, h, str);

		// 終点座標
		sprintf(str, "X[m] = %.3e", Rx1d[n].pos[1][0]);
		ev2d_drawString(x2 - 6.0 * h, y1 - 1.0 * h, h, str);
		sprintf(str, "Y[m] = %.3e", Rx1d[n].pos[1][1]);
		ev2d_drawString(x2 - 6.0 * h, y1 - 1.9 * h, h, str);
		sprintf(str, "Z[m] = %.3e", Rx1d[n].pos[1][2]);
		ev2d_drawString(x2 - 6.0 * h, y1 - 2.8 * h, h, str);

		// コメント
		sprintf(str, "No. of Rx points = %d", ndiv);
		ev2d_drawString((x1 + x2) / 2 - 6.0 * h, y1 - 1.2 * h, h, str);
		sprintf(str, "Frequency = %.3e[Hz]", Frequency);
		ev2d_drawString((x1 + x2) / 2 - 7.0 * h, y1 - 2.5 * h, h, str);
	}
}
