/*
plot2dRx2d.c
観測面の図形出力(2D)
*/

#include "ort.h"
#include "vector.h"
#include "ort_prototype.h"
#include "ev.h"


// 観測面の統計処理を図形出力する
// kind = 0/1 : power/delay
// stat = 0/1 : histogram/CDF
static void plot2dRx2d_stat(int kind, int stat, double *sdata[2])
{
	// レイアウト
	const double x1 = 0.12 * Width2d;
	const double y1 = 0.1  * Height2d;
	const double x2 = 0.9  * Width2d;
	const double y2 = 0.86 * Height2d;
	const double h = Fontsize2d;

	char str[BUFSIZ];

	// 色
	unsigned char rgb[][3] = {{255, 0, 0}, {0, 0, 255}};

	// タイトル
	char head[2][2][BUFSIZ] = {
		{"Rx power histogram",
		 "Rx power cumulative distribution"},
		{"Rx delay histogram",
		 "Rx delay cumulative distribution"}};
	char color[2][2][BUFSIZ] = {
		{"with phase",
		 "without phase"},
		{"mean delay",
		 "delay spread"}};

	assert((kind == 0) || (kind == 1));
	assert((stat == 0) || (stat == 1));
	if ((NRx2d <= 0) || (NRx[2] <= 0)) return;

	// 最大値
	double smax = sdata[0][0];
	for (int n = 0; n < NRx[2]; n++) {
		smax = MAX(smax, MAX(sdata[0][n], sdata[1][n]));
	}
	//printf("%d %d %f\n", kind, stat, dmax);

	// スケール
	scale_t scale = (kind == 0) ? Rx2d_power_scale : Rx2d_delay_scale;
	double xmin, xmax;
	int    xdiv;
	getscale(smax, scale, &xmin, &xmax, &xdiv);
	if (!scale.user) xdiv = 40;  // 大きめに取る
	//printf("%d %f %f %d\n", kind, xmin, xmax, xdiv);

	// 頻度分布 (histogram)
	double *histo[2];
	double lo[] = {0, 0};
	for (int comp = 0; comp < 2; comp++) {
		histo[comp] = (double *)calloc(xdiv + 0, sizeof(double));
		for (int n = 0; n < NRx[2]; n++) {
			const double f = (sdata[comp][n] - xmin) / (xmax - xmin);
			if (f < 0) {  // 下限以下
				lo[comp] += 1;
			}
			else if (f < 1) {  // xmin < * < xmax
				const int ibin = (int)(xdiv * f);
				histo[comp][ibin] += 1;
			}
		}
	}

	// 累積確率分布 (cumulative distribution) (0-1)
	double *cumul[2];
	if (stat == 1) {
		for (int comp = 0; comp < 2; comp++) {
			cumul[comp] = (double *)calloc(xdiv + 1, sizeof(double));
			cumul[comp][0] = lo[comp] / NRx[2];
			for (int ibin = 0; ibin < xdiv; ibin++) {
				cumul[comp][ibin + 1] = cumul[comp][ibin] + (histo[comp][ibin] / NRx[2]);
			}
		}
	}

	// 頻度分布の正規化 (0-1)
	double hmax = histo[0][0];
	for (int ibin = 0; ibin < xdiv; ibin++) {
		hmax = MAX(hmax, MAX(histo[0][ibin], histo[1][ibin]));
	}
	hmax = MAX(hmax, EPS);
	for (int comp = 0; comp < 2; comp++) {
		for (int ibin = 0; ibin < xdiv; ibin++) {
			histo[comp][ibin] /= hmax;
		}
	}

	// 描画開始
	ev2d_newPage();

	// 枠
	ev2d_drawRectangle(x1, y1, x2, y2);

	// 縦目盛り
	const double dx = (x2 - x1) / xdiv;
	for (int i = 1; i < xdiv; i++) {
		const double x = x1 + (i * dx);
		if (i % 5 == 0) {
			ev2d_setColor(210, 210, 210);
			ev2d_drawLine(x, y1, x, y2);
		}
		ev2d_setColor(0, 0, 0);
		ev2d_drawLine(x, y1, x, y1 + 0.5 * h);
	}

	// 横線
	const int ydiv = 10;
	const double dy = (y2 - y1) / ydiv;
	ev2d_setColor(210, 210, 210);
	for (int j = 1; j < ydiv; j++) {
		const double y = y1 + (j * dy);
		ev2d_drawLine(x1, y, x2, y);
	}

	ev2d_setColor(0, 0, 0);

	// Y軸
	ev2d_drawString(x1 - 3.0 * h, y2 - 0.3 * h, h, ((stat == 0) ? "MAX" : "100"));
	ev2d_drawString(x1 - 1.5 * h, y1 - 0.3 * h, h, "0");
	if (stat == 1) {
		ev2d_drawString(x1 - 3.0 * h, y2 - 2.0 * h, h, "[%]");
	}

	// 2成分の上書き
	for (int comp = 0; comp < 2; comp++) {
		// 色
		ev2d_setColor(rgb[comp][0], rgb[comp][1], rgb[comp][2]);

		if (stat == 0) {
			// 頻度分布
			double x = x1;
			double yold = y1 + (y2 - y1) * histo[comp][0];
			for (int ibin = 0; ibin < xdiv; ibin++) {
				ev2d_drawLine(x, yold, x + dx, yold);
				if (ibin < xdiv - 1) {
					x += dx;
					const double y = y1 + (y2 - y1) * histo[comp][ibin + 1];
					ev2d_drawLine(x, yold, x, y);
					yold = y;
				}
				//printf("%d %d %d %f\n", type, ib, comp, histo[comp][ib]);
			}
		}
		else {
			// 累積確率分布
			for (int ibin = 0; ibin < xdiv; ibin++) {
				const double c1 = cumul[comp][ibin + 0];
				const double c2 = cumul[comp][ibin + 1];
				ev2d_drawLine(x1 + (ibin + 0) * dx, y1 + (y2 - y1) * c1,
				              x1 + (ibin + 1) * dx, y1 + (y2 - y1) * c2);
			}
		}

		// 成分
		const double y = y2 + (1.8 - 1.3 * comp) * h;
		ev2d_drawLine(x2 - 7.0 * h, y + 0.3 * h, x2 - 5.0 * h, y + 0.3 * h);
		ev2d_drawString(x2 - 4.5 * h, y, h, color[kind][comp]);
	}

	ev2d_setColor(0, 0, 0);

	// タイトル
	ev2d_drawString(x1, y2 + 0.5 * h, h, Title);
	ev2d_drawString(x1, y2 + 1.8 * h, h, head[kind][stat]);

	// X軸ラベル
	sprintf(str, "%.1f [%s]", xmin, ((kind == 0) ? "dBW" : "nsec"));
	ev2d_drawString(x1 - 1.0 * h, y1 - 1.2 * h, h, str);
	sprintf(str, "%.1f", xmax);
	ev2d_drawString(x2 - 1.0 * h, y1 - 1.2 * h, h, str);

	// 受信点数
	sprintf(str, "No. of Rx points = %d", NRx[2]);
	ev2d_drawString(x1, y1 - 2.3 * h, h, str);
}


// 受信電力の距離特性の散布図をプロットする
// comp = 0/1 : 位相差あり/なし
// pdata : 受信電力データ[dBW]
// ldata : LOSであるか
static void plot2dRx2d_decay(int comp, double *ydata, int *ldata)
{
	// レイアウト
	const double x1 = 0.12 * Width2d;
	const double y1 = 0.1  * Height2d;
	const double x2 = 0.9  * Width2d;
	const double y2 = 0.86 * Height2d;
	const double h = Fontsize2d;

	// タイトル
	char title[2][BUFSIZ] = {"Rx power (w/ phase) vs distance",
	                         "Rx power (w/o phase) vs distance"};

	char str[BUFSIZ], str1[BUFSIZ], str2[BUFSIZ];

	assert((comp == 0) || (comp == 1));

	// 送信点の数は1のみ(複数あると距離が一意に決まらないため)
	if (NTx > 1) {
		fprintf(stderr, "*** Txs = %d > 1\n", NTx);
		return;
	}

	// X軸データ : 送受信間距離(対数)
	// 送信点は第1送信点のみ
	double *txpos = Tx[0].pos;
	double *xdata = (double *)malloc(NRx[2] * sizeof(double));
	const int irx0 = NRx[0] + NRx[1];
	for (int irx = 0; irx < NRx[2]; irx++) {
		xdata[irx] = log10(MAX(vector_distance(Rx[irx0 + irx].pos, txpos), EPS));
	}

	// X軸の最小・最大
	double xmin = +1000;
	double xmax = -1000;
	for (int irx = 0; irx < NRx[2]; irx++) {
		xmin = MIN(xmin, xdata[irx]);
		xmax = MAX(xmax, xdata[irx]);
	}
	//xmin = MAX(xmin, xmax - 3);  // 最大1:1000

	// Y軸の最小・最大
	double ymin = +1000;
	double ymax = -1000;
	for (int irx = 0; irx < NRx[2]; irx++) {
		ymin = MIN(ymin, ydata[irx]);
		ymax = MAX(ymax, ydata[irx]);
	}

	// Y軸スケール(10dBの倍数)
	ymin = 10 * floor(ymin / 10);
	ymax = 10 *  ceil(ymax / 10);
	ymin = MAX(ymin, ymax - 60);  // 最大60dB

	// 図形表示
	ev2d_newPage();

	// グリッド表示
	int ydiv = (int)((ymax - ymin) / 10 + 0.5);  // 10dB間隔
	ev2dlib_grid(x1, y1, x2, y2, 1, ydiv);

	// X軸の最小・最大は10mのべき乗
	int ixmin, ixmax;
	if (Rx2d_decay_scale.user) {
		ixmin = (int)(log10(Rx2d_decay_scale.min) + 0.5);
		ixmax = (int)(log10(Rx2d_decay_scale.max) + 0.5);
	}
	else {
		ixmin = (int)floor(xmin);
		ixmax = (int) ceil(xmax);
		ixmin = MAX(ixmin, ixmax - 3);  // 最大1:1000
	}
	//printf("%f %f %d %d\n", xmin, xmax, ixmin, ixmax);

	// 縦線表示
	const double dx = (x2 - x1) / (ixmax - ixmin);
	for (int ix = ixmin; ix < ixmax; ix++) {
		// 10mのべき乗
		const double xe = x1 + (x2 - x1) * (ix - ixmin) / (ixmax - ixmin);
		ev2d_setColor(0, 0, 0);
		ev2d_drawLine(xe, y1, xe, y2);
		// 対数目盛り
		for (int m = 2; m < 10; m++) {
			const double xd = xe + dx * log10(m);
			ev2d_setColor(192, 192, 192);
			ev2d_drawLine(xd, y1, xd, y2);
		}
	}

	// 色初期化
	ev2d_setColor(0, 0, 0);

	// 散布図
	const double hm = 0.15 * h;
	xmin = ixmin;
	xmax = ixmax;
	int num = 0;
	for (int n = 0; n < NRx[2]; n++) {
		const double x = x1 + (x2 - x1) * (xdata[n] - xmin) / (xmax - xmin);
		const double y = y1 + (y2 - y1) * (ydata[n] - ymin) / (ymax - ymin);
		if ((x >= x1) && (x <= x2) && (y >= y1) && (y <= y2)) {
			//printf("%d %f %f %f %f\n", n, xdata[n], ydata[n], x, y);
			if (ldata[n]) {
				ev2d_setColor(255, 0, 0);  // LOSは赤
			}
			else {
				ev2d_setColor(  0, 0, 0);  // NLOSは黒
			}
			ev2d_fillRectangle(x - hm, y - hm, x + hm, y + hm);
			num++;
		}
	}

	// 色初期化
	ev2d_setColor(0, 0, 0);

	// X軸スケール
	for (int ix = ixmin; ix <= ixmax; ix++) {
		const double xe = x1 + (x2 - x1) * (ix - ixmin) / (ixmax - ixmin);
		sprintf(str, "%g", pow(10, ix));
		ev2d_drawString(xe - 1.0 * h, y1 - 1.2 * h, h, str);
	}

	// X軸ラベル
	sprintf(str, "distance from Tx to Rx [m]  (No. of Rx points = %d/%d)", num, NRx[2]);
	ev2d_drawString((x1 + x2) / 2 - 17.0 * h, y1 - 2.5 * h, h, str);

	// Y軸ラベル
	sprintf(str1, "%.0f", ymin);
	sprintf(str2, "%.0f", ymax);
	ev2dlib_Yaxis(y1, y2, x1, h, str1, str2, "[dBW]");

	// タイトル
	ev2d_drawString(x1, y2 + 0.5 * h, h, Title);
	ev2d_drawString(x1, y2 + 1.8 * h, h, title[comp]);
}


// 観測面の受信点はLOSであるか(色分けに使用)
static void islos(int *ldata)
{
	// 受信点の開始番号
	const int irx0 = NRx[0] + NRx[1];

	// 観測面の受信点に関するループ
	for (int irx = 0; irx < NRx[2]; irx++) {
		if (Rx[irx0 + irx].npath > 0) {
			// LOS : 最初の伝搬経路が直接波である(npos=2)
			ldata[irx] = (Rx[irx0 + irx].path[0].npos == 2);
		}
		else {
			ldata[irx] = 0;
		}
	}
}


// 観測面の2D図形出力
void plot2dRx2d(void)
{
	// 観測面データの作成
	double *pdata[2], *tdata[2];
	for (int comp = 0; comp < 2; comp++) {
		pdata[comp] = (double *)malloc(NRx[2] * sizeof(double));
		tdata[comp] = (double *)malloc(NRx[2] * sizeof(double));
	}
	calcRx2d(pdata, tdata);

	// 頻度分布と累積確率分布
	if (Rx2d_stat[0]) {
		plot2dRx2d_stat(0, 0, pdata);
	}
	if (Rx2d_stat[1]) {
		plot2dRx2d_stat(0, 1, pdata);
	}
	if (Rx2d_stat[2]) {
		plot2dRx2d_stat(1, 0, tdata);
	}
	if (Rx2d_stat[3]) {
		plot2dRx2d_stat(1, 1, tdata);
	}

	// 距離特性
	if (Rx2d_decay[0] || Rx2d_decay[1]) {
		// LOSであるか(色分けに使用)
		int *ldata = (int *)malloc(NRx[2] * sizeof(int));
		islos(ldata);

		// 図形表示
		if (Rx2d_decay[0]) {
			plot2dRx2d_decay(0, pdata[0], ldata);
		}
		if (Rx2d_decay[1]) {
			plot2dRx2d_decay(1, pdata[1], ldata);
		}
	}
}
