/*
plot3dRx1d.c
観測線の図形出力(3D)
*/

#include "ort.h"
#include "complex.h"
#include "vector.h"
#include "ort_prototype.h"
#include "ev.h"


// 観測線の遅延プロファイル (3D)
void plot3dRx1d(void)
{
	if ((NRx1d <= 0) || (NRx[1] <= 0)) return;

	const double lx = 2;
	const double ly = 3;
	const double lz = 1;
	unsigned char rgb[3];

	// コメント
	char *comment[5];
	for (int n = 0; n < 5; n++) {
		comment[n] = (char *)malloc(BUFSIZ * sizeof(char));
	}

	// 受信点の通し番号
	int irx = NRx[0];

	// 観測線に関するループ
	for (int n = 0; n < NRx1d; n++) {
		// 受信点分割数
		const int ndiv = Rx1d[n].div;

		// 配列作成
		double **x    = (double **)malloc(ndiv * sizeof(double *));
		double **z    = (double **)malloc(ndiv * sizeof(double *));
		int    **npos =    (int **)malloc(ndiv * sizeof(int *));
		int    *npath =     (int *)malloc(ndiv * sizeof(int));

		// 受信点に関するループ
		for (int idiv = 0; idiv < ndiv; idiv++) {
			// 受信点データ
			rx_t rx = Rx[irx++];

			// No. of paths
			npath[idiv] = rx.npath;
			//printf("%d %d %d\n", n, idiv, npath[idiv]);

			// 配列作成
			x[idiv]    = (double *)malloc(npath[idiv] * sizeof(double));
			z[idiv]    = (double *)malloc(npath[idiv] * sizeof(double));
			npos[idiv] =    (int *)malloc(npath[idiv] * sizeof(int));

			// X座標 : 遅延時間 [nsec]
			// Z座標 : 受信電力 [dBW]
			for (int ipath = 0; ipath < npath[idiv]; ipath++) {
				double lng = 0;
				for (int ipos = 0; ipos < rx.path[ipath].npos - 1; ipos++) {
					lng += vector_distance(rx.path[ipath].pos[ipos + 0],
					                       rx.path[ipath].pos[ipos + 1]);
				}
				//printf("%d %d %d %f\n", idiv, ipath, rx.path[ipath].npos, lng);
				x[idiv][ipath] = lng / C * 1e9;  // m -> nsec
				z[idiv][ipath] = 20 * log10(MAX(d_abs(rx.efield[ipath]), EPS));
				//printf("%d %d %f %f\n", idiv, ipath, x[idiv][ipath], z[idiv][ipath]);
				// 頂点数 : 色分けのため
				npos[idiv][ipath] = rx.path[ipath].npos;
			}
		}

		// X座標とZ座標の最大値
		double dxmax = 0;
		double dzmax = -1000;
		for (int idiv = 0; idiv < ndiv; idiv++) {
			for (int ipath = 0; ipath < npath[idiv]; ipath++) {
				dxmax = MAX(dxmax, x[idiv][ipath]);
				dzmax = MAX(dzmax, z[idiv][ipath]);
				//printf("%d %d %f %f %f %f\n", idiv, ipath, x[idiv][ipath], z[idiv][ipath], dxmax, dzmax);
			}
		}
		//printf("%f %f\n", dxmax, dzmax);
		double xmin, zmin, xmax, zmax;
		int    xdiv, zdiv;
		getscale(dxmax, Rx1d_delay_scale, &xmin, &xmax, &xdiv);
		getscale(dzmax, Rx1d_power_scale, &zmin, &zmax, &zdiv);
		//printf("%f %f %f %f\n", xmin, xmax, zmin, zmax);

		// X軸とZ軸のスケーリング
		for (int idiv = 0; idiv < ndiv; idiv++) {
			for (int ipath = 0; ipath < npath[idiv]; ipath++) {
				x[idiv][ipath] = lx * (x[idiv][ipath] - xmin) / (xmax - xmin);
				z[idiv][ipath] = lz * (z[idiv][ipath] - zmin) / (zmax - zmin);
				z[idiv][ipath] = MAX(z[idiv][ipath], 0);
			}
		}

		// 1観測線=1ページ
		ev3d_newPage();

		// X=0
		ev3d_setColor(200, 200, 200);
		ev3d_drawRectangle('X', 0, 0, 0, ly, lz);

		// プロット
		for (int idiv = 0; idiv < ndiv; idiv++) {
			const double y = ly * idiv / ndiv;
			// 時間軸
			ev3d_setColor(200, 200, 200);
			ev3d_drawLine(0, y, 0, lx, y, 0);
			// 受信電力
			for (int ipath = 0; ipath < npath[idiv]; ipath++) {
				getpathcolor(npos[idiv][ipath] - 1, rgb);
				ev3d_setColor(rgb[0], rgb[1], rgb[2]);
				ev3d_drawLine(x[idiv][ipath], y, 0,
				              x[idiv][ipath], y, z[idiv][ipath]);
			}
		}

		// コメント
		const double dlx = Rx1d[n].pos[1][0] - Rx1d[n].pos[0][0];
		const double dly = Rx1d[n].pos[1][1] - Rx1d[n].pos[0][1];
		const double dlz = Rx1d[n].pos[1][2] - Rx1d[n].pos[0][2];
		const double ylng = sqrt((dlx * dlx) + (dly * dly) + (dlz * dlz));
		strcpy(comment[0], Title);
		strcpy(comment[1], "delay profile along a line");
		sprintf(comment[2], "X[nsec] : (%.2f, %.2f)", xmin, xmax);
		sprintf(comment[3], "Y[m] : (0, %.3f)", ylng);
		sprintf(comment[4], "Z[dBW] : (%.1f, %.1f)",  zmin, zmax);
		ev3d_setColor(0, 0, 0);
		for (int ic = 0; ic < 5; ic++) {
			ev3d_drawTitle(comment[ic]);
		}
	}

	// 色初期化
	ev3d_setColor(0, 0, 0);
}
