/*
plot3dRx2d.c
観測面の図形出力(3D)
*/

#include "ort.h"
#include "ort_prototype.h"
#include "ev.h"


// 観測面の等高線図を描く
// kind = 0/1 : power/delay
// comp = 0/1 : component
static void plot3dRx2d_contour(int kind, int comp, double *fdata)
{
	char head[2][2][BUFSIZ] = {
		{"Rx power (with phase)",
		 "Rx power (without phase)"},
		{"Rx mean delay",
		 "Rx delay spread"}};
	char str[BUFSIZ];

	assert((kind == 0) || (kind == 1));
	assert((comp == 0) || (comp == 1));
	if ((NRx2d <= 0) || (NRx[2] <= 0)) return;

	// XYZ配列作成
	double (*x)[4] = (double (*)[4])malloc(NRx[2] * 4 * sizeof(double));
	double (*y)[4] = (double (*)[4])malloc(NRx[2] * 4 * sizeof(double));
	double (*z)[4] = (double (*)[4])malloc(NRx[2] * 4 * sizeof(double));

	// XYZ座標
	double f[4][3];
	int ndata = 0;
	for (int n = 0; n < NRx2d; n++) {
		const int ndiv12 = Rx2d[n].div[0];
		const int ndiv14 = Rx2d[n].div[1];
		for (int idiv12 = 0; idiv12 < ndiv12; idiv12++) {
		for (int idiv14 = 0; idiv14 < ndiv14; idiv14++) {
			const double a0 = (double)(idiv12 + 0) / ndiv12;
			const double a1 = (double)(idiv12 + 1) / ndiv12;
			const double b0 = (double)(idiv14 + 0) / ndiv14;
			const double b1 = (double)(idiv14 + 1) / ndiv14;
			for (int m = 0; m < 3; m++) {
				f[0][m] = (1 - a0) * (1 - b0) * Rx2d[n].pos[0][m]
				        + (    a0) * (1 - b0) * Rx2d[n].pos[1][m]
				        + (    a0) * (    b0) * Rx2d[n].pos[2][m]
				        + (1 - a0) * (    b0) * Rx2d[n].pos[3][m];
				f[1][m] = (1 - a1) * (1 - b0) * Rx2d[n].pos[0][m]
				        + (    a1) * (1 - b0) * Rx2d[n].pos[1][m]
				        + (    a1) * (    b0) * Rx2d[n].pos[2][m]
				        + (1 - a1) * (    b0) * Rx2d[n].pos[3][m];
				f[2][m] = (1 - a1) * (1 - b1) * Rx2d[n].pos[0][m]
				        + (    a1) * (1 - b1) * Rx2d[n].pos[1][m]
				        + (    a1) * (    b1) * Rx2d[n].pos[2][m]
				        + (1 - a1) * (    b1) * Rx2d[n].pos[3][m];
				f[3][m] = (1 - a0) * (1 - b1) * Rx2d[n].pos[0][m]
				        + (    a0) * (1 - b1) * Rx2d[n].pos[1][m]
				        + (    a0) * (    b1) * Rx2d[n].pos[2][m]
				        + (1 - a0) * (    b1) * Rx2d[n].pos[3][m];
			}
			for (int k = 0; k < 4; k++) {
				x[ndata][k] = f[k][0];
				y[ndata][k] = f[k][1];
				z[ndata][k] = f[k][2];
			}
			ndata++;
		}
		}
	}
	assert(ndata == NRx[2]);

	// XYZ座標の範囲
	double xmin = +1 / EPS;
	double xmax = -1 / EPS;
	double ymin = +1 / EPS;
	double ymax = -1 / EPS;
	double zmin = +1 / EPS;
	double zmax = -1 / EPS;
	for (int n = 0; n < ndata; n++) {
		for (int k = 0; k < 4; k++) {
			xmin = MIN(xmin, x[n][k]);
			xmax = MAX(xmax, x[n][k]);
			ymin = MIN(ymin, y[n][k]);
			ymax = MAX(ymax, y[n][k]);
			zmin = MIN(zmin, z[n][k]);
			zmax = MAX(zmax, z[n][k]);
		}
	}
	//printf("%f %f %f %f %f %f\n", xmin, xmax, ymin, ymax, zmin, zmax);

	// 関数値の最大値
	double dmax = fdata[0];
	for (int n = 0; n < NRx[2]; n++) {
		dmax = MAX(dmax, fdata[n]);
	}

	// 関数値のスケール
	double fmax, fmin;
	int    fdiv;
	scale_t scale = (kind == 0) ? Rx2d_power_scale : Rx2d_delay_scale;
	getscale(dmax, scale, &fmin, &fmax, &fdiv);
	//printf("%f %f %f %d\n", dmax, fmin, fmax, fdiv);

	// 描画開始
	ev3d_newPage();

	// 等高線図
	for (int n = 0; n < NRx[2]; n++) {
		const double v = (fdata[n] - fmin) / (fmax - fmin);
		ev3d_setColorV(v, 1);
		ev3d_fillQuadrangle(x[n][0], y[n][0], z[n][0], x[n][1], y[n][1], z[n][1], x[n][2], y[n][2], z[n][2], x[n][3], y[n][3], z[n][3]);
	}
	ev3d_setColor(0, 0, 0);

	// タイトル
	ev3d_drawTitle(Title);
	ev3d_drawTitle(head[kind][comp]);

	// データ数
	sprintf(str, "No. of Rx points = %d", NRx[2]);
	ev3d_drawTitle(str);

	// 最大値
	sprintf(str, ((kind == 0) ? "%s = %.3fdBW" : "%s = %.3ensec"), "max", dmax);
	ev3d_drawTitle(str);
	if (kind == 0) {
		sprintf(str, "max-min = %.1fdB", fmax - fmin);
		ev3d_drawTitle(str);
	}

	// 物体形状と送信点
	if (Rx2d_geometry) {
		plot3d_geom(0, Rx2d_geometry == 2);
		plot3d_tx();
	}
}


// 観測面の3D図形出力
void plot3dRx2d(void)
{
	// 観測面データの作成
	double *pdata[2], *tdata[2];
	for (int comp = 0; comp < 2; comp++) {
		pdata[comp] = (double *)malloc(NRx[2] * sizeof(double));
		tdata[comp] = (double *)malloc(NRx[2] * sizeof(double));
	}
	calcRx2d(pdata, tdata);

	// 受信電力等高線図
	for (int comp = 0; comp < 2; comp++) {
		if (Rx2d_power[comp]) {
			plot3dRx2d_contour(0, comp, pdata[comp]);
		}
	}

	// 遅延時間等高線図
	for (int comp = 0; comp < 2; comp++) {
		if (Rx2d_delay[comp]) {
			plot3dRx2d_contour(1, comp, tdata[comp]);
		}
	}
}
