/*
antenna.c
*/

#include "ort.h"
#include "complex.h"
#include "vector.h"
#include "ort_prototype.h"

// アンテナファイルのときの(θ,φ)方向の電界
static void eantenna_file(antenna_t *a, double theta, double phi, double *r, d_complex_t *etheta, d_complex_t *ephi)
{
	assert(a->atype == 4);

	const double dtheta = 180 / a->ntheta;
	const double dphi   = 360 / a->nphi;

	// 0<θ<180, 0<φ<360
	if (phi < 0) phi += 360;   // -180...+180 -> 0...360
	theta = MAX(0, MIN(180, theta));
	phi   = MAX(0, MIN(360, phi  ));
	int itheta = (int)(theta / dtheta);
	int iphi   = (int)(phi   / dphi  );
	itheta = MAX(0, MIN(a->ntheta - 1, itheta));
	iphi   = MAX(0, MIN(a->nphi   - 1, iphi  ));

	// 補間
	const double ftheta = (theta / dtheta) - itheta;
	const double fphi   = (phi   / dphi  ) - iphi;
	*etheta = d_add4(
		 d_rmul((1 - ftheta) * (1 - fphi), a->etheta[itheta + 0][iphi + 0]),
		 d_rmul(     ftheta  * (1 - fphi), a->etheta[itheta + 1][iphi + 0]),
		 d_rmul((1 - ftheta) *      fphi,  a->etheta[itheta + 0][iphi + 1]),
		 d_rmul(     ftheta  *      fphi,  a->etheta[itheta + 1][iphi + 1]));
	*ephi = d_add4(
		 d_rmul((1 - ftheta) * (1 - fphi),   a->ephi[itheta + 0][iphi + 0]),
		 d_rmul(     ftheta  * (1 - fphi),   a->ephi[itheta + 1][iphi + 0]),
		 d_rmul((1 - ftheta) *      fphi,    a->ephi[itheta + 0][iphi + 1]),
		 d_rmul(     ftheta  *      fphi,    a->ephi[itheta + 1][iphi + 1]));
	*r = sqrt(d_norm(*etheta) + d_norm(*ephi));
	//printf("%f %f %f %f %f\n", etheta.r, etheta.i, ephi.r, ephi.i, r);
}


/*
機能:
	アンテナの相対電界利得を計算する
入力:
	a : アンテナ
	theta, phi : θ, φ [度]
関数値:
	相対電界利得(0-1)
*/
double rantenna(antenna_t *a, double theta, double phi)
{
	double r = 1;

	// 放射パターンによる分類
	if      (a->atype == 1) {
		// isotropic
		r = 1;
	}
	else if (a->atype == 2) {
		// dipole
		double d[3], v[3];
		polar2xyz(1, a->theta, a->phi, d);
		polar2xyz(1, theta, phi, v);
		const double cosa = MAX(-1, MIN(+1, vector_innerprod(d, v)));
		double arg = acos(cosa);  // [rad]
		arg = (90 / a->bw) * (PI / 2 - arg);
		arg = MAX(-PI / 2, MIN(PI / 2, arg));  // no sidelobe
		r = cos(arg);
	}
	else if (a->atype == 3) {
		// beam
		double d[3], v1[3], v2[3];
		polar2xyz(1, a->theta, a->phi, d);
		polar2xyz(1, theta, a->phi, v1);
		polar2xyz(1, a->theta, phi, v2);
		const double cos1 = MAX(-1, MIN(+1, vector_innerprod(d, v1)));
		double arg1 = acos(cos1);  // [rad]
		const double cos2 = MAX(-1, MIN(+1, vector_innerprod(d, v2)));
		double arg2 = acos(cos2);  // [rad]
		arg1 *= 90 / a->bwtheta;
		arg2 *= 90 / a->bwphi;
		arg1 = MAX(0, MIN(PI / 2, arg1));  // no sidelobe
		arg2 = MAX(0, MIN(PI / 2, arg2));  // no sidelobe
		r = cos(arg1) * cos(arg2);
		//printf("%f %f %f %f %f %f %f\n", a->bwtheta, a->bwphi, a->theta, a->phi, theta, phi, r);
	}
	else if (a->atype == 4) {
		// file
		d_complex_t etheta, ephi;  // dummy
		eantenna_file(a, theta, phi, &r, &etheta, &ephi);
	}

	return r;
}


/*
機能:
	(θ,φ)方向の電界(指向性、偏波、利得を考慮)
入力:
	a : アンテナ
	theta, phi : θ, φ [度]
出力:
	etheta, ephi : E-θ, E-φ (電圧次元、複素数)
*/
void eantenna(antenna_t *a, double theta, double phi, d_complex_t *etheta, d_complex_t *ephi)
{
	if (a->atype < 4) {
		// 数値
		// 電界利得
		const double e = rantenna(a, theta, phi) * sqrt(a->gain);
		//printf("%f %f %f %f\n", theta, phi, e, a->gain);

		// 偏波因子
		if      (a->pol == 1) {
			// V-pol
			*etheta = d_complex(e, 0);
			*ephi   = d_complex(0, 0);
		}
		else if (a->pol == 2) {
			// H-pol
			*etheta = d_complex(0, 0);
			*ephi   = d_complex(e, 0);
		}
		else if (a->pol == 3) {
			// RHCP
			*etheta = d_complex(+e / sqrt(2), 0);
			*ephi   = d_complex(0, +e / sqrt(2));
		}
		else if (a->pol == 4) {
			// LHCP
			*etheta = d_complex(+e / sqrt(2), 0);
			*ephi   = d_complex(0, -e / sqrt(2));
		}
	}
	else {
		// ファイル
		double r; // dummy
		eantenna_file(a, theta, phi, &r, etheta, ephi);
		*etheta = d_rmul(sqrt(a->gain), *etheta);
		*ephi   = d_rmul(sqrt(a->gain), *ephi);
	}
}


/*
機能:
	アンテナの利得を計算する
入力:
	a : アンテナ
	ntheta : θ方向の分割数
関数値:
	利得(電力次元)
*/
double gain(antenna_t *a, int ntheta)
{
	const int nphi = 2 * ntheta;
	const double dtheta = 180.0 / ntheta;
	const double dphi   = 360.0 / nphi;

	double sum = 0;
	for (int itheta = 1; itheta < ntheta; itheta++) {
	for (int iphi   = 0; iphi   < nphi;   iphi++  ) {
		const double theta = itheta * dtheta;
		const double phi   = iphi   * dphi;
		const double e = rantenna(a, theta, phi);
		sum += (e * e) * sin(theta * DTOR);
	}
	}

	return (4 * PI) / (sum * (dtheta * DTOR) * (dphi * DTOR));
}


// アンテナパターンファイルを読み込む
void read_antenna(const char file[], antenna_t *a)
{
	FILE   *fp;
	char   str[BUFSIZ];

	// ファイル開く
	if ((fp = fopen(file, "r")) == NULL) {
		fprintf(stderr, "*** file %s open error\n", file);
		return;
	}

	// N-θ,N-φを調べる
	int itheta_max = 0;
	int iphi_max = 0;
	fgets(str, BUFSIZ, fp);
	fgets(str, BUFSIZ, fp);
	while (fgets(str, BUFSIZ, fp) != NULL) {
		int itheta, iphi;
		sscanf(str, "%d %d", &itheta, &iphi);
		itheta_max = MAX(itheta_max, itheta);
		iphi_max   = MAX(iphi_max,   iphi);
	}
	//printf("%d %d\n", itheta_max, iphi_max);

	fclose(fp);

	// alloc
	a->ntheta = itheta_max;
	a->nphi   = iphi_max;
	a->etheta = (d_complex_t **)malloc((a->ntheta + 1) * sizeof(d_complex_t *));
	a->ephi   = (d_complex_t **)malloc((a->ntheta + 1) * sizeof(d_complex_t *));
	for (int itheta = 0; itheta <= a->ntheta; itheta++) {
		a->etheta[itheta] = (d_complex_t *)malloc((a->nphi + 1) * sizeof(d_complex_t));
		a->ephi[itheta]   = (d_complex_t *)malloc((a->nphi + 1) * sizeof(d_complex_t));
	}

	// 再度ファイルを読み込む(本体)
	fp = fopen(file, "r");

	fgets(str, BUFSIZ, fp);
	fgets(str, BUFSIZ, fp);
	while (fgets(str, BUFSIZ, fp) != NULL) {
		int itheta, iphi;
		double theta, phi, d[5];
		sscanf(str, "%d %d %lf %lf %lf %lf %lf %lf %lf",
			&itheta, &iphi, &theta, &phi, &d[0], &d[1], &d[2], &d[3], &d[4]);
		// dB,deg -> 電界利得(複素数)
		a->etheta[itheta][iphi] = d_rmul(pow(10, d[1] / 20), d_exp(d[2] * DTOR));
		a->ephi[itheta][iphi]   = d_rmul(pow(10, d[3] / 20), d_exp(d[4] * DTOR));
		//printf("%d %d %f %f\n", itheta, iphi, d_abs(a->etheta[itheta][iphi]), d_abs(a->ephi[itheta][iphi]));
	}

	fclose(fp);

	// 最大値で正規化
	double pmax = 0;
	for (int itheta = 0; itheta <= a->ntheta; itheta++) {
	for (int iphi   = 0; iphi   <= a->nphi;   iphi++  ) {
		pmax = MAX(pmax, d_norm(a->etheta[itheta][iphi])
		               + d_norm(a->ephi[itheta][iphi]));
	}
	}
	//printf("pmax=%f\n", pmax);
	const double fctr = 1 / sqrt(pmax);
	for (int itheta = 0; itheta <= a->ntheta; itheta++) {
	for (int iphi   = 0; iphi   <= a->nphi;   iphi++  ) {
		a->etheta[itheta][iphi] = d_rmul(fctr, a->etheta[itheta][iphi]);
		a->ephi[itheta][iphi]   = d_rmul(fctr, a->ephi[itheta][iphi]);
		//printf("%d %d %f %f\n", itheta, iphi, d_abs(a->etheta[itheta][iphi]), d_abs(a->ephi[itheta][iphi]));
	}
	}
}
