/*
diffract.c
*/

#include "ort.h"
#include "vector.h"

/*
機能:
	回折波の回折点を求める
入力:
	pos1, pos2 : 点1,2の座標
	vtx1, vtx2 : 稜線点1,2の座標
出力:
	pdif[0] : 回折点の座標
	pdif[1/2] : 回折点の近くの2点の座標(LOS判定に使用する)
関数値:
	0/1 : 回折波が存在しない/する
*/
int diffract(const double pos1[], const double pos2[], const double vtx1[], const double vtx2[], double pdif[][3])
{
	int    ret = 0;
	double u43[3], u21[3], u31[3];

	vector_sub(vtx1, pos1, u31);
	vector_sub(pos2, pos1, u21);
	vector_sub(vtx2, vtx1, u43);

	const double a = vector_innerprod(u21, u21);
	const double b = vector_innerprod(u21, u43) * (-2);
	const double c = vector_innerprod(u43, u43);
	const double d = vector_innerprod(u21, u31) * (-2);
	const double e = vector_innerprod(u43, u31) * 2;

	const double det = (4 * a * c) - (b * b);

	if (det > 0) {
		const double v = ((b * d) - (2 * a * e)) / det;
		const double u = ((b * e) - (2 * c * d)) / det;

		if ((v > +EPS) && (v < 1 - EPS) &&
			(u > +EPS) && (u < 1 - EPS)) {
			ret = 1;

			// (真の)回折点
			for (int k = 0; k < 3; k++) {
				pdif[0][k] = ((1 - v) * vtx1[k]) + (v * vtx2[k]);
			}

			// LOS判定に用いる回折点近くの2点 : pdif[1:2]
			const double eps = 1e-3;  // EPSより十分大きい微小量(1e-3がよい?)
			for (int k = 0; k < 3; k++) {
				const double d12 = (pdif[0][k] - pos1[k])
				                 + (pdif[0][k] - pos2[k]);
				pdif[1][k] = pdif[0][k] + (eps * d12);
				pdif[2][k] = pdif[0][k] - (eps * d12);
			}
		}
	}

	return ret;
}
