/*
erxpath.c
*/

#include "ort.h"
#include "complex.h"
#include "vector.h"
#include "ort_prototype.h"

/*
機能:
	指定した伝搬経路の受信電界を計算する
入力:
	path : 伝搬経路データ
関数値:
	受信電界(スカラー)
*/
d_complex_t erxpath(path_t *path)
{
	const double wavl = C / Frequency;
	const double wavn = (2 * PI) / wavl;
	const int itx = path->itri[0];
	const int itxa = Tx[itx].antenna;
	const int irx = path->itri[path->npos - 1];
	const int irxa = Rx[irx].antenna;

	assert((itx >= 0) && (itx < NTx) && (irx >= 0) && (irx < NRx[3]));
	assert((itxa >= 0) && (itxa < NAntenna) && (irxa >= 0) && (irxa < NAntenna));

	// (1) 送信点
	// v_tx : 送信点(点0)から点1に向かうベクトル
	double v_tx[3];
	vector_sub(path->pos[1], path->pos[0], v_tx);
	//printf("%e %e %e\n", v_tx[0], v_tx[1], v_tx[2]);

	// (r_tx, theta_tx, phi_tx) : 送信点から点1に向かう方向の(r,θ,φ)
	double r_tx, theta_tx, phi_tx;
	xyz2polar(v_tx, &r_tx, &theta_tx, &phi_tx);
	//printf("%e %e\n", theta_tx * DTOR, phi_tx * DTOR);

	// (etheta_tx, ephi_tx) : 送信点におけるEθ,Eφ(複素数スカラー)
	d_complex_t etheta_tx, ephi_tx;
	eantenna(&Antenna[itxa], theta_tx, phi_tx, &etheta_tx, &ephi_tx);
	//printf("%e %e %e %e\n", etheta_tx.r, etheta_tx.i, ephi_tx.r, ephi_tx.i);

	// (r1_tx, theta1_tx, phi1_tx) : 送信点から点1に向かう方向の(r,θ,φ)方向単位ベクトル
	double r1_tx[3], theta1_tx[3], phi1_tx[3];
	polar_vector(theta_tx, phi_tx, r1_tx, theta1_tx, phi1_tx);
	//printf("%e %e %e %e %e %e\n", theta1_tx[0], theta1_tx[2], theta1_tx[2], phi1_tx[0], phi1_tx[1], phi1_tx[2]);

	// e : 送信点における電界ベクトル
	d_complex_t e[3];
	for (int k = 0; k < 3; k++) {
		e[k] = d_add(d_rmul(theta1_tx[k], etheta_tx),
		             d_rmul(  phi1_tx[k],   ephi_tx));
	}
	//printf("%e %e %e %e %e %e\n", e[0].r, e[0].i, e[1].r, e[1].i, e[2].r, e[2].i);

	// (2) 反射・回折・透過による電界の変化
	for (int ipos = 1; ipos < path->npos - 1; ipos++) {
		rdt(path, ipos, e);
	}

	// (3) 受信点
	// v_rx : 受信点から最後の反射点に向かうベクトル
	double v_rx[3];
	vector_sub(path->pos[path->npos - 2], path->pos[path->npos - 1], v_rx);

	// (r_rx, theta_rx, phi_rx) : 受信点から最後の反射点に向かう方向の(r,θ,φ)
	double r_rx, theta_rx, phi_rx;
	xyz2polar(v_rx, &r_rx, &theta_rx, &phi_rx);

	// (r1_rx, theta1_rx, phi1_rx) : 受信点から最後の反射点に向かう方向の(r,θ,φ)方向単位ベクトル
	double r1_rx[3], theta1_rx[3], phi1_rx[3];
	polar_vector(theta_rx, phi_rx, r1_rx, theta1_rx, phi1_rx);

	// (etheta_rx, ephi_rx) : 受信点におけるEθ,Eφ(複素数スカラー)
	const d_complex_t etheta_rx = d_add3(d_rmul(theta1_rx[0], e[0]),
	                                     d_rmul(theta1_rx[1], e[1]),
	                                     d_rmul(theta1_rx[2], e[2]));
	const d_complex_t ephi_rx   = d_add3(d_rmul(  phi1_rx[0], e[0]),
	                                     d_rmul(  phi1_rx[1], e[1]),
	                                     d_rmul(  phi1_rx[2], e[2]));

	// (etheta_rxa, ephi_rxa) : 受信点におけるEθ,Eφ(アンテナ指向性込み、複素数スカラー)
	d_complex_t etheta_rxa, ephi_rxa;
	eantenna(&Antenna[irxa], theta_rx, phi_rx, &etheta_rxa, &ephi_rxa);

	// erx : 受信点における電界(スカラー)
	//printf("%e %e %e %e\n", etheta_rx.r, etheta_rxa.i, etheta_rxa.r, etheta_rxa.i);
	//printf("%e %e %e %e\n", ephi_rx.r, ephi_rxa.i, ephi_rxa.r, ephi_rxa.i);
	d_complex_t erx = d_add(d_mul(etheta_rx, etheta_rxa),
	                        d_mul(  ephi_rx,   ephi_rxa));

	// 伝搬経路の長さ
	double lng = 0;
	for (int ipos = 0; ipos < path->npos - 1; ipos++) {
		lng += vector_distance(path->pos[ipos], path->pos[ipos + 1]);
	}

	// Friisの公式の因子
	const d_complex_t cfctr = d_rmul(
		wavl / (4 * PI * lng) *
		sqrt(Tx[itx].power),
		d_exp(-(wavn * lng) + (Tx[itx].phase * DTOR)));

	// erx : 受信点における電界(伝搬損失込み、スカラー)
	erx = d_mul(cfctr, erx);
	//printf("%e %e %e %e\n", cfctr.r, cfctr.i, erx.r, erx.i);

	return erx;
}
