/*
fresnel.c
*/

#include "ort.h"
#include "complex.h"

/*
機能:
	誘電体のFresnel反射係数を計算する
入力:
	cepsr : 複素比誘電率
	ainc : 入射角 [deg]
出力:
	r[0/1] : 反射係数 (垂直偏波, V-POL, TM, P-POL / 水平偏波, H-POL, TE, S-POL)
*/
void fresnel_reflection(d_complex_t cepsr, double ainc, d_complex_t r[2])
{
	if ((fabs(cepsr.r - 1) < EPS) && (fabs(cepsr.i) < EPS)) {
		r[0] = d_complex(0, 0);
		r[1] = d_complex(0, 0);
		return;
	}

	const d_complex_t cost = d_complex(cos(ainc * DTOR), 0);
	const d_complex_t sint = d_complex(sin(ainc * DTOR), 0);
	const d_complex_t root = d_sqrt(d_sub(cepsr, d_mul(sint, sint)));

	r[0] = d_div(d_sub(root, d_mul(cepsr, cost)), d_add(root, d_mul(cepsr, cost)));
	r[1] = d_div(d_sub(cost, root), d_add(cost, root));
}

/*
機能:
	Fresnel積分を計算する
入力:
	z : 積分範囲(0:z)
出力:
	Fresnel積分値
*/
double fresnel_integral(double z)
{
	const double inf = 1e4;

	if      (z > +inf) return 0;
	else if (z < -inf) return sqrt(2);

	const double x = fabs(z);

	const double fx = (1.0 + x * 0.926) / (2.0 + x * (1.792 + x * 3.104));

	const double gx = 1.0 / (2.0 + x * (4.142 + x * (3.492 + x * 6.670)));

	const double arg = PI * 0.5 * x * x;
	d_complex_t fnc = d_mul(d_exp(-arg), d_complex(gx, -fx));

	if (z < 0) fnc = d_sub(d_complex(1, -1), fnc);

	return d_abs(fnc);
}


/*
機能:
	1層誘電板の反射係数と透過係数を計算する
入力:
	cepsr : 複素比誘電率
	thick : 厚さ [m]
	ainc : 入射角 [deg]
出力:
	r[0/1] : 反射係数(P-POL/S-POL)
	t[0/1] : 透過係数(P-POL/S-POL)
*/
void reftrans(d_complex_t cepsr, double thick, double ainc, double frequency, d_complex_t r[2], d_complex_t t[2])
{
	const double k = (2 * PI * frequency) / C;
	const double sint0 = sin(ainc * DTOR);
	const double cost0 = cos(ainc * DTOR);
	const d_complex_t ceps = d_rmul(EPS0, cepsr);
	const d_complex_t cmu = d_complex(MU0, 0);
	const d_complex_t ceta = d_sqrt(d_div(cmu, ceps));
	const d_complex_t sint = d_rmul((sint0 / C), d_inv(d_sqrt(d_mul(ceps, cmu))));
	const d_complex_t cost = d_sqrt(d_sub(d_complex(1, 0), d_mul(sint, sint)));
	const d_complex_t d = d_rmul((C * k * thick), d_mul(d_sqrt(d_mul(ceps, cmu)), cost));

	for (int i = 0; i < 2; i++) {
		const d_complex_t a1 = (i == 0) ? d_mul(ceta, cost) : d_div(ceta, cost);
		const d_complex_t m11 = d_cos(d);
		const d_complex_t m12 = d_mul(d_complex(0, 1), d_mul(d_sin(d), a1));
		const d_complex_t m21 = d_mul(d_complex(0, 1), d_div(d_sin(d), a1));
		const d_complex_t m22 = m11;
		//printf("%d %e %e\n", i, d_sub(d_mul(m11, m22), d_mul(m12, m21)).r, d_sub(d_mul(m11, m22), d_mul(m12, m21)).i);

		const double a0 = (i == 0) ? (ETA0 * cost0) : (ETA0 / cost0);
		const d_complex_t p = d_add(d_rmul(a0, m11), m12);
		const d_complex_t q = d_rmul(a0, d_add(d_rmul(a0, m21), m22));
		const d_complex_t denom = d_add(p, q);
		//const d_complex_t denom = d_add(d_add(d_rmul(a0, m11), m12), d_rmul(a0, d_add(d_rmul(a0, m21), m22)));
		r[i] = d_div(d_sub(p, q), denom);
		//r[i] = d_div(d_sub(d_add(d_rmul(a0, m11), m12), d_rmul(a0, d_add(d_rmul(a0, m21), m22))), denom);
		t[i] = d_div(d_complex(2 * a0, 0), denom);
	}
}


/*
// > cl.exe -I../include fresnel.c
int main(void)
{
	const int n = 50;
	const double d = 0.1;
	for (int i = -n; i <= +n; i++) {
		const double x = i * d;
		// exact
		const double r1 = fresnel_integral(x) / sqrt(2);
		// approx-1
		const double r2 = 0.4519 / (sqrt((x - 0.1) * (x - 0.1) + 1) + (x - 0.1));
		// approx-2
		const double r3 = (x > 0) ? (0.2259 / x) : 0;
		printf("%.1f, %.5f, %.5f, %.5f\n", x, r1, r2, r3);
	}
}
*/
/*
int main(void)
{
	d_complex_t ref[2];
	for (int i = 0; i <= 90; i++) {
		fresnel_reflection(d_complex(2, 0), (double)i, ref);
		printf("%d %f %f", i, ref[0].r, ref[1].r);
		fresnel_reflection(d_complex(10, 0), (double)i, ref);
		printf(" %f %f", ref[0].r, ref[1].r);
		fresnel_reflection(d_complex(80, 0), (double)i, ref);
		printf(" %f %f\n", ref[0].r, ref[1].r);
	}
}
*/
/*
int main(void)
{
	const double epsr_r = 2;
	const double sigma = 0.1;
	const double freq = 1e9;
	const double epsr_i = sigma / (2 * PI * freq * EPS0);
	const d_complex_t epsr = d_complex(epsr_r, -epsr_i);
	d_complex_t r[2];
	for (int i = 0; i <= 90; i += 2) {
		fresnel_reflection(epsr, (double)i, r);
		printf("%2d %7.4f %7.4f %7.4f %7.4f 0 0 0 0\n", i, r[0].r, r[0].i, r[1].r, r[1].i);
	}
}
*/
/*
int main(void)
{
	const double thick = 0.1;
	const double epsr_r = 2.0;
	const double sigma = 0.1;
	const double freq = 1e9;
	const double epsr_i = sigma / (2 * PI * freq * EPS0);
	const d_complex_t epsr = d_complex(epsr_r, -epsr_i);
	d_complex_t r[2], t[2];
	for (int i = 0; i <= 90; i += 2) {
		reftrans(epsr, thick, (double)i, freq, r, t);
		printf("%2d %7.4f %7.4f %7.4f %7.4f %7.4f %7.4f %7.4f %7.4f\n", i, r[0].r, r[0].i, r[1].r, r[1].i, t[0].r, t[0].i, t[1].r, t[1].i);
	}
}
*/
