/*
input_data.c
入力データを読み込む (計算部)
*/

#include "ort.h"
#include "complex.h"
#include "ort_prototype.h"

static int center(int, const double [], const double []);
static void make_triangle(int, double, double, double, double, double, double, double, double, double);
static void read_material(const char [], material_t *);

int input_data(FILE *fp)
{
	int    version = 0;
	char   strline[MAXCHAR], strsave[MAXCHAR];
	char   strkey[BUFSIZ], filename[BUFSIZ];
	char   *token[MAXTOKEN];
	const char sep[] = " \t";     // separator

	// 変数の初期化

	NMaterial = 2;  // 0:AIR, 1:PEC
	NAntenna  =
	NTriangle =
	NTx       =
	NRx0d     =
	NRx1d     =
	NRx2d     = 0;

	Material =     (material_t *)malloc(NMaterial * sizeof(material_t));
	Antenna  =      (antenna_t *)malloc( NAntenna * sizeof(antenna_t));
	Triangle =     (triangle_t *)malloc(ARRAY_INC * sizeof(triangle_t));

	Material[0].mtype = 2;  // Air -> transmission
	Material[1].mtype = 1;  // PEC -> no transmission

	strcpy(Title, "");

	Frequency   = 1e9;

	MaxPath     = 100;
	MaxRef      = 3;
	NDivLaunch  = 180;
	NDivAntenna = 36;

	Icomp[0] = Icomp[1] = Icomp[2] = Icomp[3] = 1;
	Log[0] = Log[1] = 0;
	Plot3d[0] = Plot3d[1] = Plot3d[2] = 0;

	// ファイルを読み込む

	int nline = 0;   // 行番号(warning用)
	while (fgets(strline, sizeof(strline), fp) != NULL) {
		//printf("%s", strline);

		// 空行スキップ
		if (strlen(strline) <= 1) continue;

		// コメント行スキップ
		if (strline[0] == '#') continue;

		// 末尾の"\n"を除く
		//printf("%zd\n", strlen(strline));
		if (strstr(strline, "\r\n") != NULL) {
			strline[strlen(strline) - 2] = '\0';
		}
		else if ((strstr(strline, "\r") != NULL) || (strstr(strline, "\n") != NULL)) {
			strline[strlen(strline) - 1] = '\0';
		}
		//printf("%zd\n", strlen(strline));

		// "end"行 : 終了
		if (!strcmp(strline, "end")) break;

		// 文字列保存
		strcpy(strsave, strline);

		// トークン分解 (文字列"strline"は破壊される)
		const int ntoken = tokenize(strline, sep, token, MAXTOKEN);
		//printf("%d %d\n", nline, ntoken);
		//for (int i = 0; i < ntoken; i++) printf("%d %s\n", i, token[i]);

		// キーワード
		strcpy(strkey, token[0]);

		// データ数(3以上)チェックと"="確認 (ヘッダーとタイトルは除く)
		if ((nline > 0) && strcmp(strkey, "title") && ((ntoken < 3) || strcmp(token[1], "="))) {
			warning(strkey, nline);
			continue;
		}

		if (nline == 0) {
			// ヘッダー(1行)
			if (strcmp(strkey, "OpenRTM")) {
				fprintf(stderr, "*** not OpenRTM data\n");
				return 1;
			}
			if (ntoken > 1) {
				version = 100 * atoi(token[1]);  // major version
			}
			if (ntoken > 2) {
				version += atoi(token[2]);       // minor version
			}
			//printf("version=%d\n", version);
		}
		else if (!strcmp(strkey, "maxpath")) {
			// 最大伝搬経路数
			if (ntoken > 2) {
				MaxPath = atoi(token[2]);
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "title")) {
			// タイトル
			if (ntoken > 2) {
				strcpy(Title, strchr(strsave, '=') + 2);
			}
			//printf("%s\n", Title);
		}
		else if (!strcmp(strkey, "material")) {
			// 物性値
			if (ntoken > 3) {
				Material = (material_t *)realloc(Material, (NMaterial + 1) * sizeof(material_t));
				Material[NMaterial].mtype = atoi(token[2]);
				//Material[NMaterial].trans = (Material[NMaterial].mtype == 2) || (Material[NMaterial].mtype == 3);
				if      ((Material[NMaterial].mtype == 1) && (ntoken > 4)) {
					// reflection
					Material[NMaterial].epsr  = atof(token[3]);
					Material[NMaterial].sigma = atof(token[4]);
					NMaterial++;
				}
				else if ((Material[NMaterial].mtype == 2) && (ntoken > 5)) {
					// reflection + transmission
					Material[NMaterial].epsr  = atof(token[3]);
					Material[NMaterial].sigma = atof(token[4]);
					Material[NMaterial].thick = atof(token[5]);
					NMaterial++;
				}
				else if ((Material[NMaterial].mtype == 3) && (ntoken > 3)) {
					// file
					sprintf(filename, "data/material/%s", token[3]);
					read_material(filename, &Material[NMaterial]);
					NMaterial++;
				}
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "antenna")) {
			// アンテナ
			if (ntoken > 3) {
				Antenna = (antenna_t *)realloc(Antenna, (NAntenna + 1) * sizeof(antenna_t));
				Antenna[NAntenna].atype = atoi(token[2]);
				if      (Antenna[NAntenna].atype == 1) {
					// isotropic
					Antenna[NAntenna].pol     = atoi(token[3]);
					NAntenna++;
				}
				else if ((Antenna[NAntenna].atype == 2) && (ntoken > 6)) {
					// dipole
					Antenna[NAntenna].pol     = atoi(token[3]);
					Antenna[NAntenna].theta   = atof(token[4]);
					Antenna[NAntenna].phi     = atof(token[5]);
					Antenna[NAntenna].bw      = atof(token[6]);
					NAntenna++;
				}
				else if ((Antenna[NAntenna].atype == 3) && (ntoken > 7)) {
					// beam
					Antenna[NAntenna].pol     = atoi(token[3]);
					Antenna[NAntenna].theta   = atof(token[4]);
					Antenna[NAntenna].phi     = atof(token[5]);
					Antenna[NAntenna].bwtheta = atof(token[6]);
					Antenna[NAntenna].bwphi   = atof(token[7]);
					NAntenna++;
				}
				else if (Antenna[NAntenna].atype == 4) {
					// file
					sprintf(filename, "data/antenna/%s", token[3]);
					read_antenna(filename, &Antenna[NAntenna]);
					NAntenna++;
				}
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "pillar")) {
			// 多角柱
			if (ntoken > 8) {
				// 頂点数
				const int nvtx = (ntoken - 5) / 2;

				// XY座標(三角形以上では+1)
				const size_t size = ((nvtx == 2) ? nvtx : (nvtx + 1)) * sizeof(double);
				double *x = (double *)malloc(size);
				double *y = (double *)malloc(size);

				// 物性値番号とXYZ座標
				const int m     = atoi(token[2]);
				const double f3 = atof(token[3]);
				const double f4 = atof(token[4]);
				const double z1 = MIN(f3, f4);
				const double z2 = MAX(f3, f4);
				for (int i = 0; i < nvtx; i++) {
					x[i] = atof(token[(2 * i) + 5]);
					y[i] = atof(token[(2 * i) + 6]);
				}

				// 物体形状作成
				if (nvtx == 2) {
					// 頂点数=2 : 壁
					make_triangle(m, x[0], y[0], z1, x[1], y[1], z1, x[0], y[0], z2);
					make_triangle(m, x[1], y[1], z2, x[0], y[0], z2, x[1], y[1], z1);
				}
				else {
					// 三角形以上
					// 側面(終点のあとに始点を追加して閉曲線にする)
					x[nvtx] = x[0];
					y[nvtx] = y[0];
					for (int i = 0; i < nvtx; i++) {
						make_triangle(m, x[i + 0], y[i + 0], z1, x[i + 1], y[i + 1], z1, x[i + 0], y[i + 0], z2);
						make_triangle(m, x[i + 1], y[i + 1], z2, x[i + 0], y[i + 0], z2, x[i + 1], y[i + 1], z1);
					}

					// 床と天井
					// i0 : 多角形の重心に一番近い頂点番号
					const int i0 = center(nvtx, x, y);

					// 三角形データ追加(i1,i2は除く)
					const int i1 = (i0 > 0) ? (i0 - 1) : (nvtx - 1);
					const int i2 = i0;
					for (int i = 0; i < nvtx; i++) {
						if ((i != i1) && (i != i2)) {
							make_triangle(m, x[i], y[i], z1, x[i + 1], y[i + 1], z1, x[i0], y[i0], z1);
							make_triangle(m, x[i], y[i], z2, x[i + 1], y[i + 1], z2, x[i0], y[i0], z2);
						}
					}
				}

				free(x);
				free(y);
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "polygon")) {
			// 多角形
			if (ntoken > 11) {
				// m : 物性値番号
				const int m = atoi(token[2]);
				//printf("%d %d\n", m, Material[m].mtype);

				// nvtx : 頂点数 > 2
				const int nvtx = (ntoken - 3) / 3;

				// XYZ座標(頂点数+1)
				double *x = (double *)malloc((nvtx + 1) * sizeof(double));
				double *y = (double *)malloc((nvtx + 1) * sizeof(double));
				double *z = (double *)malloc((nvtx + 1) * sizeof(double));
				for (int i = 0; i < nvtx; i++) {
					x[i] = atof(token[(3 * i) +  3]);
					y[i] = atof(token[(3 * i) +  4]);
					z[i] = atof(token[(3 * i) +  5]);
				}
				// 終点のあとに始点を追加して閉曲線にする
				x[nvtx] = x[0];
				y[nvtx] = y[0];
				z[nvtx] = z[0];

				// i0 : 多角形の重心に一番近い頂点番号
				const int i0 = center(nvtx, x, y);
				//printf("%d %d\n", nvtx, i0);

				// 三角形データ追加(i1,i2は除く)
				const int i1 = (i0 > 0) ? (i0 - 1) : (nvtx - 1);
				const int i2 = i0;
				for (int i = 0; i < nvtx; i++) {
					if ((i != i1) && (i != i2)) {
						make_triangle(m, x[i], y[i], z[i], x[i + 1], y[i + 1], z[i + 1], x[i0], y[i0], z[i0]);
					}
				}

				free(x);
				free(y);
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "tx")) {
			// 送信点
			if (ntoken > 7) {
				Tx = (tx_t *)realloc(Tx, (NTx + 1) * sizeof(tx_t));
				Tx[NTx].antenna = atoi(token[2]) - 1;
				Tx[NTx].pos[0]  = atof(token[3]);
				Tx[NTx].pos[1]  = atof(token[4]);
				Tx[NTx].pos[2]  = atof(token[5]);
				Tx[NTx].power   = atof(token[6]);
				Tx[NTx].phase   = atof(token[7]);
				NTx++;
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "rx0d")) {
			// 観測点
			if (ntoken > 5) {
				Rx0d = (rx0d_t *)realloc(Rx0d, (NRx0d + 1) * sizeof(rx0d_t));
				Rx0d[NRx0d].antenna = atoi(token[2]) - 1;
				Rx0d[NRx0d].pos[0]  = atof(token[3]);
				Rx0d[NRx0d].pos[1]  = atof(token[4]);
				Rx0d[NRx0d].pos[2]  = atof(token[5]);
				NRx0d++;
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "rx1d")) {
			// 観測線
			if (ntoken > 9) {
				Rx1d = (rx1d_t *)realloc(Rx1d, (NRx1d + 1) * sizeof(rx1d_t));
				Rx1d[NRx1d].antenna   = atoi(token[2]) - 1;
				Rx1d[NRx1d].pos[0][0] = atof(token[3]);
				Rx1d[NRx1d].pos[0][1] = atof(token[4]);
				Rx1d[NRx1d].pos[0][2] = atof(token[5]);
				Rx1d[NRx1d].pos[1][0] = atof(token[6]);
				Rx1d[NRx1d].pos[1][1] = atof(token[7]);
				Rx1d[NRx1d].pos[1][2] = atof(token[8]);
				Rx1d[NRx1d].div       = atoi(token[9]);
				NRx1d++;
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "rx2d")) {
			// 観測面
			if (ntoken > 16) {
				Rx2d = (rx2d_t *)realloc(Rx2d, (NRx2d + 1) * sizeof(rx2d_t));
				Rx2d[NRx2d].antenna   = atoi(token[2]) - 1;
				Rx2d[NRx2d].pos[0][0] = atof(token[3]);
				Rx2d[NRx2d].pos[0][1] = atof(token[4]);
				Rx2d[NRx2d].pos[0][2] = atof(token[5]);
				Rx2d[NRx2d].pos[1][0] = atof(token[6]);
				Rx2d[NRx2d].pos[1][1] = atof(token[7]);
				Rx2d[NRx2d].pos[1][2] = atof(token[8]);
				Rx2d[NRx2d].pos[2][0] = atof(token[9]);
				Rx2d[NRx2d].pos[2][1] = atof(token[10]);
				Rx2d[NRx2d].pos[2][2] = atof(token[11]);
				Rx2d[NRx2d].pos[3][0] = atof(token[12]);
				Rx2d[NRx2d].pos[3][1] = atof(token[13]);
				Rx2d[NRx2d].pos[3][2] = atof(token[14]);
				Rx2d[NRx2d].div[0]    = atoi(token[15]);
				Rx2d[NRx2d].div[1]    = atoi(token[16]);
				NRx2d++;
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "frequency")) {
			// 周波数
			if (ntoken > 2) {
				Frequency = atof(token[2]);
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "maxref")) {
			// 最大反射回数
			if (ntoken > 2) {
				MaxRef = atoi(token[2]);
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "ndivlaunch")) {
			// ローンチング法の緯度方向分割数
			if (ntoken > 2) {
				NDivLaunch = atoi(token[2]);
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "ndivantenna")) {
			// アンテナパターンの緯度方向分割数
			if (ntoken > 2) {
				NDivAntenna = atoi(token[2]);
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "log")) {
			// 数値出力
			if (ntoken > 3) {
				Log[0] = atoi(token[2]);
				Log[1] = atoi(token[3]);
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "component")) {
			// 直接法の計算成分
			if (ntoken > 5) {
				Icomp[0] = MAX(0, MIN(1, atoi(token[2])));
				Icomp[1] = MAX(0, MIN(1, atoi(token[3])));
				Icomp[2] = MAX(0, MIN(2, atoi(token[4])));
				Icomp[3] = MAX(0, MIN(1, atoi(token[5])));
			}
			else {
				warning(strkey, nline);
			}
		}
		else if (!strcmp(strkey, "plot3d")) {
			// 入力データの3D図形表示
			if (ntoken > 4) {
				Plot3d[0] = atoi(token[2]);
				Plot3d[1] = atoi(token[3]);
				Plot3d[2] = atoi(token[4]);
			}
			else {
				warning(strkey, nline);
			}
		}

		// 行番号++（warning用）
		nline++;
	}
/*
	// debug
	for (int n = 0; n < NMaterial; n++) {
		if      (Material[n].mtype == 1) {
			printf("Material : %d %d %f %f\n", n, Material[n].mtype, Material[n].epsr, Material[n].sigma);
		}
		else if (Material[n].mtype == 2) {
			printf("Material : %d %d %f %f %f\n", n, Material[n].mtype, Material[n].epsr, Material[n].sigma, Material[n].thick);
		}
		else if (Material[n].mtype == 3) {
			printf("Material : %d %d %d\n", n, Material[n].mtype, Material[n].nangle);
			for (int i = 0; i < Material[n].nangle; i++) {
				printf("%d %f %f %f %f %f\n", i, Material[n].angle[i], Material[n].rh[i].r, Material[n].rh[i].i, Material[n].rv[i].r, Material[n].rv[i].i);
			}
		}
		else if (Material[n].mtype == 4) {
			printf("Material : %d %d %d\n", n, Material[n].mtype, Material[n].nangle);
			for (int i = 0; i < Material[n].nangle; i++) {
				printf("%d %f %f %f %f %f %f %f %f %f\n", i, Material[n].angle[i], Material[n].rh[i].r, Material[n].rh[i].i, Material[n].rv[i].r, Material[n].rv[i].i, Material[n].th[i].r, Material[n].th[i].i, Material[n].tv[i].r, Material[n].tv[i].i);
			}
		}
	}
	for (int n = 0; n < NAntenna; n++) {
		if      (Antenna[n].atype == 1) {
			printf("Antenna : %d %d %d\n", n, Antenna[n].atype, Antenna[n].pol);
		}
		else if (Antenna[n].atype == 2) {
			printf("Antenna : %d %d %d %f %f %f\n", n, Antenna[n].atype, Antenna[n].pol, Antenna[n].theta, Antenna[n].phi, Antenna[n].bw);
		}
		else if (Antenna[n].atype == 3) {
			printf("Antenna : %d %d %d %f %f %f %f\n", n, Antenna[n].atype, Antenna[n].pol, Antenna[n].theta, Antenna[n].phi, Antenna[n].bwtheta, Antenna[n].bwphi);
		}
		else if (Antenna[n].atype == 4) {
			printf("Antenna : %d %d %d %d\n", n, Antenna[n].atype, Antenna[n].ntheta, Antenna[n].nphi);
			for (int i = 0; i <= Antenna[n].ntheta; i++) {
				for (int j = 0; j <= Antenna[n].nphi; j++) {
					printf("%d %d %f %f %f %f\n", i, j, Antenna[n].etheta[i][j].r, Antenna[n].etheta[i][j].i, Antenna[n].ephi[i][j].r, Antenna[n].ephi[i][j].i);
				}
			}
		}
	}
	for (int n = 0; n < NTx; n++) {
		printf("Tx : %d %d\n", n, Tx[n].antenna);
	}
*/
	// エラーチェック

	if (NAntenna <= 0) {
		fprintf(stderr, "%s\n", "*** no Antenna data");
		return 1;
	}

	if (NTx <= 0) {
		fprintf(stderr, "%s\n", "*** no Tx data");
		return 1;
	}

	if ((NRx0d + NRx1d + NRx2d) <= 0) {
		fprintf(stderr, "%s\n", "*** no Rx data");
		return 1;
	}

	for (int n = 0; n < NRx1d; n++) {
		if (Rx1d[n].div <= 0) {
			fprintf(stderr, "*** division <= 0 : Rx1d number = %d\n", n + 1);
			return 1;
		}
	}

	for (int n = 0; n < NRx2d; n++) {
		if ((Rx2d[n].div[0] <= 0) || (Rx2d[n].div[1] <= 0)) {
			fprintf(stderr, "*** division <= 0 : Rx2d number = %d\n", n + 1);
			return 1;
		}
	}

	return 0;
}


// n角形(x,y)の重心に一番近い頂点番号
static int center(int n, const double x[], const double y[])
{
	assert(n > 0);
	double xsum = 0;
	double ysum = 0;
	for (int i = 0; i < n; i++) {
		xsum += x[i];
		ysum += y[i];
	}
	const double xc = xsum / n;
	const double yc = ysum / n;

	int imin = -1;
	double d2min = 1 / EPS;
	for (int i = 0; i < n; i++) {
		const double d2 = (x[i] - xc) * (x[i] - xc)
		                + (y[i] - yc) * (y[i] - yc);
		if (d2 < d2min) {
			imin = i;
			d2min = d2;
		}
	}
	assert(imin >= 0);

	return imin;
}


// 三角形データ追加
static void make_triangle(int m, double x1, double y1, double z1, double x2, double y2, double z2, double x3, double y3, double z3)
{
	const int n = NTriangle;

	Triangle[n].m = m;

	// 頂点1
	Triangle[n].v[0][0] = x1;
	Triangle[n].v[0][1] = y1;
	Triangle[n].v[0][2] = z1;

	// 頂点2
	Triangle[n].v[1][0] = x2;
	Triangle[n].v[1][1] = y2;
	Triangle[n].v[1][2] = z2;

	// 頂点3
	Triangle[n].v[2][0] = x3;
	Triangle[n].v[2][1] = y3;
	Triangle[n].v[2][2] = z3;

	// 頂点4=頂点1
	Triangle[n].v[3][0] = x1;
	Triangle[n].v[3][1] = y1;
	Triangle[n].v[3][2] = z1;

	NTriangle = n + 1;

	if ((NTriangle % ARRAY_INC) == 0) {
		Triangle = (triangle_t *)realloc(Triangle, (NTriangle + ARRAY_INC) * sizeof(triangle_t));
	}
}


// 物性値ファイルを読み込む
static void read_material(const char file[], material_t *m)
{
	FILE *fp;
	char str[BUFSIZ];

	assert(m->mtype == 3);

	// ファイル開く
	if ((fp = fopen(file, "r")) == NULL) {
		fprintf(stderr, "*** file %s open error\n", file);
		return;
	}

	m->angle = NULL;
	m->rh    = NULL;
	m->rv    = NULL;
	m->th    = NULL;
	m->tv    = NULL;

	int n = 0;
	while (fgets(str, BUFSIZ, fp) != NULL) {
		// 読み込み
		double d1, d2, d3, d4, d5, d6, d7, d8, d9;
		sscanf(str, "%lf %lf %lf %lf %lf %lf %lf %lf %lf", &d1, &d2, &d3, &d4, &d5, &d6, &d7, &d8, &d9);
		// realloc(個数がわからないため)
		m->angle =      (double *)realloc(m->angle, (n + 1) * sizeof(double));
		m->rv    = (d_complex_t *)realloc(m->rv,    (n + 1) * sizeof(d_complex_t));
		m->rh    = (d_complex_t *)realloc(m->rh,    (n + 1) * sizeof(d_complex_t));
		m->tv    = (d_complex_t *)realloc(m->tv,    (n + 1) * sizeof(d_complex_t));
		m->th    = (d_complex_t *)realloc(m->th,    (n + 1) * sizeof(d_complex_t));
		// 変数に代入 : R/T, V/H, 実部/虚部
		m->angle[n] = d1;
		m->rv[n] = d_complex(d2, d3);
		m->rh[n] = d_complex(d4, d5);
		m->tv[n] = d_complex(d6, d7);
		m->th[n] = d_complex(d8, d9);
		//printf("R %d %f %f %f %f %f\n", n, m->angle[n], m->rv[n].r, m->rv[n].i, m->rh[n].r, m->rh[n].i);
		//printf("T %d %f %f %f %f %f\n", n, m->angle[n], m->tv[n].r, m->tv[n].i, m->th[n].r, m->th[n].i);
		n++;
	}
	m->nangle = n;

	fclose(fp);
}
