/*
launching.c
レイローンチング法
*/

#include "ort.h"
#include "vector.h"
#include "ort_prototype.h"

static int numray(void);
static void setray(double (*)[3]);

// ローンチング法の前処理
// 各送信点からのレイの経路を追跡して配列Rayに格納する
void setup_launching(void)
{
	// 1送信点から放射されるレイの数を求める
	const int nray = numray();

	// 全送信点から放射されるレイの数とレイの配列を作成する
	NRay = NTx * nray;
	Ray = (ray_t *)malloc(NRay * sizeof(ray_t));

	// 1送信点から放射されるレイ集合の単位方向ベクトルを求める（作業配列）
	double (*r1set)[3] = (double (*)[3])malloc(nray * 3 * sizeof(double));
	setray(r1set);

// OpenMP並列領域開始
#ifdef _OPENMP
#pragma omp parallel
#endif
{
	// レイ経路の作業配列（スレッド毎に必要）
	double (*pos)[3] = (double (*)[3])malloc((MaxRef + 2) * 3 * sizeof(double));
	int    *itri     =         (int *)malloc((MaxRef + 2)     * sizeof(int));

	// 送信点に関するループ
	for (int itx = 0; itx < NTx; itx++) {
		// レイに関するループ（OpenMPによる並列計算）
		int iray;
#ifdef _OPENMP
#pragma omp for
#endif
		for (iray = 0; iray < nray; iray++) {

			double r1[3], pcrs[3], fcol;
			int edge;

			// 頂点数初期化
			int npos = 0;

			// 始点(送信点)座標
			vector_copy(Tx[itx].pos, pos[0]);

			// 送信点の三角形番号は送信点番号とする
			itri[0] = itx;

			// 単位方向ベクトルr1(送信点で初期化、反射するごとに変わる)
			vector_copy(r1set[iray], r1);

			// 反射点に関するループ
			for (int ipos = 0; ipos < MaxRef + 1; ipos++) {
				// npos=伝搬経路の頂点数(送信点と受信点も含む)、1回反射のとき3
				npos = ipos + 2;

				// 変数の初期化
				int    collision = 0;      // 衝突フラグ
				int    ntrimin = -1;       // 衝突した三角形の番号
				double ltrimin = 1 / EPS;  // 衝突点までの距離
				double ptrimin[3];         // 衝突点の座標

				// 一番近い衝突三角形を探す
				// 三角形に関するループ
				for (int ntri = 0; ntri < NTriangle; ntri++) {
					edge = 1;
					if (collide(pos[ipos], r1, &Triangle[ntri], &fcol, edge)) {
						// レイセグメントが指定した三角形と衝突したとき

						// 衝突フラグを立てる
						collision = 1;

						// 衝突点の座標
						for (int k = 0; k < 3; k++) {
							pcrs[k] = pos[ipos][k] + (fcol * r1[k]);
						}

						// レイセグメントの始点と衝突点との距離
						const double ltri = vector_distance(pos[ipos], pcrs);

						// 最短距離を更新したとき
						if (ltri < ltrimin) {
							ltrimin = ltri;
							ntrimin = ntri;
							vector_copy(pcrs, ptrimin);
						}
					}
				}

				// 後処理
				if (collision) {
					// いずれかの三角形に衝突した : 終点の処理をして次に反射点に進む
					// 終点(=次のレイセグメントの始点)の座標
					vector_copy(ptrimin, pos[ipos + 1]);

					// 終点の三角形の番号
					itri[ipos + 1] = ntrimin;

					// 反射による単位方向ベクトルの更新
					const double nr = vector_innerprod(Triangle[ntrimin].n, r1);
					for (int k = 0; k < 3; k++) {
						r1[k] -= 2 * nr * Triangle[ntrimin].n[k];
					}
				}
				else {
					// どの三角形にも衝突しなかった : 反射点に関するループを終了する
					// 終点は十分遠方とする
					for (int k = 0; k < 3; k++) {
						pos[ipos + 1][k] = pos[ipos][k] + (r1[k] * 2.0 * Lbound);
					}

					// 遠方フラグを代入
					itri[ipos + 1] = -1;

					// 反射点に関するループを終了する
					break;
				}
			}

			// レイのデータを代入する
			int idata = (nray * itx) + iray;

			// 頂点の数
			Ray[idata].npos = npos;

			// 頂点座標
			size_t size_pos = npos * 3 * sizeof(double);
			Ray[idata].pos = (double (*)[3])malloc(size_pos);
			memcpy(Ray[idata].pos, pos, size_pos);

			// 衝突した三角形の番号
			size_t size_itri = npos * sizeof(int);
			Ray[idata].itri = (int *)malloc(size_itri);
			memcpy(Ray[idata].itri, itri, size_itri);
		}
	}
}
// OpenMP並列領域終了

	free(r1set);

	// 計算時間の短縮のために後で必要になるデータを予め用意する
	// length : セグメントの長さ [m]
	// cumlng : セグメントの始点までの送信点からの累積長さ [m]
	// vector : セグメントの単位接線ベクトル
	for (int iray = 0; iray < NRay; iray++) {
		const int npos = Ray[iray].npos;
		Ray[iray].length = (double *)     malloc((npos - 1)     * sizeof(double));
		Ray[iray].cumlng = (double *)     malloc((npos - 1)     * sizeof(double));
		Ray[iray].vector = (double (*)[3])malloc((npos - 1) * 3 * sizeof(double));
		// 反射点に関するループ
		for (int ipos = 0; ipos < npos - 1; ipos++) {
			Ray[iray].length[ipos]
				= vector_distance(Ray[iray].pos[ipos + 1],
				                  Ray[iray].pos[ipos + 0]);
			Ray[iray].cumlng[ipos] = (ipos > 0)
				? Ray[iray].cumlng[ipos - 1]
				+ Ray[iray].length[ipos + 0] : 0;
			for (int k = 0; k < 3; k++) {
				Ray[iray].vector[ipos][k]
					= (Ray[iray].pos[ipos + 1][k]
					 - Ray[iray].pos[ipos + 0][k])
					/ MAX(Ray[iray].length[ipos], EPS);
			}
		}
	}

/*
	// debug
	int num[7] = {0, 0, 0, 0, 0, 0, 0};
	for (int iray = 0; iray < NRay; iray++) {
		for (int n = 0; n < 7; n++) {
			if (Ray[iray].npos == n) {
				num[n]++;
			}
		}
	}
	printf("%d %d %d %d %d %d %d\n", num[0], num[1], num[2], num[3], num[4], num[5], num[6]);
	printf("%d %d\n", NRay, num[0] + num[1] + num[2] + num[3] + num[4] + num[5] + num[6]);
*/
}


/*
機能:
	ローンチング法の本計算
	指定した受信点の近くを通るレイを探し、真の伝搬経路であるかチェックする
入力:
	irx : 受信点番号(=0,1,...)
入力/出力:
	npath : 伝搬経路の通し番号(=0,1,...)
	path : 伝搬経路データ
関数値:
	0/1 : 伝搬経路の数が上限に達したか
*/
int launching(int irx, int *npath, path_t *path)
{
	int skip, ends, edge;
	double pos1[3], pos2[3];

	// 作業配列
	double (*img)[3] = (double (*)[3])malloc((MaxRef + 2) * 3 * sizeof(double));
	double (*pos)[3] = (double (*)[3])malloc((MaxRef + 2) * 3 * sizeof(double));

	// 受信点座標
	double *rxpos = Rx[irx].pos;

	// 受信球半径のための角度パラメーター [rad] 要調整
	const double dangle = PI / NDivLaunch * 1.2;

	// レイに関するループ
	for (int iray = 0; iray < NRay; iray++) {
		// レイデータ
		ray_t *ray = &Ray[iray];

		// レイの線分に関するループ
		// 受信球に入る線分を取り出す
		// 直接波(と1回回折波)は直接法で計算されるのでipos=2から始まる
		int npos = 0;
		for (int ipos = Icomp[1] + 1; ipos < Ray[iray].npos - 1; ipos++) {
			// 始点から垂線の足までの距離 : lfoot
			const double lfoot = (rxpos[0] - ray->pos[ipos][0]) * ray->vector[ipos][0]
			                   + (rxpos[1] - ray->pos[ipos][1]) * ray->vector[ipos][1]
			                   + (rxpos[2] - ray->pos[ipos][2]) * ray->vector[ipos][2];

			// 垂線の足が線分外のときは無効(簡易テスト)
			if ((lfoot <= 0) || (lfoot >= ray->length[ipos])) {
				continue;
			}

			// 垂線の足の座標 : foot
			double foot[3];
			for (int k = 0; k < 3; k++) {
				foot[k] = ray->pos[ipos][k] + (lfoot * ray->vector[ipos][k]);
			}

			// 受信点と垂線の足までの距離の2乗
			const double r2min = (rxpos[0] - foot[0]) * (rxpos[0] - foot[0])
			                   + (rxpos[1] - foot[1]) * (rxpos[1] - foot[1])
			                   + (rxpos[2] - foot[2]) * (rxpos[2] - foot[2]);

			// 受信球の半径の2乗
			const double r = (ray->cumlng[ipos] + lfoot) * dangle;
			const double r2 = r * r;

			// 受信球に入ったときは頂点数を取得してループを抜け出す
			if (r2min < r2) {
				npos = ipos + 2;
				break;
			}
		}

		// どの線分も受信点の近くにないときは次のレイに進む
		// ほとんどのレイはここで除外される
		if (npos < Icomp[1] + 3) continue;

		// 正しい伝搬経路であるかテストする

		// テスト1 : 同じ伝搬経路が既に登録されていないか
		// 頂点数と三角形番号を調べる、軽い処理なので最初に行う
		skip = 0;
		for (int ipath = 0; ipath < *npath; ipath++) {
			//if (ray->npos != path[ipath].npos) continue;
			if (ray->npos == path[ipath].npos) {
				int dupl = 1;
				for (int ipos = 0; ipos < ray->npos; ipos++) {
					if (ray->itri[ipos] != path[ipath].itri[ipos]) {
						dupl = 0;
						break;
					}
				}
				if (dupl) {
					skip = 1;
					break;
				}
			}
		}
		if (skip) continue;

		// テスト2 : 正しい伝搬経路であるか
		// 準備 : 各頂点の三角形の鏡像を求める
		vector_copy(ray->pos[0], img[0]);
		for (int ipos = 1; ipos < npos - 1; ipos++) {
			imagepoint(img[ipos - 1], &Triangle[ray->itri[ipos]], img[ipos]);
		}
		// 伝搬経路は三角形と本当に交差しているか、同時に交点も求める
		skip = 0;
		vector_copy(rxpos, pos[npos - 1]);
		for (int ipos = npos - 2; ipos > 0; ipos--) {
			ends = 0;
			edge = 1;
			if (!intersect(pos[ipos + 1], img[ipos], &Triangle[ray->itri[ipos]], pos[ipos], ends, edge)) {
				skip = 1;
				break;
			}
		}
		if (skip) continue;

		// テスト3 : 伝搬経路の各線分内に障害物(他の三角形)はないか(見通し内であるか)
		skip = 0;
		vector_copy(img[0], pos[0]);
		// 各反射点間のテスト
		for (int ipos = 0; ipos < npos - 1; ipos++) {
			ends = 0;
			edge = 1;
			if (!los(pos[ipos], pos[ipos + 1], ends, edge)) {
				skip = 1;
				break;
			}
		}
		// 反射点の両側にわずかに移動した点間に障害物はないか
		// 反射点がエッジ上にあるとき希にすり抜ける現象を防ぐ
		const double eps = 1e-3;
		for (int ipos = 1; ipos < npos - 1; ipos++) {
			for (int k = 0; k < 3; k++) {
				pos1[k] = (1 - eps) * pos[ipos][k] + eps * pos[ipos - 1][k];
				pos2[k] = (1 - eps) * pos[ipos][k] + eps * pos[ipos + 1][k];
			}
			ends = 0;
			edge = 1;
			if (!los(pos1, pos2, ends, edge)) {
				skip = 1;
				break;
			}
		}
		if (skip) continue;

		// テスト4 : 同じ座標の伝搬経路はないか
		if (!newpath(npos, pos, *npath, path)) continue;

		// すべてのテストにパスしたものを伝搬経路に登録する

		// 頂点数
		path[*npath].npos = npos;

		// 頂点座標
		size_t size_pos = npos * 3 * sizeof(double);
		memcpy(path[*npath].pos, pos, size_pos);

		// 三角形番号
		size_t size_itri = npos * sizeof(int);
		memcpy(path[*npath].itri, ray->itri, size_itri);

		// 受信点の三角形番号は受信点番号とする
		path[*npath].itri[npos - 1] = irx;

		// 頂点のR/D/T種別
		for (int ipos = 0; ipos < npos; ipos++) {
			path[*npath].rdt[ipos] = 1;  // R
		}

		// 伝搬経路数++
		(*npath)++;

		// 伝搬経路数が上限に達したら終了する
		if (*npath >= MaxPath) return 1;
	}

	return 0;
}


// 送信点から放射されるレイの数
static int numray(void)
{
	int ntheta = NDivLaunch;

	int num = 0;
	for (int itheta = 0; itheta < ntheta; itheta++) {
		const double theta = 180 * (itheta + 0.5) / ntheta;
		num += (int)(2 * ntheta * sin(theta * DTOR)) + 1;
	}

	return num;
}


// 送信点から放射されるレイ集合の単位方向ベクトル
static void setray(double (*r1)[3])
{
	double theta1[3], phi1[3];
	int ntheta = NDivLaunch;

	int num = 0;
	for (int itheta = 0; itheta < ntheta; itheta++) {
		const double theta = 180 * (itheta + 0.5) / ntheta;
		const int nphi = (int)(2 * ntheta * sin(theta * DTOR)) + 1;
		for (int iphi = 0; iphi < nphi; iphi++) {
			const double phi = 360 * (iphi + 0.5) / nphi;
			polar_vector(theta, phi, r1[num], theta1, phi1);
			num++;
		}
	}
}
