/*
outputlog.c
*/

#include "ort.h"
#include "complex.h"
#include "vector.h"
#include "ort_prototype.h"

// field.log : 受信電力データ
static void fieldlog(const char fn[])
{
	FILE *fp;
	if ((fp = fopen(fn, "w")) == NULL) {
		fprintf(stderr, "*** file %s open error\n", fn);
		return;
	}

	const char fmt0[] = "%6d\n";
	const char fmt1[] = "%6d%11.3f%11.3f%11.3f%9.3f%9.3f%12.4e%12.4e\n";

	// 受信点の数
	fprintf(fp, fmt0, NRx[3]);

	// 受信点に関するループ
	for (int irx = 0; irx < NRx[3]; irx++) {
		fprintf(fp, fmt1,
			irx + 1,                                        // 受信点番号=1,2,...
			Rx[irx].pos[0], Rx[irx].pos[1], Rx[irx].pos[2], // 受信点のXYZ座標[m]
			10 * log10(MAX(Rx[irx].power[0], EPS2)),        // 受信電力(位相差あり)[dBW]
			10 * log10(MAX(Rx[irx].power[1], EPS2)),        // 受信電力(位相差なし)[dBW]
			Rx[irx].delay[0] / C * 1e9,                     // 平均遅延[nsec]
			Rx[irx].delay[1] / C * 1e9);                    // 遅延スプレッド[nsec]
	}

	fclose(fp);
}


// path.log : 伝搬経路データ
static void pathlog(const char fn[])
{
	FILE *fp;
	if ((fp = fopen(fn, "w")) == NULL) {
		fprintf(stderr, "*** file %s open error\n", fn);
		return;
	}

	const char fmt0[] = "%6d\n";
	const char fmt1[] = "%6d%4d%11.3f%11.3f%11.3f\n";
	const char fmt2[] = "%10d%4d%4d%9.3f%9.3f%9.3f%9.3f%11.3f%9.3f%9.3f\n";
	const char fmt3[] = "%14d%11.3f%11.3f%11.3f\n";
	double v_tx[3], r_tx, theta_tx, phi_tx;
	double v_rx[3], r_rx, theta_rx, phi_rx;

	// 受信点の数
	fprintf(fp, fmt0, NRx[3]);

	// 受信点に関するループ
	for (int irx = 0; irx < NRx[3]; irx++) {
		rx_t rx = Rx[irx];

		// 受信点番号(=1,2,...)、伝搬経路の数、受信点のXYZ座標[m]
		fprintf(fp, fmt1,
			irx + 1, rx.npath, rx.pos[0], rx.pos[1], rx.pos[2]);

		// 伝搬経路に関するループ
		for (int ipath = 0; ipath < rx.npath; ipath++) {
			const path_t path = rx.path[ipath];
			const int npos = path.npos;

			// 送信点から頂点1を見た角度
			vector_sub(path.pos[1], path.pos[0], v_tx);
			xyz2polar(v_tx, &r_tx, &theta_tx, &phi_tx);

			// 受信点から最後の頂点を見た角度
			vector_sub(path.pos[npos - 2], path.pos[npos - 1], v_rx);
			xyz2polar(v_rx, &r_rx, &theta_rx, &phi_rx);

			// 伝搬経路の長さ
			double lng = 0;
			for (int ipos = 0; ipos < npos - 1; ipos++) {
				lng += vector_distance(path.pos[ipos], path.pos[ipos + 1]);
			}

			// 受信電力と位相
			const d_complex_t e = rx.efield[ipath];
			const double power = 20 * log10(MAX(d_abs(e), EPS));
			const double phase = atan2(e.i, e.r) / DTOR;

			// 出力
			fprintf(fp, fmt2,
				ipath + 1,         // 伝搬経路番号=1,2,...
				npos,              // 頂点の数(直接波は2)
				path.itri[0] + 1,  // 送信点の番号=1,2,...
				theta_tx, phi_tx,  // 送信点から頂点1を見た角度(θ,φ)[度]
				theta_rx, phi_rx,  // 受信点から最後の頂点を見た角度(θ,φ)[度]
				lng,               // 伝搬経路の長さ[m]
				power, phase);     // 受信電力[dBW]と位相[度]

			// 頂点に関するループ
			for (int ipos = 0; ipos < npos; ipos++) {
				fprintf(fp, fmt3,
					ipos + 1, path.pos[ipos][0], path.pos[ipos][1], path.pos[ipos][2]);
			}
		}
	}

	fclose(fp);
}


// logファイル出力
// 観測点、観測線、観測面の受信点をすべて同一に扱う
void outputlog(const char fn_field[], const char fn_path[])
{
	// field.log
	if (Log[0]) {
		fieldlog(fn_field);
	}

	// path.log
	if (Log[1]) {
		pathlog(fn_path);
	}
}
