/*
plot3d_data.c
入力データ図形表示 (3D)
*/

#include "ort.h"
#include "ort_prototype.h"
#include "ev.h"

static void material_color(int, unsigned char []);

// 物体形状 (3D) (public)
// color = 0/1 : 単色/色分け
// fill = 0/1 : 線画/塗りつぶし
void plot3d_geom(int color, int fill)
{
	unsigned char rgb[3];

	// 三角形
	for (int n = 0; n < NTriangle; n++) {
		triangle_t t = Triangle[n];
		if (color) {
			// 色分け
			material_color(t.m, rgb);
			ev3d_setColor(rgb[0], rgb[1], rgb[2]);
		}
		else {
			// 灰色
			ev3d_setColor(200, 200, 200);
		}
		if (fill) {
			// 塗りつぶし
			ev3d_fillTriangle(t.v[0][0], t.v[0][1], t.v[0][2],
			                  t.v[1][0], t.v[1][1], t.v[1][2],
			                  t.v[2][0], t.v[2][1], t.v[2][2]);
		}
		else {
			// 線画
			ev3d_drawTriangle(t.v[0][0], t.v[0][1], t.v[0][2],
			                  t.v[1][0], t.v[1][1], t.v[1][2],
			                  t.v[2][0], t.v[2][1], t.v[2][2]);
		}
	}

	ev3d_setColor(0, 0, 0);
}


// 送信点 (3D) (public)
void plot3d_tx(void)
{
	ev3d_setColor(255, 0, 0);

	// 送信点
	const double htx = 0.003 * Lbound;
	for (int n = 0; n < NTx; n++) {
		tx_t *p = &Tx[n];
		ev3d_drawBox(p->pos[0] - htx, p->pos[1] - htx, p->pos[2] - htx,
		             p->pos[0] + htx, p->pos[1] + htx, p->pos[2] + htx);
	}

	ev3d_setColor(0, 0, 0);
}


// 受信点 (3D)
static void plot3d_rx(void)
{
	ev3d_setColor(30, 180, 30);

	// 観測点
	const double hrx = 0.003 * Lbound;
	for (int n = 0; n < NRx0d; n++) {
		rx0d_t *p = &Rx0d[n];
		ev3d_drawBox(p->pos[0] - hrx, p->pos[1] - hrx, p->pos[2] - hrx,
		             p->pos[0] + hrx, p->pos[1] + hrx, p->pos[2] + hrx);
	}

	// 観測線
	for (int n = 0; n < NRx1d; n++) {
		rx1d_t *p = &Rx1d[n];
		ev3d_drawLine(p->pos[0][0], p->pos[0][1], p->pos[0][2],
		              p->pos[1][0], p->pos[1][1], p->pos[1][2]);
	}

	// 観測面
	for (int n = 0; n < NRx2d; n++) {
		rx2d_t *p = &Rx2d[n];
		ev3d_drawLine(p->pos[0][0], p->pos[0][1], p->pos[0][2],
		              p->pos[1][0], p->pos[1][1], p->pos[1][2]);
		ev3d_drawLine(p->pos[1][0], p->pos[1][1], p->pos[1][2],
		              p->pos[2][0], p->pos[2][1], p->pos[2][2]);
		ev3d_drawLine(p->pos[2][0], p->pos[2][1], p->pos[2][2],
		              p->pos[3][0], p->pos[3][1], p->pos[3][2]);
		ev3d_drawLine(p->pos[3][0], p->pos[3][1], p->pos[3][2],
		              p->pos[0][0], p->pos[0][1], p->pos[0][2]);
	}

	ev3d_setColor(0, 0, 0);
}


// アンテナパターン (3D)
static void plot3d_antenna(antenna_t *a)
{
	const int ntheta = NDivAntenna;
	const int nphi = 2 * ntheta;
	double p[4][3];

	const double dtheta = 180.0 / ntheta;
	const double dphi   = 360.0 / nphi;
	for (int itheta = 0; itheta < ntheta; itheta++) {
	for (int iphi   = 0; iphi   < nphi;   iphi++  ) {
		const double theta0 = (itheta + 0) * dtheta;
		const double theta1 = (itheta + 1) * dtheta;
		const double phi0   = (iphi   + 0) * dphi;
		const double phi1   = (iphi   + 1) * dphi;
		const double r00 = rantenna(a, theta0, phi0);
		const double r10 = rantenna(a, theta1, phi0);
		const double r11 = rantenna(a, theta1, phi1);
		const double r01 = rantenna(a, theta0, phi1);
		polar2xyz(r00, theta0, phi0, p[0]);
		polar2xyz(r10, theta1, phi0, p[1]);
		polar2xyz(r11, theta1, phi1, p[2]);
		polar2xyz(r01, theta0, phi1, p[3]);
		// 色
		const double rav = (r00 + r10 + r11 + r01) / 4;
		ev3d_setColorV(rav, 1);
		// 四角形
		ev3d_fillQuadrangle(p[0][0], p[0][1], p[0][2], p[1][0], p[1][1], p[1][2], p[2][0], p[2][1], p[2][2], p[3][0], p[3][1], p[3][2]);
	}
	}

	// XYZ軸
	ev3d_setColor(0, 0, 0);
	ev3d_drawLine(0, 0, 0, 1, 0, 0);
	ev3d_drawLine(0, 0, 0, 0, 1, 0);
	ev3d_drawLine(0, 0, 0, 0, 0, 1);
}


// レイ (3D)
static void plot3d_ray(void)
{
	for (int iray = 0; iray < NRay; iray++) {
		double (*pos)[3] = Ray[iray].pos;
		const int npos = Ray[iray].npos;
		const int kpos = Ray[iray].itri[npos - 1] >= 0 ? npos : npos - 1;
		for (int ipos = 0; ipos < kpos - 1; ipos++) {
			//printf("%f %f %f\n", pos[ipos][0], pos[ipos][1], pos[ipos][2]);
			// 色分け
			if      (ipos == 0) {
				ev3d_setColor(255, 0, 0);
			}
			else if (ipos == 1) {
				ev3d_setColor(0, 0, 255);
			}
			else if (ipos == 2) {
				ev3d_setColor(0, 255, 0);
			}
			else {
				ev3d_setColor(255, 255, 0);
			}
			// 線分
			ev3d_drawLine(pos[ipos][0], pos[ipos][1], pos[ipos][2],
				pos[ipos + 1][0], pos[ipos + 1][1], pos[ipos + 1][2]);
		}
	}

	ev3d_setColor(0, 0, 0);
}


// 入力データ図形表示 (3D)
void plot3d_data(int id)
{
	char str[BUFSIZ];

	// 物体形状、送信点、受信点
	if (id == 0) {
		ev3d_init();
		ev3d_newPage();

		// plot
		plot3d_geom(1, 1);
		plot3d_tx();
		plot3d_rx();

		// title
		sprintf(str, "triangle=%d, edge=%d, Tx=%d, Rx0d=%d, Rx1d=%d, Rx2d=%d", NTriangle, NEdge, NTx, NRx0d, NRx1d, NRx2d);
		ev3d_drawTitle(str);

		// output
		ev3d_file(!HTML, !HTML ? "geom.ev3" : "geom3d.htm", 0);
		ev3d_output();
	}

	// アンテナパターン
	else if ((id == 1) && (NAntenna > 0)) {
		ev3d_init();
		for (int i = 0; i < NAntenna; i++) {
			ev3d_newPage();

			// plot
			plot3d_antenna(&Antenna[i]);

			// title
			sprintf(str, "antenna radiation pattern #%d", i + 1);
			ev3d_drawTitle(str);
			const double gain = 10 * log10(Antenna[i].gain);
			sprintf(str, "gain=%.3fdB", gain);
			ev3d_drawTitle(str);
		}

		// output
		ev3d_file(!HTML, !HTML ? "antenna.ev3" : "antenna3d.htm", 0);
		ev3d_output();
	}

	// レイ
	else if ((id == 2) && (MaxRef > 0)) {
		ev3d_init();
		ev3d_newPage();

		// plot
		plot3d_geom(0, 1);
		plot3d_ray();

		// title
		sprintf(str, "ray launching method, rays=%d", NRay);
		ev3d_drawTitle(str);

		// output
		ev3d_file(!HTML, !HTML ? "ray.ev3" : "ray3d.htm", 0);
		ev3d_output();
	}
}


// 物性値の色
static void material_color(int m, unsigned char rgb[])
{
	if      (m == 0) {
		rgb[0] = 240;
		rgb[1] = 240;
		rgb[2] = 240;
	}
	else if (m == 1) {
		rgb[0] = 200;
		rgb[1] = 200;
		rgb[2] = 200;
	}
	else if (m == 2) {
		rgb[0] = 255;
		rgb[1] = 220;
		rgb[2] = 180;
	}
	else if (m == 3) {
		rgb[0] = 200;
		rgb[1] = 255;
		rgb[2] = 200;
	}
	else if (m == 4) {
		rgb[0] = 200;
		rgb[1] = 200;
		rgb[2] = 255;
	}
	else {
		rgb[0] = 160;
		rgb[1] = 255;
		rgb[2] = 255;
	}
}
