/*
rdt.c
*/

#include "ort.h"
#include "complex.h"
#include "vector.h"
#include "ort_prototype.h"

static void material_file(int, double, d_complex_t [], d_complex_t []);

/*
機能:
	反射・回折・透過による電界の変化
	V=P, H=S
入力:
	path : 伝搬経路データ
	ipos : 頂点番号
入力/出力:
	e : 電界
*/
void rdt(path_t *path, int ipos, d_complex_t e[])
{
	assert(ipos > 0);
	assert(ipos < path->npos - 1);

	const int rdt = path->rdt[ipos];
	assert((rdt >= 1) && (rdt <= 3));

	// 反射(=1)または透過(=3)
	if ((rdt == 1) || (rdt == 3)) {
		assert(path->itri[ipos] >= 0);
		assert(path->itri[ipos] < NTriangle);

		// 三角形
		triangle_t *tri = &Triangle[path->itri[ipos]];

		// vector ri
		double ri[3];
		for (int k = 0; k < 3; k++) {
			ri[k] = path->pos[ipos][k] - path->pos[ipos - 1][k];
		}
		vector_normalize(ri, ri);

		// 入射角[度]
		const double cost = fabs(vector_innerprod(tri->n, ri));
		const double ainc = (cost < 1) ? acos(cost) / DTOR : 0;
		//printf("%e %e\n", cost, ainc);

		// 垂直入射は微小量ずらす(P/S偏波のあいまいさをなくすため)
		if (fabs(ainc) < EPS) {
			for (int k = 0; k < 3; k++) {
				ri[k] += EPS;
			}
			vector_normalize(ri, ri);
		}
		//printf("ri : %e %e %e\n", ri[0], ri[1], ri[2]);

		// vector hi
		double hi[3];
		vector_outerprod(ri, tri->n, hi);
		vector_normalize(hi, hi);
		//printf("tri->n : %e %e %e\n", tri->n[0], tri->n[1], tri->n[2]);
		//printf("hi : %e %e %e\n", hi[0], hi[1], hi[2]);

		// vector vi = hi X ri
		double vi[3];
		vector_outerprod(hi, ri, vi);

		// vector hr = hi
		double hr[3];
		vector_copy(hi, hr);

		// vector vr
		double vr[3];
		if (rdt == 1) {
			// 反射
			// rr = ri - 2 (n * ri) n
			double rr[3];
			const double nri = vector_innerprod(tri->n, ri);
			for (int k = 0; k < 3; k++) {
				rr[k] = ri[k] - (2 * nri * tri->n[k]);
			}
			// vr = rr X hr
			vector_outerprod(rr, hr, vr);
		}
		else {
			// 透過
			// vr = vi
			vector_copy(vi, vr);
		}

		// 三角形の属性
		const int m = tri->m;
		const int mtype = Material[m].mtype;
		const int trans = (mtype == 2) || (mtype == 3);

		// 複素比誘電率
		const d_complex_t cepsr = d_complex(
			+ Material[m].epsr,
			- Material[m].sigma / (2 * PI * Frequency * EPS0));

		// 反射・透過係数 (P/S)
		d_complex_t r[2], t[2];
		if (tri->m == 1) {
			// PEC : R=-1, T=0
			r[0] = r[1] = d_complex(-1, 0);
			t[0] = t[1] = d_complex(0, 0);
		}
		else if ((rdt == 1) && !trans) {
			// 反射のみ
			if ((mtype == 1) || (mtype == 2)) {
				// 数値
				fresnel_reflection(cepsr, ainc, r);
			}
			else {
				// ファイル
				material_file(m, ainc, r, t);
			}
			t[0] = t[1] = d_complex(0, 0);  // 透過なし
			//printf("%f %f %f %f %f %f %f\n", ainc, cepsr.r, cepsr.i, r[0].r, r[0].i, r[1].r, r[1].i);
		}
		else {
			// 反射と透過
			if ((mtype == 1) || (mtype == 2)) {
				// 数値
				reftrans(cepsr, Material[tri->m].thick, ainc, Frequency, r, t);
			}
			else {
				// ファイル
				material_file(m, ainc, r, t);
			}
			//printf("R : %e %e %e %e\n", r[0].r, r[0].i, r[1].r, r[1].i);
			//printf("T : %e %e %e %e\n", t[0].r, t[0].i, t[1].r, t[1].i);
		}

		// 反射・透過後の電界(P + S) : e
		const d_complex_t eivi = d_add3(d_rmul(vi[0], e[0]), d_rmul(vi[1], e[1]), d_rmul(vi[2], e[2]));
		const d_complex_t eihi = d_add3(d_rmul(hi[0], e[0]), d_rmul(hi[1], e[1]), d_rmul(hi[2], e[2]));
		const d_complex_t cv = (rdt == 1) ? r[0] : t[0];
		const d_complex_t ch = (rdt == 1) ? r[1] : t[1];
		for (int k = 0; k < 3; k++) {
			e[k] = d_add(d_rmul(vr[k], d_mul(cv, eivi)),
			             d_rmul(hr[k], d_mul(ch, eihi)));
		}
		//printf("%e %e %e %e %e %E\n", e[0].r, e[0].i, e[1].r, e[1].i, e[2].r, e[2].i);
	}

	// 回折(=2)
	else if (rdt == 2) {
		const double eps = EPS * Lbound;

		// vector ri, ro
		double ri[3], ro[3], rio[3];
		for (int k = 0; k < 3; k++) {
			ri[k] = path->pos[ipos    ][k] - path->pos[ipos - 1][k];
			ro[k] = path->pos[ipos + 1][k] - path->pos[ipos    ][k];
			rio[k] = ri[k] + ro[k];
		}

		// vector rs (S)
		double rs[3];
		vector_outerprod(ri, ro, rs);

		// 回折因子
		const double l1 = vector_length(ri);
		const double l2 = vector_length(ro);
		if ((l1 < eps) || (l2 < eps)) return;
		const double lfoot = vector_length(rs) / vector_length(rio);
		const double z = lfoot * sqrt((2 * Frequency / C) * (1 / l1 + 1 / l2));
		const double fdif = sqrt(0.5) * fresnel_integral(z);

		// vector rpi, rpo (P)
		double rpi[3], rpo[3];
		vector_outerprod(ri, rs, rpi);
		vector_outerprod(ro, rs, rpo);

		// unit vector
		vector_normalize(rs, rs);
		vector_normalize(rpi, rpi);
		vector_normalize(rpo, rpo);

		// 回折後の電界(P + S) : e
		d_complex_t es = d_add3(d_rmul(rs[0],  e[0]), d_rmul(rs[1],  e[1]), d_rmul(rs[2],  e[2]));
		d_complex_t ep = d_add3(d_rmul(rpi[0], e[0]), d_rmul(rpi[1], e[1]), d_rmul(rpi[2], e[2]));
		for (int k = 0; k < 3; k++) {
			e[k] = d_rmul(fdif, d_add(d_rmul(rs[k], es), d_rmul(rpo[k], ep)));
		}
	}
}


// 反射・透過係数をファイルから補間して求める
static void material_file(int m, double angle, d_complex_t r[], d_complex_t t[])
{
	assert((m >= 0) && (m < NMaterial));
	assert(Material[m].mtype == 3);  // ファイル

	const int nangle = Material[m].nangle;
	assert(nangle > 0);
	assert(Material[m].angle[nangle - 1] > Material[m].angle[0]);  // 昇順

	r[0] = r[1] = t[0] = t[1] = d_complex(0, 0);

	if      (angle <= Material[m].angle[0]) {
		// 例外
		r[0] = Material[m].rv[0];
		r[1] = Material[m].rh[0];
		t[0] = Material[m].tv[0];
		t[1] = Material[m].th[0];
	}
	else if (angle >= Material[m].angle[nangle - 1]) {
		// 例外
		r[0] = Material[m].rv[nangle - 1];
		r[1] = Material[m].rh[nangle - 1];
		t[0] = Material[m].tv[nangle - 1];
		t[1] = Material[m].th[nangle - 1];
	}
	else {
		for (int i = 0; i < nangle - 1; i++) {
			// 補間
			if ((fabs(Material[m].angle[i + 1] - Material[m].angle[i]) > EPS) &&
				((angle - Material[m].angle[i]) * (angle - Material[m].angle[i + 1]) <= 0)) {
				const double f = (angle - Material[m].angle[i]) / (Material[m].angle[i + 1] - Material[m].angle[i]);
				r[0] = d_add(d_rmul(1 - f, Material[m].rv[i]),
				             d_rmul(    f, Material[m].rv[i + 1]));
				r[1] = d_add(d_rmul(1 - f, Material[m].rh[i]),
				             d_rmul(    f, Material[m].rh[i + 1]));
				t[0] = d_add(d_rmul(1 - f, Material[m].tv[i]),
				             d_rmul(    f, Material[m].tv[i + 1]));
				t[1] = d_add(d_rmul(1 - f, Material[m].th[i]),
				             d_rmul(    f, Material[m].th[i + 1]));
				break;
			}
		}
	}
}
