/*
rxstat.c
*/

#include "ort.h"
#include "complex.h"
#include "vector.h"
#include "ort_prototype.h"


/*
機能:
	受信電力と遅延時間を計算する
入力:
	npath : 伝搬経路の数
	path : 伝搬経路データ
	efield : 伝搬経路ごとの受信電界(複素数)
出力:
	power : 受信電力(位相差あり/なし)
	delay : 平均遅延、遅延スプレッド
*/
void rxstat(int npath, path_t *path, d_complex_t *efield, double power[], double delay[])
{
	power[0] = power[1] = delay[0] = delay[1] = 0;

	if (npath <= 0) return;

	// 受信電力(位相差あり) : 電界の和の2乗ノルム
	d_complex_t csum = d_complex(0, 0);
	for (int ipath = 0; ipath < npath; ipath++) {
		csum = d_add(csum, efield[ipath]);
	}
	power[0] = d_norm(csum);

	// 受信電力(位相差なし) : 電界の2乗ノルムの和
	double psum = 0;
	for (int ipath = 0; ipath < npath; ipath++) {
		psum += d_norm(efield[ipath]);
	}
	power[1] = psum;

	// 送信点からの伝搬経路の長さ
	double *length = (double *)malloc(npath * sizeof(double));
	for (int ipath = 0; ipath < npath; ipath++) {
		double lsum = 0;
		for (int ipos = 0; ipos < path[ipath].npos - 1; ipos++) {
			lsum += vector_distance(path[ipath].pos[ipos], path[ipath].pos[ipos + 1]);
		}
		length[ipath] = lsum;
	}

	// 最短伝搬経路長
	double lmin = length[0];
	for (int ipath = 0; ipath < npath; ipath++) {
		lmin = MIN(lmin, length[ipath]);
	}

	// 平均遅延と遅延スプレッド
	double sum0 = 0, sum1 = 0, sum2 = 0;
	for (int ipath = 0; ipath < npath; ipath++) {
		const double p = d_norm(efield[ipath]);
		sum0 += p;
		sum1 += p * (length[ipath] - lmin);
		sum2 += p * (length[ipath] - lmin) * (length[ipath] - lmin);
	}
	if (sum0 > 0) {
		delay[0] = sum1 / sum0;
		const double tmp = (sum2 / sum0) - (delay[0] * delay[0]);
		delay[1] = (tmp > 0) ? sqrt(tmp) : 0;
	}

	// free
	free(length);
}
