/*
setup.c
*/

#include "ort.h"
#include "vector.h"
#include "ort_prototype.h"

static void error_check(void);
static void setup_normal(void);
static double boundingbox(void);
static void setup_edge(void);
static void setup_edge_2(void);
static void setup_Rx(void);


// 各種セットアップ
void setup(void)
{
	// エラーチェック
	error_check();

	// 三角形の単位法線ベクトル
	setup_normal();

	// 計算対象の代表的な大きさ
	Lbound = boundingbox();

	// 真の稜線を探す（回折波の計算時間を減らすため）
	if (Icomp[2]) {
		setup_edge();
		setup_edge_2();
	}

	// 受信点データ作成
	setup_Rx();

	// アンテナ利得
	for (int n = 0; n < NAntenna; n++) {
		Antenna[n].gain = gain(&Antenna[n], NDivAntenna);
		//printf("gain : %d %d %f\n", n, Antenna[n].type, Antenna[n].gain);
	}
}


// エラーチェック
static void error_check(void)
{
	for (int n = 0; n < NTriangle; n++) {
		if ((Triangle[n].m < 0) || (Triangle[n].m >= NMaterial)) {
			Triangle[n].m = 1;
		}
	}
	for (int n = 0; n < NTx; n++) {
		if ((Tx[n].antenna < 0) || (Tx[n].antenna >= NAntenna)) {
			Tx[n].antenna = 0;
		}
	}
	for (int n = 0; n < NRx0d; n++) {
		if ((Rx0d[n].antenna < 0) || (Rx0d[n].antenna >= NAntenna)) {
			Rx0d[n].antenna = 0;
		}
	}
	for (int n = 0; n < NRx1d; n++) {
		if ((Rx1d[n].antenna < 0) || (Rx1d[n].antenna >= NAntenna)) {
			Rx1d[n].antenna = 0;
		}
	}
	for (int n = 0; n < NRx2d; n++) {
		if ((Rx2d[n].antenna < 0) || (Rx2d[n].antenna >= NAntenna)) {
			Rx2d[n].antenna = 0;
		}
	}
}


// 三角形の単位法線ベクトル
static void setup_normal(void)
{
	double v12[3], v13[3], vn[3];

	for (int n = 0; n < NTriangle; n++) {
		triangle_t *t = &Triangle[n];
		vector_sub(t->v[1], t->v[0], v12);
		vector_sub(t->v[2], t->v[0], v13);
		vector_outerprod(v12, v13, vn);
		vector_normalize(vn, t->n);
	}
}


// 計算対象の代表的な大きさ (bounding box)
static double boundingbox(void)
{
	double pmin[] = {+1/EPS, +1/EPS, +1/EPS};
	double pmax[] = {-1/EPS, -1/EPS, -1/EPS};
	for (int n = 0; n < NTriangle; n++) {
		triangle_t *t = &Triangle[n];
		for (int k = 0; k < 3; k++) {
			pmin[k] = MIN(pmin[k], MIN(t->v[0][k], MIN(t->v[1][k], t->v[2][k])));
			pmax[k] = MAX(pmax[k], MAX(t->v[0][k], MAX(t->v[1][k], t->v[2][k])));
		}
	}
	//printf("%f %f %f %f %f %f\n", pmin[0], pmin[1], pmin[2], pmax[0], pmax[1], pmax[2]);

	return sqrt((pmax[0] - pmin[0]) * (pmax[0] - pmin[0]) +
	            (pmax[1] - pmin[1]) * (pmax[1] - pmin[1]) +
	            (pmax[2] - pmin[2]) * (pmax[2] - pmin[2]));
}


// 稜線を求める（回折波で使用）
static void setup_edge(void)
{
	// (1) 稜線候補を探す、面内部の線分を除く

	// 作業変数
	int nedge = 0;
	double (*edge)[2][3] = (double (*)[2][3])malloc(ARRAY_INC * 2 * 3 * sizeof(double));

	// 三角形に関するループ
	for (int i = 0; i < NTriangle; i++) {
		triangle_t *ti = &Triangle[i];
		// 三辺に関するループ
		for (int iv = 0; iv < 3; iv++) {
			const double x1i = ti->v[iv + 0][0];
			const double y1i = ti->v[iv + 0][1];
			const double z1i = ti->v[iv + 0][2];
			const double x2i = ti->v[iv + 1][0];
			const double y2i = ti->v[iv + 1][1];
			const double z2i = ti->v[iv + 1][2];

			// 新しい稜線であるかチェックする
			int newedge = 1;
			for (int j = 0; j < NTriangle; j++) {
				triangle_t *tj = &Triangle[j];
				const double dn = fabs(fabs(vector_innerprod(ti->n, tj->n)) - 1);
				for (int jv = 0; jv < 3; jv++) {
					if ((i == j) && (iv == jv)) {
						continue;  // 同じ辺は飛ばす
					}
					const double x1j = tj->v[jv + 0][0];
					const double y1j = tj->v[jv + 0][1];
					const double z1j = tj->v[jv + 0][2];
					const double x2j = tj->v[jv + 1][0];
					const double y2j = tj->v[jv + 1][1];
					const double z2j = tj->v[jv + 1][2];
					const double d1 = fabs(x1i - x1j) + fabs(y1i - y1j) + fabs(z1i - z1j)
					                + fabs(x2i - x2j) + fabs(y2i - y2j) + fabs(z2i - z2j);
					const double d2 = fabs(x1i - x2j) + fabs(y1i - y2j) + fabs(z1i - z2j)
					                + fabs(x2i - x1j) + fabs(y2i - y1j) + fabs(z2i - z1j);
					// 稜線iの線分は稜線jの線分と一致し、かつ(ほぼ)同一平面内にある：稜線ではない
					//printf("%d-%d %d-%d %f %f %f\n", i, iv, j, jv, d1, d2, dn);
					if (((d1 < 1e-6) || (d2 < 1e-6)) && (dn < 1e-2)) {
						newedge = 0;
						break;
					}
				}
				if (!newedge) {
					break;
				}
			}

			// 新しい稜線を追加する
			if (newedge) {
				// 点1
				edge[nedge][0][0] = x1i;
				edge[nedge][0][1] = y1i;
				edge[nedge][0][2] = z1i;

				// 点2
				edge[nedge][1][0] = x2i;
				edge[nedge][1][1] = y2i;
				edge[nedge][1][2] = z2i;
				//printf("%d %f %f %f %f %f %f\n", nedge, x1i, y1i, z1i, x2i, y2i, z2i);

				nedge++;

				if ((nedge % ARRAY_INC) == 0) {
					edge = (double (*)[2][3])realloc(edge, (nedge + ARRAY_INC) * 2 * 3 * sizeof(double));
				}
			}
		}
	}

	// (2) 重複する稜線を除く

	NEdge = 0;
	Edge = (double (*)[2][3])malloc(ARRAY_INC * 2 * 3 * sizeof(double));

	for (int i = 0; i < nedge; i++) {
		const double x1i = edge[i][0][0];
		const double y1i = edge[i][0][1];
		const double z1i = edge[i][0][2];
		const double x2i = edge[i][1][0];
		const double y2i = edge[i][1][1];
		const double z2i = edge[i][1][2];

		// 新しい稜線か
		int newedge = 1;
		for (int j = i + 1; j < nedge; j++) {
			const double x1j = edge[j][0][0];
			const double y1j = edge[j][0][1];
			const double z1j = edge[j][0][2];
			const double x2j = edge[j][1][0];
			const double y2j = edge[j][1][1];
			const double z2j = edge[j][1][2];
			// 稜線iの線分12は稜線jの線分12または線分21と一致するか
			const double d1 = fabs(x1i - x1j) + fabs(y1i - y1j) + fabs(z1i - z1j)
			                + fabs(x2i - x2j) + fabs(y2i - y2j) + fabs(z2i - z2j);
			const double d2 = fabs(x1i - x2j) + fabs(y1i - y2j) + fabs(z1i - z2j)
			                + fabs(x2i - x1j) + fabs(y2i - y1j) + fabs(z2i - z1j);
			if (((d1 < 1e-6) || (d2 < 1e-6))) {
				newedge = 0;
				break;
			}
		}
		//printf("%d %d\n", i, newedge);

		// 新しい稜線を追加する
		if (newedge) {
			// 点1
			Edge[NEdge][0][0] = x1i;
			Edge[NEdge][0][1] = y1i;
			Edge[NEdge][0][2] = z1i;

			// 点2
			Edge[NEdge][1][0] = x2i;
			Edge[NEdge][1][1] = y2i;
			Edge[NEdge][1][2] = z2i;

			NEdge++;

			if ((NEdge % ARRAY_INC) == 0) {
				Edge = (double (*)[2][3])realloc(Edge, (NEdge + ARRAY_INC) * 2 * 3 * sizeof(double));
			}
		}
	}
	//printf("%d %d\n", nedge, NEdge);

	free(edge);
}


// 送信点と稜線はLOSであるか（回折波計算の組み合わせを減らすため）
static void setup_edge_2(void)
{
	LosTxEdge = (int **)malloc(NTx * sizeof(int *));
	for (int itx = 0; itx < NTx; itx++) {
		LosTxEdge[itx] = (int *)malloc(NEdge * sizeof(int));
	}

	const int ends = 0;  // 両端は含めないために0
	const int edge = 0;  // 近似計算なので0/1どちらでもよい
	for (int itx = 0; itx < NTx; itx++) {
		for (int iedge = 0; iedge < NEdge; iedge++) {
			double pos[5][3];
			int ilos[5];
			// 頂点1、頂点2、中間の3点、計5点でテストする
			for (int k = 0; k < 3; k++) {
				pos[0][k] = Edge[iedge][0][k];
				pos[1][k] = Edge[iedge][1][k];
				pos[2][k] = (3 * pos[0][k] +     pos[1][k]) / 4;
				pos[3][k] = (    pos[0][k] +     pos[1][k]) / 2;
				pos[4][k] = (    pos[0][k] + 3 * pos[1][k]) / 4;
			}
			for (int i = 0; i < 5; i++) {
				ilos[i] = los(Tx[itx].pos, pos[i], ends, edge);
			}
			// 1点でもLOSのときLOSとみなす
			LosTxEdge[itx][iedge] = ilos[0] || ilos[1] || ilos[2] || ilos[3] || ilos[4];
		}
	}
/*
	int sum = 0;
	for (int itx = 0; itx < NTx; itx++) {
		for (int iedge = 0; iedge < NEdge; iedge++) {
			sum += LosTxEdge[itx][iedge];
		}
	}
	printf("%d %d\n", sum, NTx * NEdge);
*/
}


// 受信点データ作成
static void setup_Rx(void)
{
	// 観測点の受信点の数 : NRx[0]
	NRx[0] = NRx0d;

	// 観測線の受信点の数 : NRx[1]
	int nrx1d = 0;
	for (int n = 0; n < NRx1d; n++) {
		nrx1d += Rx1d[n].div;
	}
	NRx[1] = nrx1d;

	// 観測面の受信点の数 : NRx[2]
	int nrx2d = 0;
	for (int n = 0; n < NRx2d; n++) {
		nrx2d += Rx2d[n].div[0] * Rx2d[n].div[1];
	}
	NRx[2] = nrx2d;

	// 受信点の数の合計 : NRx[3]
	NRx[3] = NRx[0] + NRx[1] + NRx[2];

	// 受信点配列Rx作成
	Rx = (rx_t *)malloc(NRx[3] * sizeof(rx_t));

	// 受信点の座標とアンテナ番号
	int irx = 0;  // 受信点の通し番号
	// 観測点
	for (int n = 0; n < NRx0d; n++) {
		for (int k = 0; k < 3; k++) {
			Rx[irx].pos[k] = Rx0d[n].pos[k];
		}
		Rx[irx].antenna = Rx0d[n].antenna;
		irx++;
	}
	// 観測線
	for (int n = 0; n < NRx1d; n++) {
		for (int div = 0; div < Rx1d[n].div; div++) {
			const double a = (double)(div + 0.5) / Rx1d[n].div;
			for (int k = 0; k < 3; k++) {
				Rx[irx].pos[k] = (1 - a) * Rx1d[n].pos[0][k]
				               + (    a) * Rx1d[n].pos[1][k];
			}
			Rx[irx].antenna = Rx1d[n].antenna;
			irx++;
		}
	}
	// 観測面
	for (int n = 0; n < NRx2d; n++) {
		for (int div12 = 0; div12 < Rx2d[n].div[0]; div12++) {
			for (int div14 = 0; div14 < Rx2d[n].div[1]; div14++) {
				const double a12 = (double)(div12 + 0.5) / Rx2d[n].div[0];
				const double a14 = (double)(div14 + 0.5) / Rx2d[n].div[1];
				for (int k = 0; k < 3; k++) {
					Rx[irx].pos[k] = (1 - a12) * (1 - a14) * Rx2d[n].pos[0][k]
					               + (    a12) * (1 - a14) * Rx2d[n].pos[1][k]
					               + (    a12) * (    a14) * Rx2d[n].pos[2][k]
					               + (1 - a12) * (    a14) * Rx2d[n].pos[3][k];
				}
				Rx[irx].antenna = Rx2d[n].antenna;
				irx++;
			}
		}
	}
	assert(irx == NRx[3]);
}
