/*
trace.c
直接法
*/

#include "ort.h"
#include "vector.h"
#include "ort_prototype.h"

// 直接波 (direct)
int trace_s(int itx, int irx, int *npath, path_t *path)
{
	const int npos = 2;  // 頂点数
	double pos[2][3];
	int ends, edge;

	vector_copy(Tx[itx].pos, pos[0]);  // 点0=送信点
	vector_copy(Rx[irx].pos, pos[1]);  // 点1=受信点

	// 点0-点1がLOSなら伝搬経路に追加する
	// 三角形の辺上を通るときは直接波は通らないと考える(edge=1)
	ends = 0;  // 送信点と受信点はLOS判定に含めないために0
	edge = 1;  // 送受信間に稜線があるときの直接波は認めないために1
	if (los(pos[0], pos[1], ends, edge)) {
		path[*npath].npos = npos;
		memcpy(path[*npath].pos, pos, npos * 3 * sizeof(double));
		path[*npath].itri[0] = itx;
		path[*npath].itri[1] = irx;
		(*npath)++;
		if (*npath >= MaxPath) return 1;  // 伝搬経路最大数に達したので探索終了
	}

	return 0;
}


// 1回回折波 (diffraction)
int trace_d(int itx, int irx, int *npath, path_t *path)
{
	const int npos = 3;  // 頂点数
	double pos[3][3], pdif[5][3];  // TODO pdif[3][3]
	int ends, edge;

	vector_copy(Tx[itx].pos, pos[0]);  // 点0=送信点
	vector_copy(Rx[irx].pos, pos[2]);  // 点2=受信点

	// 送受信間が見通し内のときは回折波は考えない
	// 稜線上を通るときも見通し外と考えて次に進む
	ends = 0;  // 送信点と受信点はLOS判定に含めないために0
	edge = 1;  // 送受信間に稜線があるとき回折波の計算に進むために1
	if (los(pos[0], pos[2], ends, edge)) {
		return 0;
	}

	// 稜線に関するループ
	ends = 1;  // 両側が閉じた偽の稜線を通過させないために1?
	edge = 0;  // 稜線を通過する回折波を許すために0
	for (int n = 0; n < NEdge; n++) {
		// 簡易チェック、精度を落とさず計算時間を約10%削減 (cf. setup.c)
		if ((Icomp[2] == 2) && !LosTxEdge[itx][n]) continue;

		// 以下の条件が成り立つとき回折波が存在するので伝搬経路に追加する
		// (1) 回折波が存在する
		// (2) 点0-回折点1-点2がLOSであるか点0-回折点2-点2がLOSである
		// (3) 新しい伝搬経路である
		if (diffract(pos[0], pos[2], Edge[n][0], Edge[n][1], pdif)) {
			// (真の)回折点
			vector_copy(pdif[0], pos[1]);

			// los1 : 通常の回折波ではtrue
			// los2 : これも考慮すると回折波の計算がより正確になるが計算時間が2倍になる
			const int los1 = los(pos[0], pdif[1], ends, edge) && los(pdif[1], pos[2], ends, edge);
			const int los2 = (Icomp[2] == 2) ? 0 : los(pos[0], pdif[2], ends, edge) && los(pdif[2], pos[2], ends, edge);
			if (((los1) ||
			     (los2)) &&
			     newpath(npos, pos, *npath, path)) {
				path[*npath].npos = npos;
				memcpy(path[*npath].pos, pos, npos * 3 * sizeof(double));
				path[*npath].rdt[1] = 2;  // D
				path[*npath].itri[0] = itx;
				path[*npath].itri[1] = -1;    // 回折波は物性値によらないのでダミー
				path[*npath].itri[2] = irx;
				(*npath)++;
				if (*npath >= MaxPath) return 1;  // 伝搬経路最大数に達したので探索終了
			}
		}
	}

	return 0;
}


// 1回透過波 (transmission)
int trace_t(int itx, int irx, int *npath, path_t *path)
{
	const int npos = 3;  // 頂点数
	double pos[3][3];
	int ends, edge;

	vector_copy(Tx[itx].pos, pos[0]);  // 点0=送信点
	vector_copy(Rx[irx].pos, pos[2]);  // 点2=受信点

	// 三角形に関するループ
	for (int n = 0; n < NTriangle; n++) {
		triangle_t *tri = &Triangle[n];
		assert((tri->m >= 0) && (tri->m < NMaterial));
		const int mtype = Material[tri->m].mtype;
		const int trans = (mtype == 2) || (mtype == 3);
		// 以下の条件が成り立つとき透過波が存在するので伝搬経路に追加する
		// (1) 材質が透過波を通す
		// (2) 点02が三角形を通る
		// (3) 点01がLOSである
		// (4) 点12がLOSである
		// (5) 新しい伝搬経路である
		ends = 0;  // 送信点と受信点はLOS判定に含めないために0
		edge = 1;  // 4角形の対角線上を通るときは2個の経路を許すために1
		if (trans &&
		    intersect(pos[0], pos[2], tri, pos[1], ends, edge)) {
			ends = 0;  // 線分の両端はLOS判定に含めないために0
			edge = 0;  // 線分内に他の三角形の稜線があるときも許すため0(1でも実害ない)
			if (los(pos[0], pos[1], ends, edge) &&
			    los(pos[1], pos[2], ends, edge) &&
			    newpath(npos, pos, *npath, path)) {
				path[*npath].npos = npos;
				memcpy(path[*npath].pos, pos, npos * 3 * sizeof(double));
				path[*npath].rdt[1] = 3;  // T
				path[*npath].itri[0] = itx;
				path[*npath].itri[1] = n;      // 三角形番号=0,1,...
				path[*npath].itri[2] = irx;
				(*npath)++;
				if (*npath >= MaxPath) return 1;  // 伝搬経路最大数に達したので探索終了
			}
		}
	}

	return 0;
}


// 1回反射波 (reflection)
int trace_r(int itx, int irx, int *npath, path_t *path)
{
	const int npos = 3;  // 頂点数
	double pos[3][3], pimg[3], pos1[3], pos2[3];
	int ends, edge;
	const double eps = 1e-3;

	vector_copy(Tx[itx].pos, pos[0]);  // 点0=送信点
	vector_copy(Rx[irx].pos, pos[2]);  // 点2=受信点

	// 三角形に関するループ
	for (int n = 0; n < NTriangle; n++) {
		triangle_t *tri = &Triangle[n];
		imagepoint(pos[2], tri, pimg);  // 受信点の鏡像点2'
		// 以下の条件が成り立つとき1回反射波が存在するので伝搬経路に追加する
		// (1) 点0-2'が三角形を通る
		// (2) 点0-1がLOSである
		// (3) 点1-2がLOSである
		// (4) 点1前後間がLOSである
		// (5) 新しい伝搬経路である
		ends = 0;  // 送信点と受信点はLOS判定に含めないので0
		edge = 1;  // 4角形の対角線上を通るときを許すために1
		if (intersect(pos[0], pimg, tri, pos[1], ends, edge)) {
			// 反射点の送信点、受信点側のごく近傍の点
			for (int k = 0; k < 3; k++) {
				pos1[k] = (1 - eps) * pos[1][k] + eps * pos[0][k];
				pos2[k] = (1 - eps) * pos[1][k] + eps * pos[2][k];
			}
			ends = 0;  // 線分の両端はLOS判定に含めないために0
			edge = 1;  // 線分内に他の三角形の稜線があるときを除くために1
			if (los(pos[0], pos[1], ends, edge) &&
			    los(pos[1], pos[2], ends, edge) &&
			    los(pos1, pos2, ends, edge) &&
			    newpath(npos, pos, *npath, path)) {
				path[*npath].npos = npos;
				memcpy(path[*npath].pos, pos, npos * 3 * sizeof(double));
				path[*npath].rdt[1] = 1;  // R
				path[*npath].itri[0] = itx;
				path[*npath].itri[1] = n;      // 三角形番号=0,1,...
				path[*npath].itri[2] = irx;
				(*npath)++;
				if (*npath >= MaxPath) return 1;  // 伝搬経路最大数に達したので探索終了
			}
		}
	}

	return 0;
}
