/*
utils.c

utilities
*/

#include "ort.h"
#include "complex.h"
#include "vector.h"
#include "ort_prototype.h"


// 新しい伝搬経路であるか?、座標値を調べる
int newpath(int npos, double (*pos)[3], int npath, path_t *path)
{
	for (int ipath = 0; ipath < npath; ipath++) {
		if (path[ipath].npos != npos) continue;
		int diff = 0;
		for (int ipos = 0; ipos < npos; ipos++) {
			if (vector_distance(path[ipath].pos[ipos], pos[ipos]) > EPS) {
				diff = 1;
				break;
			}
		}
		if (diff == 0) {
			return 0;
		}
	}

	return 1;
}


// 三角形による鏡像点
void imagepoint(const double pos[], triangle_t *t, double img[])
{
	double v1[3], v2[3], v3[3], n[3];

	// normal
	vector_sub(t->v[1], t->v[0], v1);
	vector_sub(t->v[2], t->v[0], v2);
	vector_outerprod(v1, v2, v3);
	vector_normalize(v3, n);

	// image
	vector_sub(pos, t->v[0], v1);
	const double tmp = vector_innerprod(v1, n);
	img[0] = pos[0] - (2 * tmp * n[0]);
	img[1] = pos[1] - (2 * tmp * n[1]);
	img[2] = pos[2] - (2 * tmp * n[2]);
}


// 極座標(r, theta, phi)方向の単位ベクトル
// theta, phi [deg]
void polar_vector(double theta, double phi, double r1[], double theta1[], double phi1[])
{
	double sint = sin(theta * DTOR);
	double cost = cos(theta * DTOR);
	double sinp = sin(phi   * DTOR);
	double cosp = cos(phi   * DTOR);

	r1[0] = + sint * cosp;
	r1[1] = + sint * sinp;
	r1[2] = + cost;

	theta1[0] = + cost * cosp;
	theta1[1] = + cost * sinp;
	theta1[2] = - sint;

	phi1[0] = - sinp;
	phi1[1] = + cosp;
	phi1[2] = 0;
}


// 極座標(r, theta, phi)をXYZ座標に変換する
// theta, phi [deg]
void polar2xyz(double r, double theta, double phi, double pos[])
{
	pos[0] = r * sin(theta * DTOR) * cos(phi * DTOR);
	pos[1] = r * sin(theta * DTOR) * sin(phi * DTOR);
	pos[2] = r * cos(theta * DTOR);
}


// XYZ座標を極座標(r, theta, phi)に変換する
// theta, phi [deg]
void xyz2polar(const double pos[], double *r, double *theta, double *phi)
{
	*r     = sqrt((pos[0] * pos[0]) + (pos[1] * pos[1]) + (pos[2] * pos[2]));
	*theta = acos(pos[2] / (*r))   / DTOR;
	*phi   = atan2(pos[1], pos[0]) / DTOR;
}


// warning出力
void warning(const char msg[], int nline)
{
	fprintf(stderr, "*** invalid <%s> data : line number = %d\n", msg, nline + 1);
}


// 文字列をトークン分解する
int tokenize(char *str, const char tokensep[], char *token[], size_t maxtoken)
{
	if ((str == NULL) || !maxtoken) return 0;

	char *thistoken = strtok(str, tokensep);

	int   count;
	for (count = 0; (count < maxtoken) && (thistoken != NULL); ) {
		token[count++] = thistoken;
		thistoken = strtok(NULL, tokensep);
	}

	token[count] = NULL;

	return count;
}
