﻿// OpenRTM
using System;
using System.Drawing;
using System.Windows.Forms;
using System.IO;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.Numerics;

namespace OpenRTM
{
	public partial class OpenRTM : Form
	{
		// 最大数
		private int MaxMaterial = 10;
		private int MaxAntenna = 10;
		private int MaxVertex = 100;
		private int MaxPillar = 10000;
		private int MaxPolygon = 10000;
		private int MaxTx = 1000;
		private int MaxRx0d = 1000;
		private int MaxRx1d = 1000;
		private int MaxRx2d = 10000;
		private int MaxMaterialAngle = 1000;

		// 変数
		private int NPillar = 0;
		private int NPolygon = 0;
		private int NTx = 0;
		private int NRx0d = 0;
		private int NRx1d = 0;
		private int NRx2d = 0;
		private Pillar_t[] Pillar;
		private Polygon_t[] Polygon;
		private Tx_t[] Tx;
		private Rx0d_t[] Rx0d;
		private Rx1d_t[] Rx1d;
		private Rx2d_t[] Rx2d;
		private string pathFile = string.Empty;  // full-path
		private string pathIni = string.Empty;  // ful-path

		// 環境変数(iniファイルの保存するもの)
		private int nThread = 8;
		private int NearPixel = 10;
		private int PointSize = 10;
		private bool bClose = false;
		private int iResolution = 2;   // =0...6
		private double dScale = 100;   // ピクセル/m
		private double Xleft = -2;     // m
		private double Ybottom = -2;   // m

		// GUI変数、物性値
		private bool bDrawR = true, bDrawT = false;
		private int NAngle = 0;
		private double[] Angle;
		private Complex[][] Ref, Trs;

		// GUI変数、アンテナ
		private int Ntheta = 0, Nphi = 0;
		private double[,] Rantenna;
		private int iPlotAntenna = -1;
		private double[,] Xantenna, Yantenna;
		private double[] Xaxis = new double[4];
		private double[] Yaxis = new double[4];

		// GUI変数、幾何形状
		private double[] dResolution = { 10, 5, 1, 0.5, 0.1, 0.05, 0.01 };
		private double[] PolygonX, PolygonY;
		private bool bDrag = false;
		private int nClicked = 0;
		private double Xold = 0, Yold = 0;
		private int iTarget = 0;     // =-1/0/1/2/3/4/5
		private int iDataid = -1;    // =-1,0,1,...
		private double Theta = 60;
		private double Phi = 30;

		// コントロール配列
		// (物性値)
		private CheckBox[] chkMaterialNo;
		private ComboBox[] cboMaterialType;
		private TextBox[] txtEpsr;
		private TextBox[] txtSigma;
		private TextBox[] txtThick;
		private TextBox[] txtMaterialFile;
		private TextBox[] txtMaterialComment;
		private Button[] btnMaterialFile;
		private Button[] btnMaterialPlot;
		// (アンテナ)
		private CheckBox[] chkAntennaNo;
		private ComboBox[] cboAntennaType;
		private ComboBox[] cboPol;
		private TextBox[] txtTheta;
		private TextBox[] txtPhi;
		private TextBox[] txtBw;
		private TextBox[] txtBwTheta;
		private TextBox[] txtBwPhi;
		private TextBox[] txtAntennaFile;
		private TextBox[] txtAntennaComment;
		private Button[] btnAntennaFile;
		private Button[] btnAntennaPlot;

		[STAThread]
		static void Main(string[] args)
		{
			Application.EnableVisualStyles();
			Application.SetCompatibleTextRenderingDefault(false);
			Application.Run(new OpenRTM(args));
		}

		public OpenRTM(string[] args)
		{
			InitializeComponent();

			pathIni = Application.StartupPath + Path.DirectorySeparatorChar + "OpenRTM.ini";

			Environment.CurrentDirectory = Application.StartupPath;

			// 環境設定ファイル入力
			readIni();

			// 形状変数
			Pillar = new Pillar_t[MaxPillar];
			Polygon = new Polygon_t[MaxPolygon];
			Tx = new Tx_t[MaxTx];
			Rx0d = new Rx0d_t[MaxRx0d];
			Rx1d = new Rx1d_t[MaxRx1d];
			Rx2d = new Rx2d_t[MaxRx2d];

			// 作業変数
			PolygonX = new double[MaxVertex];
			PolygonY = new double[MaxVertex];

			// コントロール配列作成
			chkMaterialNo = new CheckBox[MaxMaterial];
			cboMaterialType = new ComboBox[MaxMaterial];
			txtEpsr = new TextBox[MaxMaterial];
			txtSigma = new TextBox[MaxMaterial];
			txtThick = new TextBox[MaxMaterial];
			btnMaterialFile = new Button[MaxMaterial];
			txtMaterialFile = new TextBox[MaxMaterial];
			txtMaterialComment = new TextBox[MaxMaterial];
			btnMaterialPlot = new Button[MaxMaterial];

			chkAntennaNo = new CheckBox[MaxAntenna];
			cboAntennaType = new ComboBox[MaxAntenna];
			cboPol = new ComboBox[MaxAntenna];
			txtTheta = new TextBox[MaxAntenna];
			txtPhi = new TextBox[MaxAntenna];
			txtBw = new TextBox[MaxAntenna];
			txtBwTheta = new TextBox[MaxAntenna];
			txtBwPhi = new TextBox[MaxAntenna];
			txtAntennaFile = new TextBox[MaxAntenna];
			txtAntennaComment = new TextBox[MaxAntenna];
			btnAntennaFile = new Button[MaxAntenna];
			btnAntennaPlot = new Button[MaxAntenna];

			// デザイン
			setDesign();

			// 初期化
			initialize();

			// ファイル引数あり
			if (args.Length > 0)
			{
				// ファイル名
				pathFile = args[0];

				// ファイルを読み込む
				readFile(pathFile);

				// タイトルバー表示
				showTitleBar();

				// 上書き保存有効
				mnuSave.Enabled = true;

				// データ数表示
				showNumberOfData();

				// スケール設定
				initializeScale();

				// 再描画
				picMap.Invalidate();
			}
		}

		// [終了]
		private void mnuExit_Click(object sender, EventArgs e)
		{
			this.Close();
		}

		private void OpenRTM_FormClosing(object sender, FormClosingEventArgs e)
		{
			// 確認
			string msg = Application.ProductName + "を終了します。";
			string title = "終了";
			if (MessageBox.Show(msg, title, MessageBoxButtons.OKCancel, MessageBoxIcon.Exclamation) == DialogResult.Cancel) {
				e.Cancel = true;
			}
			if (this.WindowState == FormWindowState.Minimized) {
				this.WindowState = FormWindowState.Normal;  // 最小化復元
			}

			// 環境設定ファイル出力
			writeIni();
		}

		// デザイン
		private void setDesign()
		{
			const int sp = 5;
			int dh = Font.Height + 10;
			int h;

			// 物性値
			h = 10;
			for (int i = 0; i < chkMaterialNo.Length; i++) {
				chkMaterialNo[i] = new CheckBox();
				cboMaterialType[i] = new ComboBox();
				txtEpsr[i] = new TextBox();
				txtSigma[i] = new TextBox();
				txtThick[i] = new TextBox();
				btnMaterialFile[i] = new Button();
				txtMaterialFile[i] = new TextBox();
				txtMaterialComment[i] = new TextBox();
				btnMaterialPlot[i] = new Button();

				pnlMaterialData.Controls.Add(chkMaterialNo[i]);
				pnlMaterialData.Controls.Add(cboMaterialType[i]);
				pnlMaterialData.Controls.Add(txtEpsr[i]);
				pnlMaterialData.Controls.Add(txtSigma[i]);
				pnlMaterialData.Controls.Add(txtThick[i]);
				pnlMaterialData.Controls.Add(btnMaterialFile[i]);
				pnlMaterialData.Controls.Add(txtMaterialFile[i]);
				pnlMaterialData.Controls.Add(txtMaterialComment[i]);
				pnlMaterialData.Controls.Add(btnMaterialPlot[i]);

				chkMaterialNo[i].Top = h + 2;
				cboMaterialType[i].Top =
				txtEpsr[i].Top =
				txtSigma[i].Top =
				txtThick[i].Top =
				btnMaterialFile[i].Top =
				txtMaterialFile[i].Top =
				txtMaterialComment[i].Top =
				btnMaterialPlot[i].Top = h;
				h += dh;

				chkMaterialNo[i].Width = 50;
				cboMaterialType[i].Width = 205;
				txtEpsr[i].Width = 100;
				txtSigma[i].Width = 100;
				txtThick[i].Width = 100;
				btnMaterialFile[i].Width = 40;
				txtMaterialFile[i].Width = 140;
				txtMaterialComment[i].Width = 120;
				btnMaterialPlot[i].Width = 50;

				chkMaterialNo[i].Left = 15;
				cboMaterialType[i].Left = chkMaterialNo[i].Left + chkMaterialNo[i].Width + sp;
				txtEpsr[i].Left = cboMaterialType[i].Left + cboMaterialType[i].Width + sp;
				txtSigma[i].Left = txtEpsr[i].Left + txtEpsr[i].Width + sp;
				txtThick[i].Left = txtSigma[i].Left + txtSigma[i].Width + sp;
				btnMaterialFile[i].Left = txtThick[i].Left + txtThick[i].Width + sp;
				txtMaterialFile[i].Left = btnMaterialFile[i].Left + btnMaterialFile[i].Width + sp;
				txtMaterialComment[i].Left = txtMaterialFile[i].Left + txtMaterialFile[i].Width + sp;
				btnMaterialPlot[i].Left = txtMaterialComment[i].Left + txtMaterialComment[i].Width + sp;

				btnMaterialFile[i].Height = btnMaterialPlot[i].Height = cboMaterialType[i].Height;

				cboMaterialType[i].DropDownStyle = ComboBoxStyle.DropDownList;
				txtEpsr[i].TextAlign = txtSigma[i].TextAlign = txtThick[i].TextAlign = HorizontalAlignment.Right;

				chkMaterialNo[i].Text = (i + 2).ToString();
				btnMaterialFile[i].Text = "->";
				btnMaterialPlot[i].Text = "plot";

				cboMaterialType[i].Items.Clear();
				cboMaterialType[i].Items.Add("数値入力、厚さなし");
				cboMaterialType[i].Items.Add("数値入力、厚さあり");
				cboMaterialType[i].Items.Add("ファイル");

				// イベント
				chkMaterialNo[i].CheckedChanged += new System.EventHandler(this.chkMaterialNo_CheckedChanged);
				cboMaterialType[i].SelectedIndexChanged += new System.EventHandler(this.cboMaterialType_SelectedIndexChanged);
				btnMaterialFile[i].Click += new System.EventHandler(this.btnMaterialFile_Click);
				btnMaterialPlot[i].Click += new System.EventHandler(this.btnMaterialPlot_Click);
			}

			// アンテナ
			h = 10;
			for (int i = 0; i < chkAntennaNo.Length; i++) {
				chkAntennaNo[i] = new CheckBox();
				cboAntennaType[i] = new ComboBox();
				cboPol[i] = new ComboBox();
				txtTheta[i] = new TextBox();
				txtPhi[i] = new TextBox();
				txtBw[i] = new TextBox();
				txtBwTheta[i] = new TextBox();
				txtBwPhi[i] = new TextBox();
				btnAntennaFile[i] = new Button();
				txtAntennaFile[i] = new TextBox();
				txtAntennaComment[i] = new TextBox();
				btnAntennaPlot[i] = new Button();

				pnlAntennaData.Controls.Add(chkAntennaNo[i]);
				pnlAntennaData.Controls.Add(cboAntennaType[i]);
				pnlAntennaData.Controls.Add(cboPol[i]);
				pnlAntennaData.Controls.Add(txtTheta[i]);
				pnlAntennaData.Controls.Add(txtPhi[i]);
				pnlAntennaData.Controls.Add(txtBw[i]);
				pnlAntennaData.Controls.Add(txtBwTheta[i]);
				pnlAntennaData.Controls.Add(txtBwPhi[i]);
				pnlAntennaData.Controls.Add(btnAntennaFile[i]);
				pnlAntennaData.Controls.Add(txtAntennaFile[i]);
				pnlAntennaData.Controls.Add(txtAntennaComment[i]);
				pnlAntennaData.Controls.Add(btnAntennaPlot[i]);

				chkAntennaNo[i].Top = h + 2;
				cboAntennaType[i].Top =
				cboPol[i].Top =
				txtTheta[i].Top =
				txtPhi[i].Top =
				txtBw[i].Top =
				txtBwTheta[i].Top =
				txtBwPhi[i].Top =
				btnAntennaFile[i].Top =
				txtAntennaFile[i].Top =
				txtAntennaComment[i].Top =
				btnAntennaPlot[i].Top = h;
				h += dh;

				chkAntennaNo[i].Width = 50;
				cboAntennaType[i].Width = 110;
				cboPol[i].Width = 110;
				txtTheta[i].Width =
				txtPhi[i].Width =
				txtBw[i].Width =
				txtBwTheta[i].Width =
				txtBwPhi[i].Width = 70;
				btnAntennaFile[i].Width = 40;
				txtAntennaFile[i].Width = 120;
				txtAntennaComment[i].Width = 120;
				btnAntennaPlot[i].Width = 50;

				chkAntennaNo[i].Left = 15;
				cboAntennaType[i].Left = chkAntennaNo[i].Left + chkAntennaNo[i].Width + sp;
				cboPol[i].Left = cboAntennaType[i].Left + cboAntennaType[i].Width + sp;
				txtTheta[i].Left = cboPol[i].Left + cboPol[i].Width + sp;
				txtPhi[i].Left = txtTheta[i].Left + txtTheta[i].Width + sp;
				txtBw[i].Left = txtPhi[i].Left + txtPhi[i].Width + sp;
				txtBwTheta[i].Left = txtBw[i].Left + txtBw[i].Width + sp;
				txtBwPhi[i].Left = txtBwTheta[i].Left + txtBwTheta[i].Width + sp;
				btnAntennaFile[i].Left = txtBwPhi[i].Left + txtBwPhi[i].Width + sp;
				txtAntennaFile[i].Left = btnAntennaFile[i].Left + btnAntennaFile[i].Width + sp;
				txtAntennaComment[i].Left = txtAntennaFile[i].Left + txtAntennaFile[i].Width + sp;
				btnAntennaPlot[i].Left = txtAntennaComment[i].Left + txtAntennaComment[i].Width + sp;

				btnAntennaFile[i].Height = btnAntennaPlot[i].Height = cboAntennaType[i].Height;

				cboAntennaType[i].DropDownStyle = ComboBoxStyle.DropDownList;
				cboPol[i].DropDownStyle = ComboBoxStyle.DropDownList;
				txtTheta[i].TextAlign = txtPhi[i].TextAlign = txtBw[i].TextAlign = txtBwTheta[i].TextAlign = txtBwPhi[i].TextAlign = HorizontalAlignment.Right;

				chkAntennaNo[i].Text = (i + 1).ToString();
				btnAntennaFile[i].Text = "->";
				btnAntennaPlot[i].Text = "plot";

				cboAntennaType[i].Items.Clear();
				cboAntennaType[i].Items.Add("無指向性");
				cboAntennaType[i].Items.Add("ダイポール");
				cboAntennaType[i].Items.Add("ビーム");
				cboAntennaType[i].Items.Add("ファイル");

				cboPol[i].Items.Clear();
				cboPol[i].Items.Add("垂直偏波");
				cboPol[i].Items.Add("水平偏波");
				cboPol[i].Items.Add("右旋円偏波");
				cboPol[i].Items.Add("左旋円偏波");

				// イベント
				chkAntennaNo[i].CheckedChanged += new System.EventHandler(this.chkAntennaNo_CheckedChanged);
				cboAntennaType[i].SelectedIndexChanged += new System.EventHandler(this.cboAntennaType_SelectedIndexChanged);
				btnAntennaFile[i].Click += new System.EventHandler(this.btnAntennaFile_Click);
				btnAntennaPlot[i].Click += new System.EventHandler(this.btnAntennaPlot_Click);
			}

			// 最大数
			nudTarget0.Maximum = MaxPillar;
			nudTarget1.Maximum = MaxPolygon;
			nudTarget2.Maximum = MaxTx;
			nudTarget3.Maximum = MaxRx0d;
			nudTarget4.Maximum = MaxRx1d;
			nudTarget5.Maximum = MaxRx2d;

			// ToolTip
			toolTip.SetToolTip(btnGeom3d, "形状データを3D図形表示します");
			toolTip.SetToolTip(btnRay3d, "レイを3D図形表示します");

			toolTip.SetToolTip(btnSolver, "計算を実行します");
			toolTip.SetToolTip(btnPost, "計算結果をev.ev2,ev.ev3に出力します");
			toolTip.SetToolTip(btnEv2d, "ev.ev2を2D図形表示します");
			toolTip.SetToolTip(btnEv3d, "ev.ev3を3D図形表示します");

			toolTip.SetToolTip(rbnMode0, "形状を入力します");
			toolTip.SetToolTip(rbnMode1, "形状を編集します");
			toolTip.SetToolTip(btnEdit, "指定した対象の指定したデータ番号を編集します");
			toolTip.SetToolTip(btnDelete, "指定した対象の指定したデータ番号を削除します");
			toolTip.SetToolTip(btnEditAll, "指定した対象のすべてのデータを編集します");
			toolTip.SetToolTip(btnDeleteAll, "指定した対象のすべてのデータを削除します");
			toolTip.SetToolTip(btnZoomIn, "拡大");
			toolTip.SetToolTip(btnZoomOut, "縮小");
			toolTip.SetToolTip(btnZoomInitialize, "大きさを初期化します");
		}

		// [新規作成]
		private void mnuNew_Click(object sender, EventArgs e)
		{
			string msg = "新規作成します。現在のデータは消去されます。";
			string caption = "新規作成";
			if (MessageBox.Show(msg, caption, MessageBoxButtons.OKCancel, MessageBoxIcon.Exclamation) == DialogResult.OK) {
				initialize();
			}
		}

		// [開く]
		private void mnuOpen_Click(object sender, EventArgs e)
		{
			openFileDialog.Filter = "OpenRTMファイル (*.ort)|*.ort|EEM-RTMファイル (*.rtm)|*.rtm|すべてのファイル (*.*)|*.*";
			openFileDialog.FileName = String.Empty;

			if (openFileDialog.ShowDialog() == DialogResult.OK) {
				// 初期化
				initialize();

				// ファイル名取得
				pathFile = openFileDialog.FileName;

				// ファイルを読み込む
				if (Path.GetExtension(pathFile).ToLower() == ".ort")
				{
					readFile(pathFile);
				}
				else if (Path.GetExtension(pathFile).ToLower() == ".rtm")
				{
					readFile_EEM(pathFile);
				}

				// タイトルバー表示
				showTitleBar();

				// 上書き保存有効
				mnuSave.Enabled = true;

				// データ数表示
				showNumberOfData();

				// スケール設定
				initializeScale();

				// 再描画
				picMap.Invalidate();
			}
		}

		// [上書き保存]
		private void mnuSave_Click(object sender, EventArgs e)
		{
			writeFile(pathFile, 0, 0);
		}

		// [名前を付けて保存]
		private void mnuSaveAs_Click(object sender, EventArgs e)
		{
			saveFileDialog.Filter = "OpenRTM files (*.ort)|*.ort|all files (*.*)|*.*";
			saveFileDialog.FileName = Path.GetFileName(pathFile);

			// ダイアログ表示
			if (saveFileDialog.ShowDialog() == DialogResult.OK) {
				// ファイル名取得
				pathFile = saveFileDialog.FileName;

				// ファイルに保存
				writeFile(pathFile, 0, 0);

				// タイトルバー表示
				showTitleBar();

				// 上書き保存有効
				mnuSave.Enabled = true;
			}
		}

		// ファイルを読み込む
		private void readFile(string path)
		{
			if (!File.Exists(path)) return;

			Regex regex = new Regex(@"\s+");        // 1個以上の空白文字

			this.Cursor = Cursors.WaitCursor;

			using (var sr = new StreamReader(path))
			{
				try
				{
					string line;
					string[] token;
					int version = 0;

					// ヘッダー
					line = sr.ReadLine();
					token = regex.Split(line);
					if ((token.Length > 2) && (token[0] == Application.ProductName))
					{
						int major, minor;
						Int32.TryParse(token[1], out major);
						Int32.TryParse(token[2], out minor);
						version = (10 * major) + minor;
					}
					else
					{
						throw new Exception(path + " is not an OpenRTM file.");
					}

					// 物性値数とアンテナ数の初期化
					int nmaterial = 0;
					int nantenna = 0;

					while ((line = sr.ReadLine()) != null)
					{
						// 特殊行処理
						if (line.StartsWith("end")) {
							break;          // 終了行
						}
						else if (line.StartsWith("#")) {
							continue;       // コメント行
						}

						// トークン分解
						token = regex.Split(line.Trim());

						// 書式チェック
						if ((token.Length < 3) || (token[1] != "=")) {
							continue;
						}

						// キーワードで場合分け
						switch (token[0]) {
							case "title":
								// タイトル
								txtTitle.Text = (line.IndexOf('=') >= 0) ? line.Substring(line.IndexOf('=') + 2) : String.Empty;        // title行
								break;
							case "material":
								// 物性値
								if (token.Length > 3)
								{
									if (nmaterial >= MaxMaterial)
									{
										throw new Exception("物性値の数が最大数を超えました。[オプション]メニューで[物性値最大数]を大きくしてOpenRTMを再起動してください。");
									}
									int n = nmaterial;
									chkMaterialNo[n].Checked = true;
									Int32.TryParse(token[2], out int type);
									cboMaterialType[n].SelectedIndex = type - 1;
									if ((type == 1) && (token.Length > 4))
									{
										// 数値入力、厚さなし
										txtEpsr[n].Text = token[3];
										txtSigma[n].Text = token[4];
										if (token.Length > 5)
										{
											txtMaterialComment[n].Text = token[5];
										}
									}
									else if ((type == 2) && (token.Length > 5))
									{
										// 数値入力、厚さあり
										txtEpsr[n].Text = token[3];
										txtSigma[n].Text = token[4];
										txtThick[n].Text = token[5];
										if (token.Length > 6)
										{
											txtMaterialComment[n].Text = token[6];
										}
									}
									else if ((type == 3) && (token.Length > 3))
									{
										// ファイル
										txtMaterialFile[n].Text = token[3];
										if (token.Length > 4)
										{
											txtMaterialComment[n].Text = token[4];
										}
									}
									else
									{
										break;
									}
									nmaterial++;
								}
								break;
							case "antenna":
								// アンテナ
								if (token.Length > 3)
								{
									if (nantenna >= MaxAntenna)
									{
										throw new Exception("アンテナの数が最大数を超えました。[オプション]メニューで[アンテナ最大数]を大きくしてOpenRTMを再起動してください。");
									}
									int n = nantenna;
									chkAntennaNo[n].Checked = true;
									Int32.TryParse(token[2], out int type);
									cboAntennaType[n].SelectedIndex = type - 1;
									if ((type == 1) && (token.Length > 3))
									{
										cboPol[n].SelectedIndex = Int32.Parse(token[3]) - 1;
										if (token.Length > 4)
										{
											txtAntennaComment[n].Text = token[4];
										}
									}
									else if ((type == 2) && (token.Length > 6))
									{
										cboPol[n].SelectedIndex = Int32.Parse(token[3]) - 1;
										txtTheta[n].Text = token[4];
										txtPhi[n].Text = token[5];
										txtBw[n].Text = token[6];
										if (token.Length > 7)
										{
											txtAntennaComment[n].Text = token[7];
										}
									}
									else if ((type == 3) && (token.Length > 7))
									{
										cboPol[n].SelectedIndex = Int32.Parse(token[3]) - 1;
										txtTheta[n].Text = token[4];
										txtPhi[n].Text = token[5];
										txtBwTheta[n].Text = token[6];
										txtBwPhi[n].Text = token[7];
										if (token.Length > 8)
										{
											txtAntennaComment[n].Text = token[8];
										}
									}
									else if ((type == 4) && (token.Length > 3))
									{
										txtAntennaFile[n].Text = token[3];
										if (token.Length > 4)
										{
											txtAntennaComment[n].Text = token[4];
										}
									}
									else
									{
										break;
									}
									nantenna++;
								}
								break;
							case "pillar":
								// 多角柱
								if (token.Length > 8) {
									if (NPillar >= MaxPillar)
									{
										throw new Exception("多角柱の数が最大数を超えました。[オプション]メニューで[多角柱最大数]を大きくしてOpenRTMを再起動してください。");
									}
									int nvertex = (token.Length - 5) / 2;
									Pillar[NPillar] = new Pillar_t(nvertex);
									Pillar[NPillar].nvertex = nvertex;
									Int32.TryParse(token[2], out Pillar[NPillar].material);
									Double.TryParse(token[3], out Pillar[NPillar].z[0]);
									Double.TryParse(token[4], out Pillar[NPillar].z[1]);
									for (int n = 0; n < nvertex; n++)
									{
										Double.TryParse(token[5 + (2 * n)], out Pillar[NPillar].x[n]);
										Double.TryParse(token[6 + (2 * n)], out Pillar[NPillar].y[n]);
									}
									if (nvertex > 2)
									{
										Pillar[NPillar].x[nvertex] = Pillar[NPillar].x[0];
										Pillar[NPillar].y[nvertex] = Pillar[NPillar].y[0];
									}
									Pillar[NPillar].defined = true;
									NPillar++;
								}
								break;
							case "polygon":
								// 多角形
								if (token.Length > 11) {
									if (NPolygon >= MaxPolygon)
									{
										throw new Exception("多角形の数が最大数を超えました。[オプション]メニューで[多角形最大数]を大きくしてOpenRTMを再起動してください。");
									}
									int nvertex = (token.Length - 3) / 3;  // =3/4...
									Polygon[NPolygon] = new Polygon_t(nvertex);
									Polygon[NPolygon].nvertex = nvertex;
									Int32.TryParse(token[2], out Polygon[NPolygon].material);
									for (int n = 0; n < nvertex; n++)
									{
										Double.TryParse(token[3 + (3 * n)], out Polygon[NPolygon].x[n]);
										Double.TryParse(token[4 + (3 * n)], out Polygon[NPolygon].y[n]);
										Double.TryParse(token[5 + (3 * n)], out Polygon[NPolygon].z[n]);
									}
									if (nvertex > 2)
									{
										Polygon[NPolygon].x[nvertex] = Polygon[NPolygon].x[0];
										Polygon[NPolygon].y[nvertex] = Polygon[NPolygon].y[0];
										Polygon[NPolygon].z[nvertex] = Polygon[NPolygon].z[0];
									}
									Polygon[NPolygon].defined = true;
									NPolygon++;
								}
								break;
							case "tx":
								// 送信点
								if (token.Length > 7)
								{
									if (NTx >= MaxTx)
									{
										throw new Exception("送信点の数が最大数を超えました。[オプション]メニューで[送信点最大数]を大きくしてOpenRTMを再起動してください。");
									}
									Tx[NTx] = new Tx_t();
									Tx[NTx].antenna = Int32.Parse(token[2]) - 1;
									Double.TryParse(token[3], out Tx[NTx].pos[0]);
									Double.TryParse(token[4], out Tx[NTx].pos[1]);
									Double.TryParse(token[5], out Tx[NTx].pos[2]);
									Double.TryParse(token[6], out Tx[NTx].power);
									Double.TryParse(token[7], out Tx[NTx].phase);
									Tx[NTx].defined = true;
									NTx++;
								}
								break;
							case "rx0d":
								// 観測点
								if (token.Length > 5)
								{
									if (NRx0d >= MaxRx0d)
									{
										throw new Exception("観測点の数が最大数を超えました。[オプション]メニューで[観測点最大数]を大きくしてOpenRTMを再起動してください。");
									}
									Rx0d[NRx0d] = new Rx0d_t();
									Rx0d[NRx0d].antenna = Int32.Parse(token[2]) - 1;
									Double.TryParse(token[3], out Rx0d[NRx0d].pos[0]);
									Double.TryParse(token[4], out Rx0d[NRx0d].pos[1]);
									Double.TryParse(token[5], out Rx0d[NRx0d].pos[2]);
									Rx0d[NRx0d].defined = true;
									NRx0d++;
								}
								break;
							case "rx1d":
								// 観測線
								if (token.Length > 9)
								{
									if (NRx1d >= MaxRx1d)
									{
										throw new Exception("観測線の数が最大数を超えました。[オプション]メニューで[観測線最大数]を大きくしてOpenRTMを再起動してください。");
									}
									Rx1d[NRx1d] = new Rx1d_t();
									Rx1d[NRx1d].antenna = Int32.Parse(token[2]) - 1;
									Double.TryParse(token[3], out Rx1d[NRx1d].pos[0, 0]);
									Double.TryParse(token[4], out Rx1d[NRx1d].pos[0, 1]);
									Double.TryParse(token[5], out Rx1d[NRx1d].pos[0, 2]);
									Double.TryParse(token[6], out Rx1d[NRx1d].pos[1, 0]);
									Double.TryParse(token[7], out Rx1d[NRx1d].pos[1, 1]);
									Double.TryParse(token[8], out Rx1d[NRx1d].pos[1, 2]);
									Int32.TryParse(token[9], out Rx1d[NRx1d].div);
									Rx1d[NRx1d].defined = true;
									NRx1d++;
								}
								break;
							case "rx2d":
								// 観測面
								if (token.Length > 16)
								{
									if (NRx2d >= MaxRx2d)
									{
										throw new Exception("観測面の数が最大数を超えました。[オプション]メニューで[観測面最大数]を大きくしてOpenRTMを再起動してください。");
									}
									Rx2d[NRx2d] = new Rx2d_t();
									Rx2d[NRx2d].antenna = Int32.Parse(token[2]) - 1;
									Double.TryParse(token[3], out Rx2d[NRx2d].pos[0, 0]);
									Double.TryParse(token[4], out Rx2d[NRx2d].pos[0, 1]);
									Double.TryParse(token[5], out Rx2d[NRx2d].pos[0, 2]);
									Double.TryParse(token[6], out Rx2d[NRx2d].pos[1, 0]);
									Double.TryParse(token[7], out Rx2d[NRx2d].pos[1, 1]);
									Double.TryParse(token[8], out Rx2d[NRx2d].pos[1, 2]);
									Double.TryParse(token[9], out Rx2d[NRx2d].pos[2, 0]);
									Double.TryParse(token[10], out Rx2d[NRx2d].pos[2, 1]);
									Double.TryParse(token[11], out Rx2d[NRx2d].pos[2, 2]);
									Double.TryParse(token[12], out Rx2d[NRx2d].pos[3, 0]);
									Double.TryParse(token[13], out Rx2d[NRx2d].pos[3, 1]);
									Double.TryParse(token[14], out Rx2d[NRx2d].pos[3, 2]);
									Int32.TryParse(token[15], out Rx2d[NRx2d].div[0]);
									Int32.TryParse(token[16], out Rx2d[NRx2d].div[1]);
									Rx2d[NRx2d].defined = true;
									NRx2d++;
								}
								break;
							// その他の計算パラメーター
							case "frequency":
								// 周波数
								Double.TryParse(token[2], out double frequency);
								txtFrequency.Text = (frequency * 1e-9).ToString();
								break;
							case "maxpath":
								// 最大伝搬経路数
		
								Int32.TryParse(token[2], out int maxpath);
								nudMaxPath.Value = maxpath;
								break;
							case "maxref":
								//　最大反射回数
								Int32.TryParse(token[2], out int maxref);
								nudLaunch0.Value = maxref;
								break;
							case "ndivlaunch":
								// ローンチング法、緯度分割数
								Int32.TryParse(token[2], out int ndivlaunch);
								nudLaunch1.Value = ndivlaunch;
								break;
							case "component":
								// 直接法、計算成分
								if (token.Length > 4)
								{
									chkComponent0.Checked = (token[2] == "1");
									chkComponent1.Checked = (token[3] == "1");
									chkComponent2.Checked = (token[4] == "1") || (token[4]== "2");
									chkComponent2a.Checked = (token[4] == "2");
									chkComponent3.Checked = (token[5] == "1");
								}
								break;
							case "ndivantenna":
								// アンテナパターン、緯度分割数
								Int32.TryParse(token[2], out int ndivantenna);
								nudNdivAntenna.Value = ndivantenna;
								break;
							case "log":
								// 数値出力
								if (token.Length > 3)
								{
									chkLog0.Checked = (token[2] == "1");
									chkLog1.Checked = (token[3] == "1");
								}
								break;
							// ポスト処理
							// Rx0d
							case "rx0d_path":
								chkRx0d_path.Checked = (token[2] == "1");
								break;
							case "rx0d_profile":
								chkRx0d_profile.Checked = (token[2] == "1");
								break;
							case "rx0d_profile_xscale":
								if (token.Length > 5)
								{
									chkRx0d_profile_xscale.Checked = (token[2] == "0");
									txtRx0d_profile_xscale_min.Text = token[3];
									txtRx0d_profile_xscale_max.Text = token[4];
									Int32.TryParse(token[5], out int div);
									nudRx0d_profile_xscale_div.Value = div;
								}
								break;
							case "rx0d_profile_yscale":
								if (token.Length > 5)
								{
									chkRx0d_profile_yscale.Checked = (token[2] == "0");
									txtRx0d_profile_yscale_min.Text = token[3];
									txtRx0d_profile_yscale_max.Text = token[4];
									Int32.TryParse(token[5], out int div);
									nudRx0d_profile_yscale_div.Value = div;
								}
								break;
							// Rx1d
							case "rx1d_path":
								chkRx1d_path.Checked = (token[2] == "1");
								break;
							case "rx1d_profile":
								chkRx1d_profile.Checked = (token[2] == "1");
								break;
							case "rx1d_power":
								chkRx1d_power.Checked = (token[2] == "1");
								break;
							case "rx1d_delay":
								chkRx1d_delay.Checked = (token[2] == "1");
								break;
							case "rx1d_power_scale":
								if (token.Length > 5)
								{
									chkRx1d_power_scale.Checked = (token[2] == "0");
									txtRx1d_power_scale_min.Text = token[3];
									txtRx1d_power_scale_max.Text = token[4];
									Int32.TryParse(token[5], out int div);
									nudRx1d_power_scale_div.Value = div;
								}
								break;
							case "rx1d_delay_scale":
								if (token.Length > 5)
								{
									chkRx1d_delay_scale.Checked = (token[2] == "0");
									txtRx1d_delay_scale_min.Text = token[3];
									txtRx1d_delay_scale_max.Text = token[4];
									Int32.TryParse(token[5], out int div);
									nudRx1d_delay_scale_div.Value = div;
								}
								break;
							// Rx2d
							case "rx2d_path":
								chkRx2d_path.Checked = (token[2] == "1");
								break;
							case "rx2d_power":
								if (token.Length > 3)
								{
									chkRx2d_power0.Checked = (token[2] == "1");
									chkRx2d_power1.Checked = (token[3] == "1");
								}
								break;
							case "rx2d_delay":
								if (token.Length > 3)
								{
									chkRx2d_delay0.Checked = (token[2] == "1");
									chkRx2d_delay1.Checked = (token[3] == "1");
								}
								break;
							case "rx2d_stat":
								if (token.Length > 5)
								{
									chkRx2d_stat0.Checked = (token[2] == "1");
									chkRx2d_stat1.Checked = (token[3] == "1");
									chkRx2d_stat2.Checked = (token[4] == "1");
									chkRx2d_stat3.Checked = (token[5] == "1");
								}
								break;
							case "rx2d_decay":
								if (token.Length > 3)
								{
									chkRx2d_decay0.Checked = (token[2] == "1");
									chkRx2d_decay1.Checked = (token[3] == "1");
								}
								break;
							case "rx2d_power_scale":
								if (token.Length > 5)
								{
									chkRx2d_power_scale.Checked = (token[2] == "0");
									txtRx2d_power_scale_min.Text = token[3];
									txtRx2d_power_scale_max.Text = token[4];
									Int32.TryParse(token[5], out int div);
									nudRx2d_power_scale_div.Value = div;
								}
								break;
							case "rx2d_delay_scale":
								if (token.Length > 5)
								{
									chkRx2d_delay_scale.Checked = (token[2] == "0");
									txtRx2d_delay_scale_min.Text = token[3];
									txtRx2d_delay_scale_max.Text = token[4];
									Int32.TryParse(token[5], out int div);
									nudRx2d_delay_scale_div.Value = div;
								}
								break;
							case "rx2d_decay_scale":
								if (token.Length > 4)
								{
									chkRx2d_decay_scale.Checked = (token[2] == "0");
									txtRx2d_decay_scale_min.Text = token[3];
									txtRx2d_decay_scale_max.Text = token[4];
								}
								break;
							case "rx2d_geometry":
								rbnRx2d_geometry0.Checked = (token[2] == "0");
								rbnRx2d_geometry1.Checked = (token[2] == "1");
								rbnRx2d_geometry2.Checked = (token[2] == "2");
								break;
							default:
								break;
						}
					}
				}
				catch (Exception ex) {
					MessageBox.Show(ex.Message + "\n" + ex.StackTrace , "readFile", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
				}
			}

			this.Cursor = Cursors.Default;
		}
		// ファイルを保存する
		private void writeFile(string path, int i1, int i2)
		{
			this.Cursor = Cursors.WaitCursor;

			using (var sw = new StreamWriter(path))   // UTF8 BOMなし
			//using (var sw = new StreamWriter(path, false, Encoding.Unicode))
			//using (var sw = new StreamWriter(path, false, Encoding.UTF8))  // BOMあり
			//using (var sw = new StreamWriter(path, false, Encoding.Default))
			{
				try
				{
					// header
					Version ver = new Version(Application.ProductVersion);
					sw.WriteLine("{0} {1} {2}", Application.ProductName, ver.Major, ver.Minor);

					// title
					sw.WriteLine("title = {0}", txtTitle.Text);

					// material
					for (int n = 0; n < chkMaterialNo.Length; n++)
					{
						if (chkMaterialNo[n].Checked)
						{
							string cm = txtMaterialComment[n].Text;
							if (cboMaterialType[n].SelectedIndex == 0)
							{
								sw.WriteLine("material = 1 {0} {1} {2}", txtEpsr[n].Text, txtSigma[n].Text, cm);
							}
							else if (cboMaterialType[n].SelectedIndex == 1)
							{
								sw.WriteLine("material = 2 {0} {1} {2} {3}", txtEpsr[n].Text, txtSigma[n].Text, txtThick[n].Text, cm);
							}
							else if (cboMaterialType[n].SelectedIndex == 2)
							{
								sw.WriteLine("material = 3 {0} {1}", txtMaterialFile[n].Text, cm);
							}
						}
					}

					// antenna
					for (int n = 0; n < chkAntennaNo.Length; n++)
					{
						if (chkAntennaNo[n].Checked)
						{
							int pol = cboPol[n].SelectedIndex + 1;  // =1/2/3/4
							string cm = txtAntennaComment[n].Text;
							if (cboAntennaType[n].SelectedIndex == 0)
							{
								sw.WriteLine("antenna = 1 {0} {1}", pol, cm);
							}
							else if (cboAntennaType[n].SelectedIndex == 1)
							{
								sw.WriteLine("antenna = 2 {0} {1} {2} {3} {4}", pol, txtTheta[n].Text, txtPhi[n].Text, txtBw[n].Text, cm);
							}
							else if (cboAntennaType[n].SelectedIndex == 2)
							{
								sw.WriteLine("antenna = 3 {0} {1} {2} {3} {4} {5}", pol, txtTheta[n].Text, txtPhi[n].Text, txtBwTheta[n].Text, txtBwPhi[n].Text, cm);
							}
							else if (cboAntennaType[n].SelectedIndex == 3)
							{
								sw.WriteLine("antenna = 4 {0} {1}", txtAntennaFile[n].Text, cm);
							}
						}
					}

					// pillar
					for (int n = 0; n < NPillar; n++)
					{
						Pillar_t p = Pillar[n];
						if (p.defined)
						{
							sw.Write("pillar = {0} {1} {2}", p.material, p.z[0], p.z[1]);
							for (int i = 0; i < p.nvertex; i++)
							{
								sw.Write(" {0} {1}", p.x[i], p.y[i]);
							}
							sw.WriteLine();
						}
					}

					// polygon
					for (int n = 0; n < NPolygon; n++)
					{
						Polygon_t p = Polygon[n];
						if (p.defined)
						{
							sw.Write("polygon = {0}", p.material);
							for (int i = 0; i < p.nvertex; i++)
							{
								sw.Write(" {0} {1} {2}", p.x[i], p.y[i], p.z[i]);
							}
							sw.WriteLine();
						}
					}

					// tx
					for (int n = 0; n < NTx; n++)
					{
						Tx_t p = Tx[n];
						if (p.defined)
						{
							sw.WriteLine("tx = {0} {1} {2} {3} {4} {5}", p.antenna + 1, p.pos[0], p.pos[1], p.pos[2], p.power, p.phase);
						}
					}

					// rx0d
					for (int n = 0; n < NRx0d; n++)
					{
						Rx0d_t p = Rx0d[n];
						if (p.defined)
						{
							sw.WriteLine("rx0d = {0} {1} {2} {3}", p.antenna + 1, p.pos[0], p.pos[1], p.pos[2]);
						}
					}

					// rx1d
					for (int n = 0; n < NRx1d; n++)
					{
						Rx1d_t p = Rx1d[n];
						if (p.defined)
						{
							sw.Write("rx1d = {0}", p.antenna + 1);
							for (int i = 0; i < 2; i++)
							{
								sw.Write(" {0} {1} {2}", p.pos[i, 0], p.pos[i, 1], p.pos[i, 2]);
							}
							sw.WriteLine(" {0}", p.div);
						}
					}

					// rx2d
					for (int n = 0; n < NRx2d; n++)
					{
						Rx2d_t p = Rx2d[n];
						if (p.defined)
						{
							sw.Write("rx2d = {0}", p.antenna + 1);
							for (int i = 0; i < 4; i++)
							{
								sw.Write(" {0} {1} {2}", p.pos[i, 0], p.pos[i, 1], p.pos[i, 2]);
							}
							sw.WriteLine(" {0} {1}", p.div[0], p.div[1]);
						}
					}

					// その他の計算パラメーター
					Double.TryParse(txtFrequency.Text, out double frequency);
					sw.WriteLine("frequency = {0:0.0#####e0#}", frequency * 1e9);
					sw.WriteLine("maxpath = {0}", nudMaxPath.Value);
					sw.WriteLine("maxref = {0}", nudLaunch0.Value);
					sw.WriteLine("ndivlaunch = {0}", nudLaunch1.Value);
					sw.WriteLine("component = {0} {1} {2} {3}", (chkComponent0.Checked ? 1 : 0), (chkComponent1.Checked ? 1 : 0), (chkComponent2.Checked ? (chkComponent2a.Checked ? 2 : 1) : 0), (chkComponent3.Checked ? 1 : 0));
					sw.WriteLine("ndivantenna = {0}", nudNdivAntenna.Value);
					sw.WriteLine("log = {0} {1}", (chkLog0.Checked ? 1 : 0), (chkLog1.Checked ? 1 : 0));
					sw.WriteLine("plot3d = {0} 0 {1}", i1, i2);  // i1, i2

					// ポスト処理
					// Rx0d
					sw.WriteLine("rx0d_path = {0}", (chkRx0d_path.Checked ? 1 : 0));
					sw.WriteLine("rx0d_profile = {0}", (chkRx0d_profile.Checked ? 1 : 0));
					if (!chkRx0d_profile_xscale.Checked)
					{
						sw.WriteLine("rx0d_profile_xscale = 1 {0} {1} {2}", txtRx0d_profile_xscale_min.Text, txtRx0d_profile_xscale_max.Text, nudRx0d_profile_xscale_div.Value);
					}
					if (!chkRx0d_profile_yscale.Checked)
					{
						sw.WriteLine("rx0d_profile_yscale = 1 {0} {1} {2}", txtRx0d_profile_yscale_min.Text, txtRx0d_profile_yscale_max.Text, nudRx0d_profile_yscale_div.Value);
					}
					// Rx1d
					sw.WriteLine("rx1d_path = {0}", (chkRx1d_path.Checked ? 1 : 0));
					sw.WriteLine("rx1d_profile = {0}", (chkRx1d_profile.Checked ? 1 : 0));
					sw.WriteLine("rx1d_power = {0}", (chkRx1d_power.Checked ? 1 : 0));
					sw.WriteLine("rx1d_delay = {0}", (chkRx1d_delay.Checked ? 1 : 0));
					if (!chkRx1d_power_scale.Checked)
					{
						sw.WriteLine("rx1d_power_scale = 1 {0} {1} {2}", txtRx1d_power_scale_min.Text, txtRx1d_power_scale_max.Text, nudRx1d_power_scale_div.Value);
					}
					if (!chkRx1d_delay_scale.Checked)
					{
						sw.WriteLine("rx1d_delay_scale = 1 {0} {1} {2}", txtRx1d_delay_scale_min.Text, txtRx1d_delay_scale_max.Text, nudRx1d_delay_scale_div.Value);
					}
					// Rx2d
					sw.WriteLine("rx2d_path = {0}", (chkRx2d_path.Checked ? 1 : 0));
					sw.WriteLine("rx2d_power = {0} {1}", (chkRx2d_power0.Checked ? 1 : 0), (chkRx2d_power1.Checked ? 1 : 0));
					sw.WriteLine("rx2d_delay = {0} {1}", (chkRx2d_delay0.Checked ? 1 : 0), (chkRx2d_delay1.Checked ? 1 : 0));
					sw.WriteLine("rx2d_stat = {0} {1} {2} {3}", (chkRx2d_stat0.Checked ? 1 : 0), (chkRx2d_stat1.Checked ? 1 : 0), (chkRx2d_stat2.Checked ? 1 : 0), (chkRx2d_stat3.Checked ? 1 : 0));
					sw.WriteLine("rx2d_decay = {0} {1}", (chkRx2d_decay0.Checked ? 1 : 0), (chkRx2d_decay1.Checked ? 1 : 0));
					if (!chkRx2d_power_scale.Checked)
					{
						sw.WriteLine("rx2d_power_scale = 1 {0} {1} {2}", txtRx2d_power_scale_min.Text, txtRx2d_power_scale_max.Text, nudRx2d_power_scale_div.Value);
					}
					if (!chkRx2d_delay_scale.Checked)
					{
						sw.WriteLine("rx2d_delay_scale = 1 {0} {1} {2}", txtRx2d_delay_scale_min.Text, txtRx2d_delay_scale_max.Text, nudRx2d_delay_scale_div.Value);
					}
					if (!chkRx2d_decay_scale.Checked)
					{
						sw.WriteLine("rx2d_decay_scale = 1 {0} {1}", txtRx2d_decay_scale_min.Text, txtRx2d_decay_scale_max.Text);
					}
					sw.WriteLine("rx2d_geometry = {0}", (rbnRx2d_geometry0.Checked ? 0 : rbnRx2d_geometry1.Checked ? 1 : 2));

					// end
					sw.WriteLine("end");
				}
				catch (Exception ex)
				{
					MessageBox.Show(ex.Message, "writeFile", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
				}
			}

			this.Cursor = Cursors.Default;
		}

		// 初期化
		private void initialize()
		{
			try
			{
				// ファイル名消去
				pathFile = String.Empty;

				// 上書き保存無効
				mnuSave.Enabled = false;

				// タイトルバー
				showTitleBar();

				// 全般
				txtTitle.Text = string.Empty;
				chkComponent0.Checked = chkComponent1.Checked = chkComponent2.Checked = chkComponent3.Checked = true;
				chkComponent2a.Checked = false;
				nudLaunch0.Value = 3;
				nudLaunch1.Value = 180;
				txtFrequency.Text = "1";
				nudMaxPath.Value = 30;
				chkLog0.Checked = chkLog1.Checked = false;

				// 物性値
				initializeMaterial();
				bDrawR = bDrawT = false;
				picR.Invalidate();
				picT.Invalidate();

				// アンテナ
				initializeAntenna();
				Ntheta = Nphi = 0;
				iPlotAntenna = -1;
				picAntenna3d.Invalidate();

				// 物体形状
				rbnMode1.Checked = true;
				rbnTarget0.Checked = true;
				NPillar = NPolygon = NTx = NRx0d = NRx1d = NRx2d = 0;
				nudTarget0.Value = nudTarget1.Value = nudTarget2.Value = nudTarget3.Value = nudTarget4.Value = nudTarget5.Value = 1;
				lblTarget0.Text = lblTarget1.Text = lblTarget2.Text = lblTarget3.Text = lblTarget4.Text = lblTarget5.Text = "0/0";
				picMap.Invalidate();

				// ポスト処理制御
				chkRx0d_path.Checked = chkRx0d_profile.Checked =
				chkRx1d_path.Checked = chkRx1d_profile.Checked = chkRx1d_power.Checked = chkRx1d_delay.Checked =
				chkRx2d_path.Checked = chkRx2d_power0.Checked = chkRx2d_power1.Checked = chkRx2d_delay0.Checked = chkRx2d_delay1.Checked =
				chkRx2d_stat0.Checked = chkRx2d_stat1.Checked = chkRx2d_stat2.Checked = chkRx2d_stat3.Checked =
				chkRx2d_decay0.Checked = chkRx2d_decay1.Checked = false;
				chkRx0d_profile_xscale.Checked = chkRx0d_profile_yscale.Checked =
				chkRx1d_power_scale.Checked = chkRx1d_delay_scale.Checked =
				chkRx2d_power_scale.Checked = chkRx2d_delay_scale.Checked = chkRx2d_decay_scale.Checked = true;
				txtRx0d_profile_xscale_min.Text = txtRx1d_delay_scale_min.Text = txtRx2d_delay_scale_min.Text = "0";
				txtRx0d_profile_xscale_max.Text = txtRx1d_delay_scale_max.Text = txtRx2d_delay_scale_max.Text = "1000";
				txtRx0d_profile_yscale_min.Text = txtRx1d_power_scale_min.Text = txtRx2d_power_scale_min.Text = "-130";
				txtRx0d_profile_yscale_max.Text = txtRx1d_power_scale_max.Text = txtRx2d_power_scale_max.Text = "-30";
				txtRx2d_decay_scale_min.Text = "1";
				txtRx2d_decay_scale_max.Text = "1000";
				nudRx0d_profile_xscale_div.Value = nudRx1d_delay_scale_div.Value = 10;
				nudRx2d_delay_scale_div.Value = 50;
				nudRx0d_profile_yscale_div.Value = nudRx1d_power_scale_div.Value = 10;
				nudRx2d_power_scale_div.Value = 50;

				// データ数表示
				showNumberOfData();
			}
			catch { }
		}

		// 物性値初期化
		private void initializeMaterial()
		{
			for (int n = 0; n < chkMaterialNo.Length; n++)
			{
				chkMaterialNo[n].Checked = false;
				txtEpsr[n].Text = "1";
				txtSigma[n].Text = "0";
				txtThick[n].Text = "0";
				txtMaterialFile[n].Text = string.Empty;
				txtMaterialComment[n].Text = string.Empty;
				cboMaterialType[n].Enabled = txtMaterialComment[n].Enabled = txtEpsr[n].Enabled = txtSigma[n].Enabled = txtThick[n].Enabled =
				btnMaterialFile[n].Enabled = txtMaterialFile[n].Enabled = btnMaterialPlot[n].Enabled = false;
			}
		}

		// アンテナ初期化
		private void initializeAntenna()
		{
			for (int n = 0; n < chkAntennaNo.Length; n++)
			{
				chkAntennaNo[n].Checked = false;
				txtTheta[n].Text = "0";
				txtPhi[n].Text = "0";
				txtBw[n].Text = "90";
				txtBwTheta[n].Text = "90";
				txtBwPhi[n].Text = "90";
				txtAntennaFile[n].Text = string.Empty;
				txtAntennaComment[n].Text = string.Empty;
				cboAntennaType[n].Enabled = cboPol[n].Enabled =
				txtTheta[n].Enabled = txtPhi[n].Enabled = txtBw[n].Enabled = txtBwTheta[n].Enabled = txtBwPhi[n].Enabled =
				btnAntennaFile[n].Enabled = txtAntennaFile[n].Enabled = txtAntennaComment[n].Enabled = btnAntennaPlot[n].Enabled = false;
			}
		}

		// タイトルバー表示
		private void showTitleBar()
		{
			this.Text = Application.ProductName + " " + Path.GetFileName(pathFile);
			//this.Text = Application.ProductName + " " + pathFile;
		}

		// 環境設定ファイル出力
		private void writeIni()
		{
			using (var sw = new StreamWriter(pathIni))
			{
				try
				{
					Version ver = new Version(Application.ProductVersion);
					sw.WriteLine("Program={0}", Application.ProductName);
					sw.WriteLine("Version={0},{1}", ver.Major, ver.Minor);
					sw.WriteLine("Window={0},{1},{2},{3}", Math.Max(this.Left, 0), Math.Max(this.Top, 0), this.ClientSize.Width, this.ClientSize.Height);
					sw.WriteLine("MaxMaterial={0}", MaxMaterial);
					sw.WriteLine("MaxAntenna={0}", MaxAntenna);
					sw.WriteLine("MaxPillar={0}", MaxPillar);
					sw.WriteLine("MaxPlane={0}", MaxPolygon);
					sw.WriteLine("MaxTx={0}", MaxTx);
					sw.WriteLine("MaxRx0d={0}", MaxRx0d);
					sw.WriteLine("MaxRx1d={0}", MaxRx1d);
					sw.WriteLine("MaxRx2d={0}", MaxRx2d);
					sw.WriteLine("MaxMaterialAngle={0}", MaxMaterialAngle);
					sw.WriteLine("MaxVertex={0}", MaxVertex);
					sw.WriteLine("Resolution={0}", iResolution);
					sw.WriteLine("ScaleFactor={0:0.000}", dScale);
					sw.WriteLine("Xleft={0:0.000}", Xleft);
					sw.WriteLine("Ybottom={0:0.000}", Ybottom);
					sw.WriteLine("NearPixel={0}", NearPixel);
					sw.WriteLine("PointSize={0}", PointSize);
					sw.WriteLine("AntennaView={0},{1}", nudAntennaView1.Value, nudAntennaView2.Value);
					sw.WriteLine("DataNumber={0}", (chkDataNumber.Checked ? 1 : 0));
					sw.WriteLine("Prompt={0}", (bClose ? 0 : 1));
					sw.WriteLine("Thread={0}", nThread);
				}
				catch (Exception ex)
				{
					MessageBox.Show("writeIni : " + pathIni + "\n" + ex.Message);
				}
			}
		}

		// 環境設定ファイル入力
		private void readIni()
		{
			if (!File.Exists(pathIni)) return;

			using (var sr = new StreamReader(pathIni))
			{
				try
				{
					char[] delim = { '=', ',' };
					string[] token;
					string line;
					int version = 0;
					while ((line = sr.ReadLine()) != null)
					{
						token = line.Trim().Split(delim);
						switch (token[0])
						{
							case "Program":
								if ((token.Length > 1) && (token[1] != Application.ProductName)) return;
								break;
							case "Version":
								if (token.Length > 2) {
									int major, minor;
									Int32.TryParse(token[1], out major);
									Int32.TryParse(token[2], out minor);
									version = (10 * major) + minor;
								}
								break;
							case "Window":
								if (token.Length > 4) {
									this.Left = Int32.Parse(token[1]);
									this.Top = Int32.Parse(token[2]);
									//this.Width = Int32.Parse(token[3]);
									//this.Height = Int32.Parse(token[4]);
									this.ClientSize = new Size(Int32.Parse(token[3]), Int32.Parse(token[4]));
								}
								break;
							case "MaxMaterial":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxMaterial);
								}
								break;
							case "MaxAntenna":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxAntenna);
								}
								break;
							case "MaxPillar":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxPillar);
								}
								break;
							case "MaxPlane":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxPolygon);
								}
								break;
							case "MaxTx":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxTx);
								}
								break;
							case "MaxRx0d":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxRx0d);
								}
								break;
							case "MaxRx1d":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxRx1d);
								}
								break;
							case "MaxRx2d":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxRx2d);
								}
								break;
							case "MaxMaterialAngle":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxMaterialAngle);
								}
								break;
							case "MaxVertex":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out MaxVertex);
								}
								break;
							case "Resolution":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out iResolution);
									tssResolution.Text = dResolution[iResolution] + "m/div";
								}
								break;
							case "ScaleFactor":
								if (token.Length > 1) {
									Double.TryParse(token[1], out dScale);
								}
								break;
							case "Xleft":
								if (token.Length > 1) {
									Double.TryParse(token[1], out Xleft);
								}
								break;
							case "Ybottom":
								if (token.Length > 1) {
									Double.TryParse(token[1], out Ybottom);
								}
								break;
							case "NearPixel":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out NearPixel);
								}
								break;
							case "PointSize":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out PointSize);
								}
								break;
							case "AntennaView":
								if (token.Length > 2) {
									int i1, i2;
									Int32.TryParse(token[1], out i1);
									Int32.TryParse(token[2], out i2);
									nudAntennaView1.Value = i1;
									nudAntennaView2.Value = i2;
								}
								break;
							case "DataNumber":
								if (token.Length > 1)
								{
									chkDataNumber.Checked = (token[1] == "1");
								}
								break;
							case "Prompt":
								if (token.Length > 1) {
									bClose = (token[1] == "0");
								}
								break;
							case "Thread":
								if (token.Length > 1) {
									Int32.TryParse(token[1], out nThread);
								}
								break;
							default:
								break;
						}
					}
				}
				catch (Exception ex)
				{
					MessageBox.Show("readIni : " + pathIni + "\n" + ex.Message);
				}
			}
		}

		// [オプション]メニュー
		private void mnuOption_Click(object sender, EventArgs e)
		{
			try
			{
				frmOption f = new frmOption();

				// ウィンドウに現在の値を代入する
				f.nudMaxMaterial.Value = Math.Max(f.nudMaxMaterial.Minimum, Math.Min(f.nudMaxMaterial.Maximum, MaxMaterial));
				f.nudMaxAntenna.Value = Math.Max(f.nudMaxAntenna.Minimum, Math.Min(f.nudMaxAntenna.Maximum, MaxAntenna));
				f.nudMaxPillar.Value = Math.Max(f.nudMaxPillar.Minimum, Math.Min(f.nudMaxPillar.Maximum, MaxPillar));
				f.nudMaxPolygon.Value = Math.Max(f.nudMaxPolygon.Minimum, Math.Min(f.nudMaxPolygon.Maximum, MaxPolygon));
				f.nudMaxTx.Value = Math.Max(f.nudMaxTx.Minimum, Math.Min(f.nudMaxTx.Maximum, MaxTx));
				f.nudMaxRx0d.Value = Math.Max(f.nudMaxRx0d.Minimum, Math.Min(f.nudMaxRx0d.Maximum, MaxRx0d));
				f.nudMaxRx1d.Value = Math.Max(f.nudMaxRx1d.Minimum, Math.Min(f.nudMaxRx1d.Maximum, MaxRx1d));
				f.nudMaxRx2d.Value = Math.Max(f.nudMaxRx2d.Minimum, Math.Min(f.nudMaxRx2d.Maximum, MaxRx2d));
				f.nudMaxMaterialAngle.Value = Math.Max(f.nudMaxMaterialAngle.Minimum, Math.Min(f.nudMaxMaterialAngle.Maximum, MaxMaterialAngle));
				f.nudMaxVertex.Value = Math.Max(f.nudMaxVertex.Minimum, Math.Min(f.nudMaxVertex.Maximum, MaxVertex));
				f.cboResolution.SelectedIndex = Math.Max(0, Math.Min(f.cboResolution.Items.Count - 1, iResolution));
				f.nudNearPixel.Value = NearPixel;
				f.nudPointSize.Value = PointSize;
				f.nudThread.Value = nThread;
				f.chkClose.Checked = bClose;

				// [OK]がクリックされたらウィンドウの値を変数に代入する 
				if (f.ShowDialog() == DialogResult.OK)
				{
					MaxMaterial = (int)f.nudMaxMaterial.Value;
					MaxAntenna = (int)f.nudMaxAntenna.Value;
					MaxPillar = (int)f.nudMaxPillar.Value;
					MaxPolygon = (int)f.nudMaxPolygon.Value;
					MaxTx = (int)f.nudMaxTx.Value;
					MaxRx0d = (int)f.nudMaxRx0d.Value;
					MaxRx1d = (int)f.nudMaxRx1d.Value;
					MaxRx2d = (int)f.nudMaxRx2d.Value;
					MaxMaterialAngle = (int)f.nudMaxMaterialAngle.Value;
					MaxVertex = (int)f.nudMaxVertex.Value;
					iResolution = f.cboResolution.SelectedIndex;
					NearPixel = (int)f.nudNearPixel.Value;
					PointSize = (int)f.nudPointSize.Value;
					nThread = (int)f.nudThread.Value;
					bClose = f.chkClose.Checked;

					// 分解能
					tssResolution.Text = dResolution[iResolution] + "m/div";
				}
			}
			catch { }
		}

		// [計算]/[図形出力]ボタン
		private void cmdRun_Click(object sender, EventArgs e)
		{
			try
			{
				// 一時ファイル名
				string tmpfile = "_tmp.ort";
				string path = Application.StartupPath + Path.DirectorySeparatorChar + tmpfile;

				// 各プロセス
				if (sender == btnGeom3d)
				{
					// [形状表示3D]
					// 上書きされるファイルは削除しておく
					string fname = "geom.ev3";
					if (File.Exists(fname)) File.Delete(fname);
					// 一時ファイルへ保存する
					writeFile(path, 2, 0);
					// プロセス
					ProcessStartInfo psi = new ProcessStartInfo();
					// ウィンドウ非表示
					psi.WindowStyle = ProcessWindowStyle.Minimized;
					// 実行プログラム
					psi.FileName = "ort.exe";
					// 引数
					psi.Arguments = tmpfile;
					// プロセス実行
					Process proc = Process.Start(psi);

					// 終了するまで待機
					proc.WaitForExit();

					// 引き続き図形表示を行う
					ProcessStartInfo psi2 = new ProcessStartInfo();
					// 実行プログラム
					psi2.FileName = "ev3d_otk.exe";
					// 引数
					psi2.Arguments = fname;
					// 実行
					Process proc2 = Process.Start(psi2);
				}
				else if (sender == btnRay3d)
				{
					// [レイ表示3D]
					// 上書きされるファイルは削除しておく
					string fname = "ray.ev3";
					if (File.Exists(fname)) File.Delete(fname);
					// 一時ファイルへ保存する
					writeFile(path, 0, 1);
					// プロセス
					ProcessStartInfo psi = new ProcessStartInfo();
					// ウィンドウ非表示
					psi.WindowStyle = ProcessWindowStyle.Minimized;
					// 実行プログラム
					psi.FileName = "ort.exe";
					// 引数
					psi.Arguments = tmpfile;
					// プロセス実行
					Process proc = Process.Start(psi);

					// 終了するまで待機
					proc.WaitForExit();

					// 引き続き図形表示を行う
					ProcessStartInfo psi2 = new ProcessStartInfo();
					// 実行プログラム
					psi2.FileName = "ev3d_otk.exe";
					// 引数
					psi2.Arguments = fname;
					// 実行
					Process proc2 = Process.Start(psi2);
				}
				else if (sender == btnSolver)
				{
					// [計算]
					// 上書きされるファイルは削除しておく
					string[] strfile = { "ort.out", "ort.log", "field.log", "path.log" };
					foreach (string str in strfile)
					{
						if (File.Exists(str)) File.Delete(str);
					}
					// 一時ファイルへ保存する
					writeFile(path, 0, 0);
					// プロセス
					Process proc = new Process();
					// 実行プログラム
					proc.StartInfo.FileName = "ort.exe";
					// 引数
					string prompt = bClose ? "" : "-prompt ";
					proc.StartInfo.Arguments = "-n " + nThread + " " + prompt + tmpfile;
					// プロセス実行
					proc.Start();
				}
				else if (sender == btnPost)
				{
					// [ポスト処理]
					// 上書きされるファイルは削除しておく
					string[] strfile = { "ev.ev2", "ev.ev3" };
					foreach (string str in strfile)
					{
						if (File.Exists(str)) File.Delete(str);
					}
					// 一時ファイルへ保存する
					writeFile(path, 0, 0);
					// プロセス
					Process proc = new Process();
					// 実行プログラム
					proc.StartInfo.FileName = "ort_post.exe";
					// 引数
					proc.StartInfo.Arguments = tmpfile;
					// プロセス実行
					proc.Start();
				}
				else if (sender == btnEv2d)
				{
					// [図形出力2D]
					string fn = "ev.ev2";
					FileInfo fi = new FileInfo(fn);
					if (fi.Length == 0)
					{
						throw new Exception("ファイル " + fn + " にデータがありません。");
					}
					// プロセス
					Process proc = new Process();
					// 実行プログラム
					proc.StartInfo.FileName = "ev2d.exe";
					// 引数
					proc.StartInfo.Arguments = fn;
					// プロセス実行
					proc.Start();
				}
				else if (sender == btnEv3d)
				{
					// [図形出力3D]
					string fn = "ev.ev3";
					FileInfo fi = new FileInfo(fn);
					if (fi.Length == 0)
					{
						throw new Exception("ファイル " + fn + " にデータがありません。");
					}
					// プロセス
					Process proc = new Process();
					// 実行プログラム
					//proc.StartInfo.FileName = "ev3d.exe";
					proc.StartInfo.FileName = "ev3d_otk.exe";
					// 引数
					proc.StartInfo.Arguments = fn;
					// プロセス実行
					proc.Start();
				}
			}
			catch (Exception ex)
			{
				MessageBox.Show(ex.Message, Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
			}
			finally { }
		}

		// その他のコマンド：関連付けを使用する
		private void cmdProcess_Click(object sender, EventArgs e)
		{
			try
			{
				// ファイル名取得
				string path = String.Empty;
				if (sender == mnuortlog)
				{
					path = "ort.log";
					if (!File.Exists(path)) throw new Exception(path);
				}
				else if (sender == mnufieldlog)
				{
					path = "field.log";
					if (!File.Exists(path)) throw new Exception(path);
				}
				else if (sender == mnupathlog)
				{
					path = "path.log";
					if (!File.Exists(path)) throw new Exception(path);
				}
				else if (sender == mnuOpenRTM)
				{
					path = "http://emoss.starfree.jp/OpenRTM/";
				}

				// プロセス開始
				Process.Start(path);
			}
			catch (Exception ex)
			{
				string msg = "ファイル " + ex.Message + " は出力されていません。";
				MessageBox.Show(msg, Application.ProductName);
			}
		}

		// ウィンドウサイズ変更
		private void OpenRTM_Resize(object sender, EventArgs e)
		{
			// サイズ変更
			tabPage.Height = this.ClientSize.Height - menuStrip.Height - statusStrip.Height - btnSolver.Height - 20;
			picMap.Width = tabPage.Width - grpMode.Width - 20;

			// 再描画
			picMap.Invalidate();
		}

		// 物体形状：[編集]/[削除] ボタン/コンテキストメニュー
		private void EditDelete_Click(object sender, EventArgs e)
		{
			if ((sender == btnEdit) || (sender == btnDelete)) {
				// iTarget = 0...5 : 対象番号
				iTarget =
					rbnTarget0.Checked ? 0 :
					rbnTarget1.Checked ? 1 :
					rbnTarget2.Checked ? 2 :
					rbnTarget3.Checked ? 3 :
					rbnTarget4.Checked ? 4 :
					rbnTarget5.Checked ? 5 : -1;
				// iDataid = 0,1,... : データ番号
				iDataid =
					rbnTarget0.Checked ? (int)nudTarget0.Value - 1 :
					rbnTarget1.Checked ? (int)nudTarget1.Value - 1 :
					rbnTarget2.Checked ? (int)nudTarget2.Value - 1 :
					rbnTarget3.Checked ? (int)nudTarget3.Value - 1 :
					rbnTarget4.Checked ? (int)nudTarget4.Value - 1 :
					rbnTarget5.Checked ? (int)nudTarget5.Value - 1 : -1;

				if (iDataid < 0) {
					return;
				}
				// データ番号の上限チェック
				string msg1 = "データ番号=" + (iDataid + 1) + "がデータ数=";
				string msg2 = "を超えています。";
				string strbutton = ((Button)sender).Text;
				if (rbnTarget0.Checked && (iDataid >= NPillar)) {
					string msg = msg1 + NPillar + msg2;
					string caption = strbutton + " " + rbnTarget0.Text;
					MessageBox.Show(msg, caption, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
					return;
				}
				if (rbnTarget1.Checked && (iDataid >= NPolygon)) {
					string msg = msg1 + NPolygon + msg2;
					string caption = strbutton + " " + rbnTarget1.Text;
					MessageBox.Show(msg, caption, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
					return;
				}
				if (rbnTarget2.Checked && (iDataid >= NTx)) {
					string msg = msg1 + NTx + msg2;
					string caption = strbutton + " " + rbnTarget2.Text;
					MessageBox.Show(msg, caption, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
					return;
				}
				if (rbnTarget3.Checked && (iDataid >= NRx0d)) {
					string msg = msg1 + NRx0d + msg2;
					string caption = strbutton + " " + rbnTarget3.Text;
					MessageBox.Show(msg, caption, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
					return;
				}
				if (rbnTarget4.Checked && (iDataid >= NRx1d)) {
					string msg = msg1 + NRx1d + msg2;
					string caption = strbutton + " " + rbnTarget4.Text;
					MessageBox.Show(msg, caption, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
					return;
				}
				if (rbnTarget5.Checked && (iDataid >= NRx2d)) {
					string msg = msg1 + NRx2d + msg2;
					string caption = strbutton + " " + rbnTarget5.Text;
					MessageBox.Show(msg, caption, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
					return;
				}
			}

			if ((sender == mnuEdit) || (sender == btnEdit)) {
				// [編集]
				EditTarget();
			}
			else if ((sender == mnuDelete) || (sender == btnDelete)) {
				// [削除]
				DeleteTarget();
			}

			// 再描画
			picMap.Invalidate();
		}

		// 物体形状：[編集]
		private void EditTarget()
		{
			if ((iTarget < 0) || (iDataid < 0)) return;

			int xypos =
				(iTarget == 0) ? Pillar[iDataid].nvertex :
				(iTarget == 1) ? Polygon[iDataid].nvertex :
				(iTarget == 2) ? 1 :
				(iTarget == 3) ? 1 :
				(iTarget == 4) ? 2 :
				(iTarget == 5) ? 4 : 0;
			int zpos =
				(iTarget == 0) ? 2 :
				(iTarget == 1) ? Polygon[iDataid].nvertex :
				(iTarget == 2) ? 1 :
				(iTarget == 3) ? 1 :
				(iTarget == 4) ? 2 :
				(iTarget == 5) ? 4 : 0;
			string title = mnuEdit.Text + " " + (
				(iTarget == 0) ? rbnTarget0.Text :
				(iTarget == 1) ? rbnTarget1.Text :
				(iTarget == 2) ? rbnTarget2.Text :
				(iTarget == 3) ? rbnTarget3.Text :
				(iTarget == 4) ? rbnTarget4.Text :
				(iTarget == 5) ? rbnTarget5.Text : string.Empty) +
				" No. " + (iDataid + 1).ToString();

			// [編集]ウィンドウを開く
			frmEdit f = new frmEdit(iTarget, title, xypos, zpos);

			// ウィンドウに値を代入する
			if (iTarget == 0) {
				// 多角柱
				f.txtZ[0].Text = Pillar[iDataid].z[1].ToString(); // 上面
				f.txtZ[1].Text = Pillar[iDataid].z[0].ToString(); // 下面
				for (int i = 0; i < xypos; i++) {
					f.txtX[i].Text = Pillar[iDataid].x[i].ToString();
					f.txtY[i].Text = Pillar[iDataid].y[i].ToString();
				}
				setEditList(0, f.cboList, Pillar[iDataid].material);
			}
			else if (iTarget == 1) {
				// 多角形
				for (int i = 0; i < xypos; i++) {
					f.txtX[i].Text = Polygon[iDataid].x[i].ToString();
					f.txtY[i].Text = Polygon[iDataid].y[i].ToString();
					f.txtZ[i].Text = Polygon[iDataid].z[i].ToString();
				}
				setEditList(0, f.cboList, Polygon[iDataid].material);
			}
			else if (iTarget == 2) {
				// 送信点
				f.txtX[0].Text = Tx[iDataid].pos[0].ToString();
				f.txtY[0].Text = Tx[iDataid].pos[1].ToString();
				f.txtZ[0].Text = Tx[iDataid].pos[2].ToString();
				f.txtTx1.Text = Tx[iDataid].power.ToString();
				f.txtTx2.Text = Tx[iDataid].phase.ToString();
				setEditList(1, f.cboList, Tx[iDataid].antenna);
			}
			else if (iTarget == 3) {
				// 観測点
				f.txtX[0].Text = Rx0d[iDataid].pos[0].ToString();
				f.txtY[0].Text = Rx0d[iDataid].pos[1].ToString();
				f.txtZ[0].Text = Rx0d[iDataid].pos[2].ToString();
				setEditList(1, f.cboList, Rx0d[iDataid].antenna);
			}
			else if (iTarget == 4) {
				// 観測線
				for (int i = 0; i < 2; i++) {
					f.txtX[i].Text = Rx1d[iDataid].pos[i, 0].ToString();
					f.txtY[i].Text = Rx1d[iDataid].pos[i, 1].ToString();
					f.txtZ[i].Text = Rx1d[iDataid].pos[i, 2].ToString();
				}
				f.nudDiv12.Value = Rx1d[iDataid].div;
				setEditList(1, f.cboList, Rx1d[iDataid].antenna);
			}
			else if (iTarget == 5) {
				// 観測面
				for (int i = 0; i < 4; i++) {
					f.txtX[i].Text = Rx2d[iDataid].pos[i, 0].ToString();
					f.txtY[i].Text = Rx2d[iDataid].pos[i, 1].ToString();
					f.txtZ[i].Text = Rx2d[iDataid].pos[i, 2].ToString();
				}
				f.nudDiv12.Value = Rx2d[iDataid].div[0];
				f.nudDiv14.Value = Rx2d[iDataid].div[1];
				setEditList(1, f.cboList, Rx2d[iDataid].antenna);
			}

			// 編集ウィンドウで[OK]クリックした: 変数に値を代入する
			if (f.ShowDialog() == DialogResult.OK) {
				// XY座標は分解能あり
				if (iTarget == 0) {
					// 多角柱
					Double.TryParse(f.txtZ[0].Text, out Pillar[iDataid].z[1]);
					Double.TryParse(f.txtZ[1].Text, out Pillar[iDataid].z[0]);
					for (int i = 0; i < xypos; i++) {
						Pillar[iDataid].x[i] = number_resolution(f.txtX[i].Text);
						Pillar[iDataid].y[i] = number_resolution(f.txtY[i].Text);
					}
					if (xypos > 2) {
						Pillar[iDataid].x[xypos] = Pillar[iDataid].x[0];
						Pillar[iDataid].y[xypos] = Pillar[iDataid].y[0];
					}
					Pillar[iDataid].material = f.cboList.SelectedIndex;
					Pillar[iDataid].defined = true;
				}
				else if (iTarget == 1) {
					// 多角形
					for (int i = 0; i < xypos; i++) {
						Polygon[iDataid].x[i] = number_resolution(f.txtX[i].Text);
						Polygon[iDataid].y[i] = number_resolution(f.txtY[i].Text);
						Double.TryParse(f.txtZ[i].Text, out Polygon[iDataid].z[i]);
					}
					Polygon[iDataid].x[xypos] = Polygon[iDataid].x[0];
					Polygon[iDataid].y[xypos] = Polygon[iDataid].y[0];
					Polygon[iDataid].material = f.cboList.SelectedIndex;
					Polygon[iDataid].defined = true;
				}
				else if (iTarget == 2) {
					// 送信点
					Tx[iDataid].pos[0] = number_resolution(f.txtX[0].Text);
					Tx[iDataid].pos[1] = number_resolution(f.txtY[0].Text);
					Double.TryParse(f.txtZ[0].Text, out Tx[iDataid].pos[2]);
					Double.TryParse(f.txtTx1.Text, out Tx[iDataid].power);
					Double.TryParse(f.txtTx2.Text, out Tx[iDataid].phase);
					if (f.cboList.SelectedIndex >= 0) {
						Tx[iDataid].antenna = f.cboList.SelectedIndex;
						Tx[iDataid].defined = true;
					}
				}
				else if (iTarget == 3) {
					// 観測点
					Rx0d[iDataid].pos[0] = number_resolution(f.txtX[0].Text);
					Rx0d[iDataid].pos[1] = number_resolution(f.txtY[0].Text);
					Double.TryParse(f.txtZ[0].Text, out Rx0d[iDataid].pos[2]);
					if (f.cboList.SelectedIndex >= 0) {
						Rx0d[iDataid].antenna = f.cboList.SelectedIndex;
						Rx0d[iDataid].defined = true;
					}
				}
				else if (iTarget == 4) {
					// 観測線
					for (int i = 0; i < 2; i++) {
						Rx1d[iDataid].pos[i, 0] = number_resolution(f.txtX[i].Text);
						Rx1d[iDataid].pos[i, 1] = number_resolution(f.txtY[i].Text);
						Double.TryParse(f.txtZ[i].Text, out Rx1d[iDataid].pos[i, 2]);
					}
					Rx1d[iDataid].div = (int)f.nudDiv12.Value;
					if (f.cboList.SelectedIndex >= 0) {
						Rx1d[iDataid].antenna = f.cboList.SelectedIndex;
						Rx1d[iDataid].defined = true;
					}
				}
				else if (iTarget == 5) {
					// 観測面
					for (int i = 0; i < 4; i++) {
						Rx2d[iDataid].pos[i, 0] = number_resolution(f.txtX[i].Text);
						Rx2d[iDataid].pos[i, 1] = number_resolution(f.txtY[i].Text);
						//Rx2d[iDataid].pos[i, 2] = number_resolution(f.txtZ[i].Text);
						Double.TryParse(f.txtZ[i].Text, out Rx2d[iDataid].pos[i, 2]);
					}
					Rx2d[iDataid].div[0] = (int)f.nudDiv12.Value;
					Rx2d[iDataid].div[1] = (int)f.nudDiv14.Value;
					if (f.cboList.SelectedIndex >= 0) {
						Rx2d[iDataid].antenna = f.cboList.SelectedIndex;
						Rx2d[iDataid].defined = true;
					}
				}

				// 統計表示(再描画は不要)
				showNumberOfData();
			}

			// 選択解除
			//iTarget = -1;
			//iDataid = -1;
		}

		// 物体形状：[削除]
		private void DeleteTarget()
		{
			if ((iTarget < 0) || (iDataid < 0)) return;

			string obj =
				(iTarget == 0) ? rbnTarget0.Text :
				(iTarget == 1) ? rbnTarget1.Text :
				(iTarget == 2) ? rbnTarget2.Text :
				(iTarget == 3) ? rbnTarget3.Text :
				(iTarget == 4) ? rbnTarget4.Text :
				(iTarget == 5) ? rbnTarget5.Text : string.Empty;
			string msg = obj + " No." + (iDataid + 1).ToString() + " を削除します";

			if (MessageBox.Show(msg, mnuDelete.Text, MessageBoxButtons.OKCancel, MessageBoxIcon.Exclamation) == DialogResult.OK) {
				if (iTarget == 0) {
					for (int n = iDataid; n < NPillar - 1; n++) {
						Pillar[n] = Pillar[n + 1];
					}
					NPillar--;
				}
				else if (iTarget == 1) {
					for (int n = iDataid; n < NPolygon - 1; n++) {
						Polygon[n] = Polygon[n + 1];
					}
					NPolygon--;
				}
				else if (iTarget == 2) {
					for (int n = iDataid; n < NTx - 1; n++) {
						Tx[n] = Tx[n + 1];
					}
					NTx--;
				}
				else if (iTarget == 3) {
					for (int n = iDataid; n < NRx0d - 1; n++) {
						Rx0d[n] = Rx0d[n + 1];
					}
					NRx0d--;
				}
				else if (iTarget == 4) {
					for (int n = iDataid; n < NRx1d - 1; n++) {
						Rx1d[n] = Rx1d[n + 1];
					}
					NRx1d--;
				}
				else if (iTarget == 5) {
					for (int n = iDataid; n < NRx2d - 1; n++) {
						Rx2d[n] = Rx2d[n + 1];
					}
					NRx2d--;
				}

				// 統計表示(再描画不要)
				showNumberOfData();
			}

			// 選択解除
			//iTarget = -1;
			//iDataid = -1;
		}

		// 物体形状：[全削除]ボタン
		private void btnDeleteAll_Click(object sender, EventArgs e) {
			string strobj = String.Empty;
			if (rbnTarget0.Checked) {
				strobj = rbnTarget0.Text;
			}
			else if (rbnTarget1.Checked) {
				strobj = rbnTarget1.Text;
			}
			else if (rbnTarget2.Checked) {
				strobj = rbnTarget2.Text;
			}
			else if (rbnTarget3.Checked) {
				strobj = rbnTarget3.Text;
			}
			else if (rbnTarget4.Checked) {
				strobj = rbnTarget4.Text;
			}
			else if (rbnTarget5.Checked) {
				strobj = rbnTarget5.Text;
			}

			string msg = strobj + "をすべて削除します。";
			if (MessageBox.Show(msg, btnDeleteAll.Text, MessageBoxButtons.OKCancel, MessageBoxIcon.Exclamation) == DialogResult.OK) {
				// データ数=0
				if (rbnTarget0.Checked) {
					NPillar = 0;
					nudTarget0.Value = nudTarget0.Minimum;
				}
				else if (rbnTarget1.Checked) {
					NPolygon = 0;
					nudTarget1.Value = nudTarget1.Minimum;
				}
				else if (rbnTarget2.Checked) {
					NTx = 0;
					nudTarget2.Value = nudTarget2.Minimum;
				}
				else if (rbnTarget3.Checked) {
					NRx0d = 0;
					nudTarget3.Value = nudTarget3.Minimum;
				}
				else if (rbnTarget4.Checked) {
					NRx1d = 0;
					nudTarget4.Value = nudTarget4.Minimum;
				}
				else if (rbnTarget5.Checked) {
					NRx2d = 0;
					nudTarget5.Value = nudTarget5.Minimum;
				}

				// 画面再表示
				picMap.Invalidate();

				// 統計再表示
				showNumberOfData();
			}
		}

		// 物体形状：[全編集]ボタン
		private void btnEditAll_Click(object sender, EventArgs e) {
			// iTarget = 0...5 : 対象番号
			iTarget =
				rbnTarget0.Checked ? 0 :
				rbnTarget1.Checked ? 1 :
				rbnTarget2.Checked ? 2 :
				rbnTarget3.Checked ? 3 :
				rbnTarget4.Checked ? 4 :
				rbnTarget5.Checked ? 5 : -1;
			if (iTarget < 0) {
				return;
			}
			// 編集するZ座標の数
			int zpos =
				(iTarget == 0) ? 2 :
				(iTarget == 1) ? 0 :
				(iTarget == 2) ? 0 :
				(iTarget == 3) ? 0 :
				(iTarget == 4) ? 0 :
				(iTarget == 5) ? 0 : 0;
			// 対象名
			string strobj =
				(iTarget == 0) ? rbnTarget0.Text :
				(iTarget == 1) ? rbnTarget1.Text :
				(iTarget == 2) ? rbnTarget2.Text :
				(iTarget == 3) ? rbnTarget3.Text :
				(iTarget == 4) ? rbnTarget4.Text :
				(iTarget == 5) ? rbnTarget5.Text : String.Empty;
			// データがあるかチェックする
			if (((iTarget == 0) && (NPillar <= 0)) ||
				((iTarget == 1) && (NPolygon <= 0)) ||
				((iTarget == 2) && (NTx <= 0)) ||
				((iTarget == 3) && (NRx0d <= 0)) ||
				((iTarget == 4) && (NRx1d <= 0)) ||
				((iTarget == 5) && (NRx2d <= 0))) {
				string msg = strobj + "のデータがありません。";
				MessageBox.Show(msg, "全編集", MessageBoxButtons.OK);
				return;
			}

			// [編集]ウィンドウを開く
			string caption = "全編集" + " " + strobj;
			frmEdit f = new frmEdit(iTarget, caption, 0, zpos);

			// [編集]ウィンドウに値を代入する
			if (iTarget == 0) {
				f.txtZ[0].Text = f.txtZ[1].Text = "0";
				setEditList(0, f.cboList, 0);
			}
			else if (iTarget == 1) {
				setEditList(0, f.cboList, 0);
			}
			else if (iTarget == 2) {
				setEditList(1, f.cboList, 0);
			}
			else if (iTarget == 3) {
				setEditList(1, f.cboList, 0);
			}
			else if (iTarget == 4) {
				setEditList(1, f.cboList, 0);
			}
			else if (iTarget == 5) {
				setEditList(1, f.cboList, 0);
			}

			// [編集]ウィンドウで[OK]クリックした: 変数に値を代入する
			if (f.ShowDialog() == DialogResult.OK) {
				if (iTarget == 0) {
					// 多角柱
					for (int n = 0; n < NPillar; n++) {
						Double.TryParse(f.txtZ[0].Text, out Pillar[n].z[1]);  // 上面
						Double.TryParse(f.txtZ[1].Text, out Pillar[n].z[0]);  // 下面
						Pillar[n].material = f.cboList.SelectedIndex;
						Pillar[n].defined = true;
					}
				}
				else if (iTarget == 1) {
					// 多角形
					for (int n = 0; n < NPolygon; n++) {
						Polygon[n].material = f.cboList.SelectedIndex;
						Polygon[n].defined = true;
					}
				}
				else if (iTarget == 2) {
					// 送信点
					for (int n = 0; n < NTx; n++) {
						Double.TryParse(f.txtTx1.Text, out Tx[n].power);
						Double.TryParse(f.txtTx2.Text, out Tx[n].phase);
						if (f.cboList.SelectedIndex >= 0) {
							Tx[n].antenna = f.cboList.SelectedIndex;
							Tx[n].defined = true;
						}
					}
				}
				else if (iTarget == 3) {
					// 観測点
					for (int n = 0; n < NRx0d; n++) {
						if (f.cboList.SelectedIndex >= 0) {
							Rx0d[n].antenna = f.cboList.SelectedIndex;
							Rx0d[n].defined = true;
						}
					}
				}
				else if (iTarget == 4) {
					// 観測線
					for (int n = 0; n < NRx1d; n++) {
						Rx1d[n].div = (int)f.nudDiv12.Value;
						if (f.cboList.SelectedIndex >= 0) {
							Rx1d[n].antenna = f.cboList.SelectedIndex;
							Rx1d[n].defined = true;
						}
					}
				}
				else if (iTarget == 5) {
					// 観測面
					for (int n = 0; n < NRx2d; n++) {
						Rx2d[n].div[0] = (int)f.nudDiv12.Value;
						Rx2d[n].div[1] = (int)f.nudDiv14.Value;
						if (f.cboList.SelectedIndex >= 0) {
							Rx2d[n].antenna = f.cboList.SelectedIndex;
							Rx2d[n].defined = true;
						}
					}
				}
			}

			// 再描画
			picMap.Invalidate();

			// 統計情報更新
			showNumberOfData();
		}

		// 物体形状
		// [物性値]/[アンテナ]リストを表示する
		// kind = 0/1 : 物性値/アンテナ
		// id = 0,1,... : データ番号
		private void setEditList(int kind, ComboBox cbo, int id)
		{
			cbo.Items.Clear();

			if (kind == 0)
			{
				// 物性値
				cbo.Items.Add("[0] Air");
				cbo.Items.Add("[1] PEC");
				int nmaterial = 2;
				for (int n = 0; n < chkMaterialNo.Length; n++)
				{
					if (chkMaterialNo[n].Checked)
					{
						int type = cboMaterialType[n].SelectedIndex;
						string str = "[" + (n + 2).ToString() + "] (" + cboMaterialType[n].SelectedItem.ToString() + ") ";
						if (type == 0)
						{
							str += txtEpsr[n].Text + " " + txtSigma[n].Text + " " + txtMaterialComment[n].Text;
						}
						else if (type == 1)
						{
							str += txtEpsr[n].Text + " " + txtSigma[n].Text + " " + txtThick[n].Text + " " + txtMaterialComment[n].Text;
						}
						else if (type == 2)
						{
							str += txtMaterialFile[n].Text + " " + txtMaterialComment[n].Text;
						}
						cbo.Items.Add(str);
						nmaterial++;
					}
				}
				cbo.SelectedIndex = (id < nmaterial) ? id : 1;
			}
			else if (kind == 1)
			{
				// アンテナ
				int nantenna = 0;
				for (int n = 0; n < chkAntennaNo.Length; n++)
				{
					if (chkAntennaNo[n].Checked)
					{
						int type = cboAntennaType[n].SelectedIndex;
						string str = "[" + (n + 1).ToString() + "] (" + cboAntennaType[n].SelectedItem.ToString() + ") ";
						if (type == 0)
						{
							str += "(" + cboPol[n].SelectedItem.ToString() + ") " + txtAntennaComment[n].Text;
						}
						else if (type == 1)
						{
							str += "(" + cboPol[n].SelectedItem.ToString() + ") " + txtTheta[n].Text + " " + txtPhi[n].Text + " " + txtBw[n].Text + " " + txtAntennaComment[n].Text;
						}
						else if (type == 2)
						{
							str += "(" + cboPol[n].SelectedItem.ToString() + ") " + txtTheta[n].Text + " " + txtPhi[n].Text + " " + txtBwTheta[n].Text + " " + txtBwPhi[n].Text + " " + txtAntennaComment[n].Text;
						}
						else if (type == 3)
						{
							str += txtAntennaFile[n].Text + " " + txtAntennaComment[n].Text;
						}
						cbo.Items.Add(str);
						nantenna++;
					}
				}

				if (nantenna > 0)
				{
					cbo.SelectedIndex = ((id >= 0) && (id < nantenna)) ? id : 0;
				}
				else
				{
					MessageBox.Show("アンテナデータが入力されていません。", "編集");
				}
			}
		}

		// 物体形状：ズーム
		private void btnZoom(object sender, EventArgs e)
		{
			double px = 0, py = 0;
			double scale_old = dScale;

			if (sender == btnZoomIn)
			{
				// 拡大ボタン
				screen2physical(picMap.Width / 2, picMap.Height / 2, out px, out py);
				dScale *= Math.Sqrt(2);
			}
			else if (sender == btnZoomOut)
			{
				// 縮小ボタン
				screen2physical(picMap.Width / 2, picMap.Height / 2, out px, out py);
				dScale *= Math.Sqrt(0.5);
			}
			else if (sender == picMap)
			{
				// マウスホイール
				MouseEventArgs me = (MouseEventArgs)e;
				screen2physical(me.X, me.Y, out px, out py);
				const double f = 1.1;
				int d = ((MouseEventArgs)e).Delta * SystemInformation.MouseWheelScrollLines / 120;
				dScale *= (d > 0) ? f : (1 / f);
			}

			// 左下の物理座標
			Xleft = px + (scale_old / dScale) * (Xleft - px);
			Ybottom = py + (scale_old / dScale) * (Ybottom - py);

			// 描画
			picMap.Invalidate();
		}

		// 物体形状：[尺度初期化]ボタン
		private void btnZoomInitialize_Click(object sender, EventArgs e)
		{
			// スケール設定
			initializeScale();

			// 描画
			picMap.Invalidate();
		}

		// 物体形状：スケール設定
		private void initializeScale()
		{
			// XY座標の最小・最大
			getXYSpan(out double xmin, out double ymin, out double xmax, out double ymax);

			// 範囲がないときの処理
			double d0 = 5;
			double eps = 1e-6;
			if (NPillar + NPolygon + NTx + NRx0d + NRx1d + NRx2d <= 0)
			{
				xmin = ymin = -d0;
				xmax = ymax = +d0;
			}
			else if (Math.Abs(xmax - xmin) < eps && Math.Abs(ymax - ymin) < eps)
			{
				xmin -= d0; xmax += d0;
				ymin -= d0; ymax += d0;
			}
			else if (Math.Abs(xmax - xmin) < eps)
			{
				xmin -= d0; xmax += d0;
			}
			else if (Math.Abs(ymax - ymin) < eps)
			{
				ymin -= d0; ymax += d0;
			}

			// 左下のXY座標
			float w = picMap.Width;
			float h = picMap.Height;
			//Console.WriteLine("{0:0.000} {1:0.000}", w, h);
			double x0 = (xmin + xmax) / 2;
			double y0 = (ymin + ymax) / 2;
			const double a = 0.1;       // margin
			double lx = (1 + a) * (xmax - xmin);
			double ly = (1 + a) * (ymax - ymin);
			if ((h / w) < (ly / lx))
			{
				Ybottom = y0 - (ly / 2);
				Xleft = x0 - (ly / 2) * (w / h);
			}
			else
			{
				Xleft = x0 - (lx / 2);
				Ybottom = y0 - (lx / 2) * (h / w);
			}

			// スケール
			dScale = Math.Min(picMap.Width / lx, picMap.Height / ly);
		}

		// 物体形状：ウィンドウ座標を物理座標に変換する
		private void screen2physical(float sx, float sy, out double px, out double py)
		{
			px = Xleft + (sx / dScale);
			py = Ybottom + (picMap.Height - sy) / dScale;
		}

		// 物体形状：物理座標をウィンドウ座標に変換する
		private void physical2screen(double px, double py, out float sx, out float sy)
		{
			sx = (float)(dScale * (px - Xleft));
			sy = picMap.Height - (float)(dScale * (py - Ybottom));
		}

		// 物体形状：描画
		private void picMap_Paint(object sender, PaintEventArgs e)
		{
			try
			{
				Graphics g = e.Graphics;

				// グラフィックリソース
				Pen pen_grid1 = new Pen(Color.FromArgb(220, 220, 220));
				Pen pen_grid2 = new Pen(Color.FromArgb(140, 140, 140)); ;
				Pen pen_pillar1 = new Pen(rbnTarget0.ForeColor, 1);
				Pen pen_pillar2 = new Pen(rbnTarget0.ForeColor, 2);
				Pen pen_polygon1 = new Pen(rbnTarget1.ForeColor, 1);
				Pen pen_polygon2 = new Pen(rbnTarget1.ForeColor, 2);
				Pen pen_tx1 = new Pen(rbnTarget2.ForeColor, 1);
				Pen pen_tx2 = new Pen(rbnTarget2.ForeColor, 2);
				Pen pen_rx1 = new Pen(rbnTarget3.ForeColor, 1);
				Pen pen_rx2 = new Pen(rbnTarget3.ForeColor, 2);
				Brush brush1 = new SolidBrush(Color.Red);
				Brush brush2 = new SolidBrush(Color.Black);// SkyBlue);
				Brush brush3 = new SolidBrush(Color.FromArgb(128, 255, 255, 214));// Color.LightYellow);

				// ウィンドウサイズ
				float w = picMap.Width;
				float h = picMap.Height;
				//Console.WriteLine("{0:0.0} {1:0.0}", w, h);

				// 数字の大きさ
				float vx = 0.6f * Font.Size;
				float vy = 1.0f * Font.Size;

				// グリッド
				float s = (float)dScale;
				float r = (float)dResolution[iResolution];
				// 縦線
				int ix1 = (int)Math.Ceiling(Xleft / r);
				int ix2 = (int)Math.Floor((Xleft + w / s) / r);
				for (int ix = ix1; ix <= ix2; ix++)
				{
					float x = (float)(s * ((ix * r) - Xleft));
					g.DrawLine((ix % 10 != 0 ? pen_grid1 : pen_grid2), x, 0, x, h);
				}
				// 横線
				int iy1 = (int)Math.Ceiling(Ybottom / r);
				int iy2 = (int)Math.Floor((Ybottom + h / s) / r);
				for (int iy = iy1; iy <= iy2; iy++)
				{
					float y = h - (float)(s * ((iy * r) - Ybottom));
					g.DrawLine((iy % 10 != 0 ? pen_grid1 : pen_grid2), 0, y, w, y);
				}

				// 多角柱
				if (NPillar > 0)
				{
					for (int n = 0; n < NPillar; n++) {
						// 頂点座標
						PointF[] sp = new PointF[Pillar[n].nvertex];
						for (int i = 0; i < Pillar[n].nvertex; i++)
						{
							float sx, sy;
							physical2screen(Pillar[n].x[i], Pillar[n].y[i], out sx, out sy);
							sp[i] = new PointF(sx, sy);
						}
						// 頂点番号と内部
						if (rbnTarget0.Checked && (n == nudTarget0.Value - 1))
						{
							for (int i = 0; i < Pillar[n].nvertex; i++)
							{
								g.DrawString((i + 1).ToString(), Font, brush1, sp[i].X - vx, sp[i].Y - vy);
							}
							if (Pillar[n].nvertex > 2)
							{
								g.FillPolygon(brush3, sp);
							}
						}
						// 周囲
						Pen pen = Pillar[n].defined ? pen_pillar2 : pen_pillar1;
						g.DrawPolygon(pen, sp);
						// データ番号
						if (chkDataNumber.Checked && rbnTarget0.Checked)
						{
							float sx0 = 0, sy0 = 0;
							for (int i = 0; i < Pillar[n].nvertex; i++)
							{
								sx0 += sp[i].X;
								sy0 += sp[i].Y;
							}
							sx0 /= Pillar[n].nvertex;
							sy0 /= Pillar[n].nvertex;
							g.DrawString((n + 1).ToString(), Font, brush2, sx0 - vx, sy0 - vy);
						}
					}
				}

				// 多角形
				if (NPolygon > 0)
				{
					for (int n = 0; n < NPolygon; n++)
					{
						// 頂点座標
						PointF[] sp = new PointF[Polygon[n].nvertex];
						for (int i = 0; i < Polygon[n].nvertex; i++)
						{
							float sx, sy;
							physical2screen(Polygon[n].x[i], Polygon[n].y[i], out sx, out sy);
							sp[i] = new PointF(sx, sy);
						}
						// 頂点番号と内部
						if (rbnTarget1.Checked && (n == nudTarget1.Value - 1))
						{
							for (int i = 0; i < Polygon[n].nvertex; i++)
							{
								g.DrawString((i + 1).ToString(), Font, brush1, sp[i].X - vx, sp[i].Y - vy);
							}
							g.FillPolygon(brush3, sp);
						}
						// 周囲
						Pen pen = Polygon[n].defined ? pen_polygon2 : pen_polygon1;
						g.DrawPolygon(pen, sp);
						// データ番号
						if (chkDataNumber.Checked && rbnTarget1.Checked)
						{
							float sx0 = 0, sy0 = 0;
							for (int i = 0; i < Polygon[n].nvertex; i++)
							{
								sx0 += sp[i].X;
								sy0 += sp[i].Y;
							}
							sx0 /= Polygon[n].nvertex;
							sy0 /= Polygon[n].nvertex;
							g.DrawString((n + 1).ToString(), Font, brush2, sx0 - vx, sy0 - vy);
						}
					}
				}

				// 送信点
				if (NTx > 0)
				{
					for (int n = 0; n < NTx; n++)
					{
						Pen pen = Tx[n].defined ? pen_tx2 : pen_tx1;
						float sx, sy;
						physical2screen(Tx[n].pos[0], Tx[n].pos[1], out sx, out sy);
						//g.FillRectangle(brush3, sx - PointSize / 2, sy - PointSize / 2, PointSize, PointSize);
						g.DrawRectangle(pen, sx - PointSize / 2, sy - PointSize / 2, PointSize, PointSize);
						// データ番号
						if (chkDataNumber.Checked && rbnTarget2.Checked)
						{
							g.DrawString((n + 1).ToString(), Font, brush2, sx - vx, sy - vy);
						}
					}
				}

				// 観測点
				if (NRx0d > 0)
				{
					for (int n = 0; n < NRx0d; n++)
					{
						Pen pen = Rx0d[n].defined ? pen_rx2 : pen_rx1;
						float sx, sy;
						physical2screen(Rx0d[n].pos[0], Rx0d[n].pos[1], out sx, out sy);
						g.DrawRectangle(pen, sx - PointSize / 2, sy - PointSize / 2, PointSize, PointSize);
						// データ番号
						if (chkDataNumber.Checked && rbnTarget3.Checked)
						{
							g.DrawString((n + 1).ToString(), Font, brush2, sx - vx, sy - vy);
						}
					}
				}

				// 観測線
				if (NRx1d > 0)
				{
					// 頂点座標
					PointF[] sp = new PointF[2];
					for (int n = 0; n < NRx1d; n++)
					{
						// 頂点座標
						for (int i = 0; i < 2; i++)
						{
							float sx, sy;
							physical2screen(Rx1d[n].pos[i, 0], Rx1d[n].pos[i, 1], out sx, out sy);
							sp[i] = new PointF(sx, sy);
						}
						// 頂点番号
						if (rbnTarget4.Checked && (n == nudTarget4.Value - 1))
						{
							for (int i = 0; i < 2; i++)
							{
								g.DrawString((i + 1).ToString(), Font, brush1, sp[i].X - vx, sp[i].Y - vy);
							}
						}
						// 線分
						Pen pen = Rx1d[n].defined ? pen_rx2 : pen_rx1;
						g.DrawLines(pen, sp);
						// データ番号
						if (chkDataNumber.Checked && rbnTarget4.Checked)
						{
							float sx0 = 0, sy0 = 0;
							for (int i = 0; i < 2; i++)
							{
								sx0 += sp[i].X;
								sy0 += sp[i].Y;
							}
							g.DrawString((n + 1).ToString(), Font, brush2, sx0 / 2 - vx, sy0 / 2 - vy);
						}
					}
				}

				// 観測面
				if (NRx2d > 0)
				{
					PointF[] sp = new PointF[4];
					for (int n = 0; n < NRx2d; n++)
					{
						// 頂点座標
						for (int i = 0; i < 4; i++)
						{
							float sx, sy;
							physical2screen(Rx2d[n].pos[i, 0], Rx2d[n].pos[i, 1], out sx, out sy);
							sp[i] = new PointF(sx, sy);
						}
						// 頂点番号と内部
						if (rbnTarget5.Checked && (n == nudTarget5.Value - 1))
						{
							for (int i = 0; i < 4; i++)
							{
								g.DrawString((i + 1).ToString(), Font, brush1, sp[i].X - vx, sp[i].Y - vy);
							}
							g.FillPolygon(brush3, sp);
						}
						// 周囲
						Pen pen = Rx2d[n].defined ? pen_rx2 : pen_rx1;
						g.DrawPolygon(pen, sp);
						// データ番号
						if (chkDataNumber.Checked && rbnTarget5.Checked)
						{
							float sx0 = 0, sy0 = 0;
							for (int i = 0; i < 4; i++)
							{
								sx0 += sp[i].X;
								sy0 += sp[i].Y;
							}
							g.DrawString((n + 1).ToString(), Font, brush2, sx0 / 4 - vx, sy0 / 4 - vy);
						}
					}
				}

				// 入力中の線分
				if (rbnMode0.Checked && (nClicked > 0))
				{
					Pen pen = rbnTarget0.Checked ? pen_pillar1
							: rbnTarget1.Checked ? pen_polygon1
							: rbnTarget4.Checked ? pen_rx1
							: rbnTarget5.Checked ? pen_rx1 : Pens.Black;
					for (int i = 0; i < nClicked; i++)
					{
						physical2screen(PolygonX[i + 0], PolygonY[i + 0], out float sx1, out float sy1);
						physical2screen(PolygonX[i + 1], PolygonY[i + 1], out float sx2, out float sy2);
						double px, py;  // dummy
						resolution(sx1, sy1, out float rsx1, out float rsy1, out px, out py);
						resolution(sx2, sy2, out float rsx2, out float rsy2, out px, out py);
						g.DrawLine(pen, rsx1, rsy1, rsx2, rsy2);
					}
				}

				// リソース開放
				pen_grid1.Dispose();
				pen_grid2.Dispose();
				pen_pillar1.Dispose();
				pen_pillar2.Dispose();
				pen_polygon1.Dispose();
				pen_polygon2.Dispose();
				pen_tx1.Dispose();
				pen_tx2.Dispose();
				pen_rx1.Dispose();
				pen_rx2.Dispose();
				brush1.Dispose();
				brush2.Dispose();
				brush3.Dispose();
			}
			catch { }
		}

		// 物体形状：マウス移動
		private void picMap_MouseMove(object sender, MouseEventArgs e)
		{
			if (bDrag)
			{
				// マウスドラッグ：平行移動
				screen2physical(e.X, e.Y, out double px, out double py);
				Xleft += Xold - px;
				Ybottom += Yold - py;
				Xold = px;
				Yold = py;
			}
			else
			{
				// マウス移動：マウス位置の座標表示
				string fmt = "0.0";
				if ((iResolution == 0) || (iResolution == 1) || (iResolution == 2))
				{
					fmt = "0";
				}
				else if ((iResolution == 3) || (iResolution == 4))
				{
					fmt = "0.0";
				}
				else if ((iResolution == 5) || (iResolution == 6))
				{
					fmt = "0.00";
				}
				screen2physical(e.X, e.Y, out double px, out double py);
				tssX.Text = "X=" + px.ToString(fmt) + "m";
				tssY.Text = "Y=" + py.ToString(fmt) + "m";

				// マウス位置の物理座標(分解能あり)
				if (nClicked > 0)
				{
					resolution(e.X, e.Y, out float sx, out float sy, out double px_r, out double py_r);
					PolygonX[nClicked] = px_r;
					PolygonY[nClicked] = py_r;
				}
			}

			// 描画
			picMap.Invalidate();
		}

		// 物体形状：マウスクリック
		private void picMap_MouseDown(object sender, MouseEventArgs e)
		{
			// 現在地の物理座標(分解能なし)
			double px, py;
			screen2physical(e.X, e.Y, out px, out py);

			if (rbnMode0.Checked)
			{
				// === 入力 ===

				// 頂点の数が上限を超えた
				if (nClicked > MaxVertex - 2)
				{
					nClicked = 0;
					return;
				}

				// 同じ点は飛ばす
				if (nClicked > 0)
				{
					double d = dResolution[iResolution] / 2;
					if ((Math.Abs(px - PolygonX[nClicked - 1]) < d) &&
						(Math.Abs(py - PolygonY[nClicked - 1]) < d)) {
						return;
					}
				}

				// 現在地の座標(分解能あり)
				resolution(e.X, e.Y, out float sx, out float sy, out PolygonX[nClicked], out PolygonY[nClicked]);
				
				// 以下2行がないとクリックした瞬間に原点と結ばれた線がちらつく
				PolygonX[nClicked + 1] = PolygonX[nClicked];
				PolygonY[nClicked + 1] = PolygonY[nClicked];

				if (rbnTarget0.Checked && (NPillar < MaxPillar))
				{
					// N角柱(N=2,3...)
					if (e.Button == MouseButtons.Left)
					{
						// 左クリック
						nClicked++;
					}
					else if ((e.Button == MouseButtons.Right) && (nClicked > 0))
					{
						// 右クリック(点2以降):入力確定
						int nvertex = nClicked + 1;
						Pillar[NPillar] = new Pillar_t(nvertex);
						Pillar[NPillar].nvertex = nvertex;
						for (int i = 0; i < nvertex; i++)
						{
							Pillar[NPillar].x[i] = PolygonX[i];
							Pillar[NPillar].y[i] = PolygonY[i];
						}
						if (nClicked > 1)
						{
							// 3角形以上は最後の点と最初の点を結んで閉曲線にする
							Pillar[NPillar].x[nvertex] = Pillar[NPillar].x[0];
							Pillar[NPillar].y[nvertex] = Pillar[NPillar].y[0];
						}
						NPillar++;
						nudTarget0.Value = NPillar;
						showNumberOfData();
						nClicked = 0;
					}
					else if ((e.Button == MouseButtons.Right) && (nClicked == 0))
					{
						// 右クリック(未入力):編集
						if (isSelected(px, py, (Control.ModifierKeys & Keys.Shift) == Keys.Shift))
						{
							nudTarget0.Value = iDataid + 1;
							contextMenuStrip.Show(picMap, new Point(e.X, e.Y));
						}
					}
				}
				else if (rbnTarget1.Checked && (NPolygon < MaxPolygon))
				{
					// N角形(N=3,4...)
					if (e.Button == MouseButtons.Left)
					{
						// 左クリック
						nClicked++;
					}
					else if ((e.Button == MouseButtons.Right) && (nClicked > 1))
					{
						// 右クリック(点3以降):入力確定
						int nvertex = nClicked + 1;
						Polygon[NPolygon] = new Polygon_t(nvertex);
						Polygon[NPolygon].nvertex = nvertex;
						for (int i = 0; i < nvertex; i++)
						{
							Polygon[NPolygon].x[i] = PolygonX[i];
							Polygon[NPolygon].y[i] = PolygonY[i];
						}
						// 最後の点と最初の点を結んで閉曲線にする
						Polygon[NPolygon].x[nvertex] = Polygon[NPolygon].x[0];
						Polygon[NPolygon].y[nvertex] = Polygon[NPolygon].y[0];
						NPolygon++;
						nudTarget1.Value = NPolygon;
						showNumberOfData();
						nClicked = 0;
					}
					else if ((e.Button == MouseButtons.Right) && (nClicked == 0))
					{
						// 右クリック(未入力):編集
						if (isSelected(px, py, (Control.ModifierKeys & Keys.Shift) == Keys.Shift))
						{
							nudTarget1.Value = iDataid + 1;
							contextMenuStrip.Show(picMap, new Point(e.X, e.Y));
						}
					}
				}

				else if (rbnTarget2.Checked && (NTx < MaxTx))
				{
					// 送信点
					if (e.Button == MouseButtons.Left)
					{
						// 左クリック
						Tx[NTx] = new Tx_t();
						Tx[NTx].pos[0] = PolygonX[0];
						Tx[NTx].pos[1] = PolygonY[0];
						NTx++;
						nudTarget2.Value = NTx;
						showNumberOfData();
					}
					else if (e.Button == MouseButtons.Right)
					{
						// 右クリック:編集
						if (isSelected(px, py, (Control.ModifierKeys & Keys.Shift) == Keys.Shift))
						{
							nudTarget2.Value = iDataid + 1;
							contextMenuStrip.Show(picMap, new Point(e.X, e.Y));
						}
					}
				}

				else if (rbnTarget3.Checked && (NRx0d < MaxRx0d))
				{
					// 観測点
					if (e.Button == MouseButtons.Left)
					{
						// 左クリック
						Rx0d[NRx0d] = new Rx0d_t();
						Rx0d[NRx0d].pos[0] = PolygonX[0];
						Rx0d[NRx0d].pos[1] = PolygonY[0];
						NRx0d++;
						nudTarget3.Value = NRx0d;
						showNumberOfData();
					}
					else if (e.Button == MouseButtons.Right)
					{
						// 右クリック:編集
						if (isSelected(px, py, (Control.ModifierKeys & Keys.Shift) == Keys.Shift))
						{
							nudTarget3.Value = iDataid + 1;
							contextMenuStrip.Show(picMap, new Point(e.X, e.Y));
						}
					}
				}

				else if (rbnTarget4.Checked && (NRx0d < MaxRx0d))
				{
					// 観測線
					if (e.Button == MouseButtons.Left)
					{
						// 左クリック
						if (nClicked < 1)
						{
							nClicked++;
						}
						else
						{
							// 2クリック:入力確定
							Rx1d[NRx1d] = new Rx1d_t();
							for (int i = 0; i < nClicked + 1; i++)
							{
								Rx1d[NRx1d].pos[i, 0] = PolygonX[i];
								Rx1d[NRx1d].pos[i, 1] = PolygonY[i];
							}
							NRx1d++;
							nudTarget4.Value = NRx1d;
							showNumberOfData();
							nClicked = 0;
						}
					}
					else if ((e.Button == MouseButtons.Right) && (nClicked == 0))
					{
						// 右クリック(未入力):編集
						if (isSelected(px, py, (Control.ModifierKeys & Keys.Shift) == Keys.Shift))
						{
							nudTarget4.Value = iDataid + 1;
							contextMenuStrip.Show(picMap, new Point(e.X, e.Y));
						}
					}
				}

				else if (rbnTarget5.Checked && (NRx1d < MaxRx1d))
				{
					// 観測面
					if (e.Button == MouseButtons.Left)
					{
						// 左クリック
						if (nClicked < 3)
						{
							nClicked++;
						}
						else
						{
							// 4クリック:入力確定
							Rx2d[NRx2d] = new Rx2d_t();
							for (int i = 0; i < nClicked + 1; i++)
							{
								Rx2d[NRx2d].pos[i, 0] = PolygonX[i];
								Rx2d[NRx2d].pos[i, 1] = PolygonY[i];
							}
							NRx2d++;
							nudTarget5.Value = NRx2d;
							showNumberOfData();
							nClicked = 0;
						}
					}
					else if ((e.Button == MouseButtons.Right) && (nClicked == 0))
					{
						// 右クリック(未入力):編集
						if (isSelected(px, py, (Control.ModifierKeys & Keys.Shift) == Keys.Shift))
						{
							nudTarget5.Value = iDataid + 1;
							contextMenuStrip.Show(picMap, new Point(e.X, e.Y));
						}
					}
				}
			}
			else if (rbnMode1.Checked)
			{
				// === 編集 ===

				if (e.Button == MouseButtons.Left)
				{
					// 左クリック:マウスドラッグON
					bDrag = true;
					Xold = px;
					Yold = py;
				}
				else
				{
					// 右クリック、Shiftキーを押しているときは逆順に選択される
					if (isSelected(px, py, (Control.ModifierKeys & Keys.Shift) == Keys.Shift))
					{
						// データ番号代入
						if (iTarget == 0)
						{
							nudTarget0.Value = iDataid + 1;
						}
						else if (iTarget == 1)
						{
							nudTarget1.Value = iDataid + 1;
						}
						else if (iTarget == 2)
						{
							nudTarget2.Value = iDataid + 1;
						}
						else if (iTarget == 3)
						{
							nudTarget3.Value = iDataid + 1;
						}
						else if (iTarget == 4)
						{
							nudTarget4.Value = iDataid + 1;
						}
						else if (iTarget == 5)
						{
							nudTarget5.Value = iDataid + 1;
						}
						// ポップアップメニュー(Paint発生)
						contextMenuStrip.Show(picMap, new Point(e.X, e.Y));
					}
				}
			}

			// 描画
			picMap.Invalidate();
		}

		// 物体形状：マウスアップ
		private void picMap_MouseUp(object sender, MouseEventArgs e)
		{
			bDrag = false;
		}

		// 物体形状
		// ウィンドウ座標(sx,sy)から分解能を考慮したウィンドウ座標(sxout,syout)と物理座標(px,py)を取得する
		private void resolution(float sx, float sy, out float sxout, out float syout, out double pxout, out double pyout)
		{
			double r = dResolution[iResolution];  // [m]
			screen2physical(sx, sy, out double px, out double py);
			pxout = r * Math.Floor((px / r) + 0.5);
			pyout = r * Math.Floor((py / r) + 0.5);
			physical2screen(pxout, pyout, out sxout, out syout);
		}

		// 物体形状：データ数表示
		private void showNumberOfData()
		{
			int nmaterial = 2;
			for (int n = 0; n < chkMaterialNo.Length; n++) 
			{
				if (chkMaterialNo[n].Checked)
				{
					nmaterial++;
				}
			}

			int nantenna = 0;
			for (int n = 0; n < chkAntennaNo.Length; n++)
			{
				if (chkAntennaNo[n].Checked) {
					nantenna++;
				}
			}

			int npillar = 0;
			for (int n = 0; n < NPillar; n++) {
				if (Pillar[n].defined)
				{
					npillar++;
				}
			}
			int npolygon = 0;
			for (int n = 0; n < NPolygon; n++) {
				if (Polygon[n].defined)
				{
					npolygon++;
				}
			}
			int ntx = 0;
			for (int n = 0; n < NTx; n++)
			{
				if (Tx[n].defined)
				{
					ntx++;
				}
			}
			int nrx0d = 0;
			for (int n = 0; n < NRx0d; n++)
			{
				if (Rx0d[n].defined)
				{
					nrx0d++;
				}
			}
			int nrx1d = 0;
			for (int n = 0; n < NRx1d; n++) {
				if (Rx1d[n].defined) {
					nrx1d++;
				}
			}
			int nrx2d = 0;
			for (int n = 0; n < NRx2d; n++)
			{
				if (Rx2d[n].defined)
				{
					nrx2d++;
				}
			}

			// ステータスバー
			tssMaterial.Text = "物性値数=" + (nmaterial - 1).ToString();
			tssAntenna.Text = "アンテナ数=" + nantenna;

			// ラベル
			lblTarget0.Text = npillar + "/" + NPillar;
			lblTarget1.Text = npolygon + "/" + NPolygon;
			lblTarget2.Text = ntx + "/" + NTx;
			lblTarget3.Text = nrx0d + "/" + NRx0d;
			lblTarget4.Text = nrx1d + "/" + NRx1d;
			lblTarget5.Text = nrx2d + "/" + NRx2d;
		}

		// 物体形状：マウスでデータを選択したか
		// iTarget = -1/0/1/2/3/4/5
		// iDataid = -1/0...
		// 未選択のときはiTarget=-1,iDataid=-1
		// last : 後優先
		private bool isSelected(double x, double y, bool last)
		{
			iTarget = -1;
			iDataid = -1;

			if (rbnTarget0.Checked)
			{
				for (int n = 0; n < NPillar; n++)
				{
					int nline = (Pillar[n].nvertex < 3) ? (Pillar[n].nvertex - 1) : Pillar[n].nvertex;
					for (int i = 0; i < nline; i++)
					{
						if (nearline(x, y, Pillar[n].x[i], Pillar[n].y[i], Pillar[n].x[i + 1], Pillar[n].y[i + 1]))
						{
							iTarget = 0;
							iDataid = n;
							if (!last) return true;
						}
					}
				}
			}
			else if (rbnTarget1.Checked)
			{
				for (int n = 0; n < NPolygon; n++)
				{
					for (int i = 0; i < Polygon[n].nvertex; i++)
					{
						if (nearline(x, y, Polygon[n].x[i], Polygon[n].y[i], Polygon[n].x[i + 1], Polygon[n].y[i + 1])) {
							iTarget = 1;
							iDataid = n;
							if (!last) return true; ;
						}
					}
				}
			}
			else if (rbnTarget2.Checked)
			{
				for (int n = 0; n < NTx; n++)
				{
					if (nearpoint(x, y, Tx[n].pos[0], Tx[n].pos[1]))
					{
						iTarget = 2;
						iDataid = n;
						if (!last) return true; ;
					}
				}
			}
			else if (rbnTarget3.Checked)
			{
				for (int n = 0; n < NRx0d; n++)
				{
					if (nearpoint(x, y, Rx0d[n].pos[0], Rx0d[n].pos[1]))
					{
						iTarget = 3;
						iDataid = n;
						if (!last) return true; ;
					}
				}
			}
			else if (rbnTarget4.Checked)
			{
				for (int n = 0; n < NRx1d; n++)
				{
					if (nearline(x, y, Rx1d[n].pos[0, 0], Rx1d[n].pos[0, 1], Rx1d[n].pos[1, 0], Rx1d[n].pos[1, 1]))
					{
						iTarget = 4;
						iDataid = n;
						if (!last) return true; ;
					}
				}
			}
			else if (rbnTarget5.Checked)
			{
				for (int n = 0; n < NRx2d; n++)
				{
					for (int i = 0; i < 4; i++)
					{
						int ip1 = (i < 3) ? (i + 1) : 0;
						if (nearline(x, y, Rx2d[n].pos[i, 0], Rx2d[n].pos[i, 1], Rx2d[n].pos[ip1, 0], Rx2d[n].pos[ip1, 1]))
						{
							iTarget = 5;
							iDataid = n;
							if (!last) return true; ;
						}
					}
				}
			}

			return (iTarget >= 0) && (iDataid >= 0);
		}

		// 物体形状：選択対象ON/OFF
		private void rbnTarget_CheckedChanged(object sender, EventArgs e)
		{
			iTarget =
				(sender == rbnTarget0) ? 0 :
				(sender == rbnTarget1) ? 1 :
				(sender == rbnTarget2) ? 2 :
				(sender == rbnTarget3) ? 3 :
				(sender == rbnTarget4) ? 4 :
				(sender == rbnTarget5) ? 5 : -1;

			// データ番号選択は編集モードで対象が選択されたとき有効になる
			nudTarget0.Enabled = (sender == rbnTarget0) && rbnMode1.Checked;
			nudTarget1.Enabled = (sender == rbnTarget1) && rbnMode1.Checked;
			nudTarget2.Enabled = (sender == rbnTarget2) && rbnMode1.Checked;
			nudTarget3.Enabled = (sender == rbnTarget3) && rbnMode1.Checked;
			nudTarget4.Enabled = (sender == rbnTarget4) && rbnMode1.Checked;
			nudTarget5.Enabled = (sender == rbnTarget5) && rbnMode1.Checked;

			// 再表示
			picMap.Invalidate();
		}

		// 物体形状：データ番号表示
		private void chkDataNumber_Click(object sender, EventArgs e)
		{
			picMap.Invalidate();
		}

		// 物体形状：[入力]/[編集]モード変更
		private void rbnMode_CheckedChanged(object sender, EventArgs e)
		{
			// データ番号変更は編集モードのみで可能
			nudTarget0.Enabled = rbnTarget0.Checked && rbnMode1.Checked;
			nudTarget1.Enabled = rbnTarget1.Checked && rbnMode1.Checked;
			nudTarget2.Enabled = rbnTarget2.Checked && rbnMode1.Checked;
			nudTarget3.Enabled = rbnTarget3.Checked && rbnMode1.Checked;
			nudTarget4.Enabled = rbnTarget4.Checked && rbnMode1.Checked;
			nudTarget5.Enabled = rbnTarget5.Checked && rbnMode1.Checked;

			// 編集、削除は編集モードのみで可能
			btnEdit.Enabled = btnDelete.Enabled = btnEditAll.Enabled = btnDeleteAll.Enabled = rbnMode1.Checked;
		}

		// 物体形状：[データ番号]変更
		private void nudTarget_ValueChanged(object sender, EventArgs e)
		{
			iDataid = (int)((NumericUpDown)sender).Value - 1;  // =0,1,...
			picMap.Invalidate();
		}

		// 物性値：選択ON/OFF
		private void chkMaterialNo_CheckedChanged(object sender, System.EventArgs e)
		{
			int n = 0;
			for (int i = 0; i < chkMaterialNo.Length; i++)
			{
				if (sender == chkMaterialNo[i])
				{
					n = i;
					break;
				}
			}

			if (chkMaterialNo[n].Checked)
			{
				cboMaterialType[n].Enabled = true;
				txtEpsr[n].Enabled = txtSigma[n].Enabled = true;
				txtThick[n].Enabled = false;
				btnMaterialFile[n].Enabled = txtMaterialFile[n].Enabled = false;
				txtMaterialComment[n].Enabled = true;
				btnMaterialPlot[n].Enabled = true;
				// 初期化
				cboMaterialType[n].SelectedIndex = 0;
				txtEpsr[n].Text = "1";
				txtSigma[n].Text = "0";
				txtThick[n].Text = "0";
				txtMaterialFile[n].Text = string.Empty;
				txtMaterialComment[n].Text = string.Empty;
			}
			else
			{
				cboMaterialType[n].Enabled = txtEpsr[n].Enabled = txtSigma[n].Enabled = txtThick[n].Enabled =
				btnMaterialFile[n].Enabled = txtMaterialFile[n].Enabled = txtMaterialComment[n].Enabled = btnMaterialPlot[n].Enabled = false;
			}

			// 統計表示
			showNumberOfData();
		}

		// 物性値：種別選択
		private void cboMaterialType_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			int n = 0;
			for (int i = 0; i < cboMaterialType.Length; i++)
			{
				if (sender == cboMaterialType[i])
				{
					n = i;
					break;
				}
			}

			int id = cboMaterialType[n].SelectedIndex;
			txtEpsr[n].Enabled = txtSigma[n].Enabled = (id == 0) || (id == 1);
			txtThick[n].Enabled = (id == 1);
			btnMaterialFile[n].Enabled = (id == 2);
			txtMaterialFile[n].Enabled = (id == 2);
		}

		// 物性値：ファイル選択
		private void btnMaterialFile_Click(object sender, System.EventArgs e)
		{
			openFileDialog.InitialDirectory = Application.StartupPath + Path.DirectorySeparatorChar + "data" + Path.DirectorySeparatorChar + "material";
			openFileDialog.Filter = "すべてのファイル (*.*)|*.*";
			openFileDialog.FileName = String.Empty;

			if (openFileDialog.ShowDialog() == DialogResult.OK)
			{
				for (int i = 0; i < btnMaterialFile.Length; i++)
				{
					if (sender == btnMaterialFile[i])
					{
						// ファイル名表示
						txtMaterialFile[i].Text = Path.GetFileName(openFileDialog.FileName);
						break;
					}
				}
			}
		}

		// [物性値]：[plot]ボタン
		private void btnMaterialPlot_Click(object sender, System.EventArgs e)
		{
			// 行番号(=0,1,...)を取得
			int n = 0;
			for (int i = 0; i < btnMaterialPlot.Length; i++)
			{
				if (sender == btnMaterialPlot[i])
				{
					n = i;
					break;
				}
			}

			if ((cboMaterialType[n].SelectedIndex == 0) || (cboMaterialType[n].SelectedIndex == 1))
			{
				// 数値・厚さなし/あり
				NAngle = 91;
				Angle = new double[NAngle];
				Ref = new Complex[NAngle][];
				Trs = new Complex[NAngle][];

				double epsr, sigma, frequency;
				Double.TryParse(txtFrequency.Text, out frequency);
				Double.TryParse(txtEpsr[n].Text, out epsr);
				Double.TryParse(txtSigma[n].Text, out sigma);
				frequency *= 1e9;  // GHz -> Hz

				for (int i = 0; i < NAngle; i++)
				{
					Angle[i] = 90.0 * i / (NAngle - 1);
					Ref[i] = new Complex[2];
					if (cboMaterialType[n].SelectedIndex == 0)
					{
						// 厚さなし
						Utils.Fresnel(epsr, sigma, Angle[i], frequency, ref Ref[i]);
					}
					else if (cboMaterialType[n].SelectedIndex == 1)
					{
						// 厚さあり
						Trs[i] = new Complex[2];
						double thick;
						Double.TryParse(txtThick[n].Text, out thick);
						Utils.SingleLayer(epsr, sigma, thick, Angle[i], frequency, ref Ref[i], ref Trs[i]);
					}
				}
			}
			else if (cboMaterialType[n].SelectedIndex == 2)
			{
				// ファイル名
				string path = Application.StartupPath + Path.DirectorySeparatorChar + "data" + Path.DirectorySeparatorChar + "material" + Path.DirectorySeparatorChar + txtMaterialFile[n].Text;
			
				if (!File.Exists(path))
				{
					string msg = "ファイル " + path + " が存在しません";
					MessageBox.Show(msg, this.Text, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
					return;
				}

				Angle = new double[MaxMaterialAngle];
				Ref = new Complex[this.MaxMaterialAngle][];
				Trs = new Complex[this.MaxMaterialAngle][];
				Utils.readMaterialFile(path, MaxMaterialAngle, ref NAngle, ref Angle, ref Ref, ref Trs);
			}

			// 描画
			bDrawR = true;
			bDrawT = (cboMaterialType[n].SelectedIndex == 1) || (cboMaterialType[n].SelectedIndex == 2);
			picR.Invalidate();
			picT.Invalidate();
		}

		// 物性値：反射透過グラフの説明を表示する
		private void pnlPic_Paint(object sender, PaintEventArgs e)
		{
			string str1 = (sender == pnlR) ? "反射係数の入射角特性" : "透過係数の入射角特性";
			string str2 = "入射角[度]";

			try
			{
				Graphics g = e.Graphics;
				Brush brush0 = new SolidBrush(Color.Black);
				Brush brush1 = new SolidBrush(Color.Red);
				Brush brush2 = new SolidBrush(Color.Blue);
				Pen pen1 = new Pen(Color.Red, 2);
				Pen pen2 = new Pen(Color.Blue, 2);
				Pen pen3 = new Pen(Color.Red, 1);
				Pen pen4 = new Pen(Color.Blue, 1);

				float h = Font.Height;
				float x1 = picR.Left;
				float y1 = picR.Top + picR.Height;
				float x2 = picR.Left + picR.Width;
				float y2 = picR.Top;
				float x0 = (x1 + x2) / 2;
				float y0 = (y1 + y2) / 2;

				g.DrawString(str1, Font, brush0, x1 + 1.0f * h, y2 - 1.1f * h);
				g.DrawString("0", Font, brush0, x1 - 0.8f * h, y2 - 0.5f * h);
				g.DrawString("[dB]", Font, brush0, x1 - 1.8f * h, y2 + 0.5f * h);
				g.DrawString("-40", Font, brush0, x1 - 1.5f * h, y1 - 0.5f * h);
				g.DrawString("+180", Font, brush0, x2 + 0.2f * h, y2 - 0.5f * h);
				g.DrawString("[deg]", Font, brush0, x2 + 0.2f * h, y2 + 0.5f * h);
				g.DrawString("0", Font, brush0, x2 + 0.2f * h, y0 - 0.5f * h);
				g.DrawString("-180", Font, brush0, x2 + 0.2f * h, y1 - 0.5f * h);
				g.DrawString("0", Font, brush0, x1 - 0.2f * h, y1 + 0.1f * h);
				g.DrawString("90", Font, brush0, x2 - 0.5f * h, y1 + 0.1f * h);
				g.DrawString(str2, Font, brush0, x0 - (str2.Length * h * 0.25f), y1 + 0.1f * h);

				g.DrawString("dB      deg", Font, brush0, x2 - 5.0f * h, y2 - 1.9f * h);
				g.DrawString("V(P)", Font, brush1, x2 - 8.0f * h, y2 - 1.5f * h);
				g.DrawString("H(S)", Font, brush2, x2 - 8.0f * h, y2 - 0.8f * h);
				g.DrawLine(pen1, x2 - 5.5f * h, y2 - 1.0f * h, x2 - 3.5f * h, y2 - 1.0f * h);
				g.DrawLine(pen2, x2 - 5.5f * h, y2 - 0.4f * h, x2 - 3.5f * h, y2 - 0.4f * h);
				g.DrawLine(pen3, x2 - 3.0f * h, y2 - 1.0f * h, x2 - 1.0f * h, y2 - 1.0f * h);
				g.DrawLine(pen4, x2 - 3.0f * h, y2 - 0.4f * h, x2 - 1.0f * h, y2 - 0.4f * h);

				brush0.Dispose();
				brush1.Dispose();
				brush2.Dispose();
				pen1.Dispose();
				pen2.Dispose();
				pen3.Dispose();
				pen4.Dispose();
			}
			catch { }
		}

		// 物性値：反射透過係数の入射角特性をプロットする
		private void plotMaterial(object sender, PaintEventArgs e)
		{
			Graphics g = e.Graphics;

			g.Clear(((PictureBox)sender).BackColor);

			float x1 = 0;
			float y1 = 0;
			float x2 = picR.ClientSize.Width - 1;
			float y2 = picR.ClientSize.Height - 1;

			const int ydiv = 8;
			Utils.plotGrid(g, x1, y1, x2, y2, 9, ydiv);

			if (((sender == picR) && bDrawR) || ((sender == picT) && bDrawT))
			{
				const double dbmin = -40;
				const double dbmax = 0;
				const double degmin = -Math.PI;
				const double degmax = +Math.PI;

				double[] f1 = new double[NAngle];
				double[] f2 = new double[NAngle];
				double[] f3 = new double[NAngle];
				double[] f4 = new double[NAngle];

				for (int i = 0; i < NAngle; i++)
				{
					Complex v = (sender == picR) ? Ref[i][0] : Trs[i][0];
					Complex h = (sender == picR) ? Ref[i][1] : Trs[i][1];
					f1[i] = 20 * Math.Log10(v.Magnitude);
					f2[i] = 20 * Math.Log10(h.Magnitude);
					f3[i] = v.Phase;
					f4[i] = h.Phase;
				}

				Utils.plotFunc(g, NAngle, Angle, f1, dbmin, dbmax, x1, y1, x2, y2, new Pen(Color.Red, 2));
				Utils.plotFunc(g, NAngle, Angle, f2, dbmin, dbmax, x1, y1, x2, y2, new Pen(Color.Blue, 2));
				Utils.plotFunc(g, NAngle, Angle, f3, degmin, degmax, x1, y1, x2, y2, Pens.Red);
				Utils.plotFunc(g, NAngle, Angle, f4, degmin, degmax, x1, y1, x2, y2, Pens.Blue);
			}
		}

		// アンテナ：選択ON/OFF
		private void chkAntennaNo_CheckedChanged(object sender, System.EventArgs e)
		{
			// 行番号(=0,1,...)を取得
			int n = 0;
			for (int i = 0; i < chkAntennaNo.Length; i++)
			{
				if (sender == chkAntennaNo[i])
				{
					n = i;
					break;
				}
			}

			if (chkAntennaNo[n].Checked)
			{
				// ON
				cboAntennaType[n].Enabled = true;
				cboPol[n].Enabled = true;
				txtTheta[n].Enabled = txtPhi[n].Enabled = txtBw[n].Enabled = txtBwTheta[n].Enabled = txtBwPhi[n].Enabled = false;
				btnAntennaFile[n].Enabled = txtAntennaFile[n].Enabled = false;
				txtAntennaComment[n].Enabled = true;
				btnAntennaPlot[n].Enabled = true;
				// 初期化
				cboAntennaType[n].SelectedIndex = 0;
				cboPol[n].SelectedIndex = 0;
				txtTheta[n].Text = "0";
				txtPhi[n].Text = "0";
				txtBw[n].Text = "90";
				txtBwTheta[n].Text = "90";
				txtBwPhi[n].Text = "90";
				txtAntennaFile[n].Text = string.Empty;
				txtAntennaComment[n].Text = string.Empty;
			}
			else
			{
				// OFF
				cboAntennaType[n].Enabled = cboPol[n].Enabled =
				txtTheta[n].Enabled = txtPhi[n].Enabled = txtBw[n].Enabled = txtBwTheta[n].Enabled = txtBwPhi[n].Enabled =
				btnAntennaFile[n].Enabled = txtAntennaFile[n].Enabled = txtAntennaComment[n].Enabled = btnAntennaPlot[n].Enabled = false;
			}

			// 統計表示
			showNumberOfData();
		}

		// アンテナ：種別選択
		private void cboAntennaType_SelectedIndexChanged(object sender, EventArgs e)
		{
			for (int i = 0; i < cboAntennaType.Length; i++)
			{
				if (sender == cboAntennaType[i])
				{
					// id=0/1/2/3:種別
					int id = cboAntennaType[i].SelectedIndex;
					cboPol[i].Enabled = (id == 0) || (id == 1) || (id == 2);
					txtTheta[i].Enabled = txtPhi[i].Enabled = (id == 1) || (id == 2);
					txtBw[i].Enabled = (id == 1);
					txtBwTheta[i].Enabled = txtBwPhi[i].Enabled = (id == 2);
					btnAntennaFile[i].Enabled = txtAntennaFile[i].Enabled = (id == 3);
					break;
				}
			}
		}

		// アンテナ：ファイル選択
		private void btnAntennaFile_Click(object sender, EventArgs e)
		{
			openFileDialog.InitialDirectory = Application.StartupPath + Path.DirectorySeparatorChar + "data" + Path.DirectorySeparatorChar + "antenna";
			openFileDialog.Filter = "すべてのファイル (*.*)|*.*";
			openFileDialog.FileName = String.Empty;

			if (openFileDialog.ShowDialog() == DialogResult.OK)
			{
				for (int i = 0; i < btnAntennaFile.Length; i++)
				{
					if (sender == btnAntennaFile[i])
					{
						// ファイル名表示
						txtAntennaFile[i].Text = Path.GetFileName(openFileDialog.FileName);
						break;
					}
				}
			}
		}

		// アンテナ：[plot]ボタン
		private void btnAntennaPlot_Click(object sender, System.EventArgs e)
		{
			// 行番号(Iantenna=0,1,...)取得
			int n = 0;
			for (int i = 0; i < btnAntennaPlot.Length; i++)
			{
				if (sender == btnAntennaPlot[i])
				{
					n = i;
					break;
				}
			}
			iPlotAntenna = n;

			// アンテナ指向性データ作成
			if (cboAntennaType[n].SelectedIndex < 3)
			{
				// 数値
				Ntheta = (int)nudNdivAntenna.Value;
				Nphi = 2 * Ntheta;
				Rantenna = new double[Ntheta + 1, Nphi + 1];
				makeAntennaPattern();
			}
			else
			{
				// ファイル読み込み
				string path = Application.StartupPath + Path.DirectorySeparatorChar + "data" + Path.DirectorySeparatorChar + "antenna" + Path.DirectorySeparatorChar + txtAntennaFile[n].Text;
				//if (!Utils.readAantennaFile(path, ref Ntheta_file, ref Nphi_file, ref Rantenna_file)) return;
				if (!Utils.readAantennaFile(path, ref Ntheta, ref Nphi, ref Rantenna)) return;
			}

			// 描画座標作成
			Xantenna = new double[Ntheta + 1, Nphi + 1];
			Yantenna = new double[Ntheta + 1, Nphi + 1];

			// 利得表示
			double gain = Utils.gain(Ntheta, Nphi, Rantenna);
			lblAntennaGain.Text = "利得=" + gain.ToString("0.000") + "dB";

			// 描画
			picAntenna3d.Invalidate();
		}

		// アンテナ指向性データ(数値入力のみ)
		private void makeAntennaPattern()
		{
			int n = iPlotAntenna;
			int atype = cboAntennaType[n].SelectedIndex;

			if ((n >= 0) && (atype < 3))  // ファイル除外
			{
				Double.TryParse(txtTheta[n].Text,   out double atheta);
				Double.TryParse(txtPhi[n].Text,     out double aphi);
				Double.TryParse(txtBw[n].Text,      out double bw);
				Double.TryParse(txtBwTheta[n].Text, out double bwtheta);
				Double.TryParse(txtBwPhi[n].Text,   out double bwphi);
				for (int itheta = 0; itheta <= Ntheta; itheta++)
				{
					for (int iphi = 0; iphi <= Nphi; iphi++)
					{
						double theta = 180.0 * itheta / Ntheta;
						double phi   = 360.0 * iphi   / Nphi;
						Rantenna[itheta, iphi] = Utils.rantenna(theta, phi, atype, atheta, aphi, bw, bwtheta, bwphi);
					}
				}
			}
		}

		// アンテナ：視点(θ,φ)変更
		private void nudAntennaView_ValueChanged(object sender, EventArgs e)
		{
			Theta = (int)nudAntennaView1.Value;
			Phi = (int)nudAntennaView2.Value;
			picAntenna3d.Invalidate();
		}

		// アンテナ：視点[X]/[Y]/{Z]
		private void btnXYZ_Click(object sender, EventArgs e)
		{
			if (sender == btnAntennaX)
			{
				nudAntennaView1.Value = 90;
				nudAntennaView2.Value = 0;
			}
			else if (sender == btnAntennaY)
			{
				nudAntennaView1.Value = 90;
				nudAntennaView2.Value = 270;
			}
			else if (sender == btnAntennaZ) 
			{
				nudAntennaView1.Value = 0;
				nudAntennaView2.Value = 270;
			}
		}

		// アンテナ：[θ分割数]変更
		private void nudNdivAntenna_ValueChanged(object sender, EventArgs e)
		{
			int n = iPlotAntenna;
			int atype = cboAntennaType[n].SelectedIndex;

			if ((n >= 0) && (atype < 3))  // ファイルは除外
			{
				// アンテナ指向性データ作成
				Ntheta = (int)nudNdivAntenna.Value;
				Nphi = 2 * Ntheta;
				Rantenna = new double[Ntheta + 1, Nphi + 1];
				makeAntennaPattern();

				// 描画座標作成
				Xantenna = new double[Ntheta + 1, Nphi + 1];
				Yantenna = new double[Ntheta + 1, Nphi + 1];

				// 利得表示
				double gain = Utils.gain(Ntheta, Nphi, Rantenna);
				lblAntennaGain.Text = "利得=" + gain.ToString("0.000") + "dB";

				// 描画
				picAntenna3d.Invalidate();
			}
		}

		// アンテナ：マウスクリック
		private void picAntenna_MouseDown(object sender, MouseEventArgs e)
		{
			bDrag = true;
			Xold = e.X;
			Yold = e.Y;
		}

		// アンテナ：マウス移動
		private void picAntenna_MouseMove(object sender, MouseEventArgs e)
		{
			// 左ボタンドラッグ：回転
			if ((e.Button == MouseButtons.Left) && bDrag)
			{
				double f = 500.0 / (picAntenna3d.Width + picAntenna3d.Height);
				Theta -= f * (e.Y - Yold);
				Phi -= f * (e.X - Xold);
				if (Theta < 0) Theta = 0;
				if (Theta > 180) Theta = 180;
				if (Phi < 0) Phi += 360;
				if (Phi > 360) Phi -= 360;
				Xold = e.X;
				Yold = e.Y;

				// 描画
				nudAntennaView1.Value = (int)Theta;
				nudAntennaView2.Value = (int)Phi;
			}
		}

		// アンテナ：マウスアップ
		private void picAntenna_MouseUp(object sender, MouseEventArgs e)
		{
			bDrag = false;
		}

		// アンテナ：パターンプロット
		private void picAntenna3d_Paint(object sender, PaintEventArgs e)
		{
			// 描画座標計算
			makeProjection();

			Graphics g = e.Graphics;

			g.Clear(((PictureBox)sender).BackColor);

			if ((Ntheta <= 0) || (Nphi <= 0)) return;

			// φ一定線
			for (int itheta = 0; itheta < Ntheta; itheta++)
			{
				for (int iphi = 0; iphi < Nphi; iphi++)
				{
					float x1 = (float)Xantenna[itheta, iphi];
					float x2 = (float)Xantenna[itheta + 1, iphi];
					float y1 = picAntenna3d.Height - (float)Yantenna[itheta, iphi];
					float y2 = picAntenna3d.Height - (float)Yantenna[itheta + 1, iphi];
					g.DrawLine(Pens.Black, x1, y1, x2, y2);
				}
			}

			// θ一定線
			for (int itheta = 1; itheta < Ntheta; itheta++)
			{
				for (int iphi = 0; iphi < Nphi; iphi++)
				{
					float x1 = (float)Xantenna[itheta, iphi];
					float x2 = (float)Xantenna[itheta, iphi + 1];
					float y1 = picAntenna3d.Height - (float)Yantenna[itheta, iphi];
					float y2 = picAntenna3d.Height - (float)Yantenna[itheta, iphi + 1];
					g.DrawLine(Pens.Black, x1, y1, x2, y2);
				}
			}

			// XYZ軸
			for (int n = 1; n < 4; n++)
			{
				Pen pen = (n == 1) ? new Pen(Color.Red, 2) : (n == 2) ? new Pen(Color.Green, 2) : new Pen(Color.Blue, 2);
				float x1 = (float)Xaxis[0];
				float x2 = (float)Xaxis[n];
				float y1 = picAntenna3d.Height - (float)Yaxis[0];
				float y2 = picAntenna3d.Height - (float)Yaxis[n];
				g.DrawLine(pen, x1, y1, x2, y2);
			}

			// アンテナ番号
			string str = "No." + (iPlotAntenna + 1).ToString();
			g.DrawString(str, Font, Brushes.Black, 5, 5);
		}

		// アンテナ：射影座標の計算
		private void makeProjection()
		{
			if ((Ntheta <= 0) || (Nphi <= 0)) return;

			// 射影ベクトルの成分の計算
			const double dtor = Math.PI / 180;
			double sint = Math.Sin(Theta * dtor);
			double cost = Math.Cos(Theta * dtor);
			double sinp = Math.Sin(Phi * dtor);
			double cosp = Math.Cos(Phi * dtor);
			double[] xu = new double[3];
			double[] yu = new double[3];
			xu[0] = -sinp;
			xu[1] = +cosp;
			xu[2] = 0;
			yu[0] = -cost * cosp;
			yu[1] = -cost * sinp;
			yu[2] = +sint;

			// スケーリングファクター
			double fctr = 0.48 * Math.Min(picAntenna3d.Width, picAntenna3d.Height);

			// ウィンドウの中心
			double x0 = picAntenna3d.Width / 2;
			double y0 = picAntenna3d.Height / 2;

			// プロット座標の計算
			double[] pos = new double[3];
			for (int itheta = 0; itheta <= Ntheta; itheta++)
			{
				for (int iphi = 0; iphi <= Nphi; iphi++)
				{
					double theta = 180.0 * itheta / Ntheta;
					double phi = 360.0 * iphi / Nphi;
					Utils.polar2xyz(Rantenna[itheta, iphi], theta, phi, pos);
					double xprj = Utils.vector_innerprod(xu, pos);
					double yprj = Utils.vector_innerprod(yu, pos);
					Xantenna[itheta, iphi] = x0 + (fctr * xprj);
					Yantenna[itheta, iphi] = y0 + (fctr * yprj);
				}
			}

			// XYZ軸
			double[][] axis = new double[4][];
			axis[0] = new double[3] { 0, 0, 0 };
			axis[1] = new double[3] { 1.2, 0, 0 };
			axis[2] = new double[3] { 0, 1.2, 0 };
			axis[3] = new double[3] { 0, 0, 1.2 };
			for (int n = 0; n < 4; n++)
			{
				double xprj = Utils.vector_innerprod(xu, axis[n]);
				double yprj = Utils.vector_innerprod(yu, axis[n]);
				Xaxis[n] = x0 + (fctr * xprj);
				Yaxis[n] = y0 + (fctr * yprj);
			}
		}

		// Rx0d
		private void chkRx0d_profile_xscale_CheckedChanged(object sender, EventArgs e)
		{
			lblRx0d_profile_xscale_min.Enabled = lblRx0d_profile_xscale_max.Enabled = lblRx0d_profile_xscale_div.Enabled =
			txtRx0d_profile_xscale_min.Enabled = txtRx0d_profile_xscale_max.Enabled = nudRx0d_profile_xscale_div.Enabled = !((CheckBox)sender).Checked;
		}
		private void chkRx0d_profile_yscale_CheckedChanged(object sender, EventArgs e)
		{
			lblRx0d_profile_yscale_min.Enabled = lblRx0d_profile_yscale_max.Enabled = lblRx0d_profile_yscale_div.Enabled =
			txtRx0d_profile_yscale_min.Enabled = txtRx0d_profile_yscale_max.Enabled = nudRx0d_profile_yscale_div.Enabled = !((CheckBox)sender).Checked;
		}

		// Rx1d
		private void chkRx1d_power_scale_CheckedChanged(object sender, EventArgs e)
		{
			lblRx1d_power_scale_min.Enabled = lblRx1d_power_scale_max.Enabled = lblRx1d_power_scale_div.Enabled =
			txtRx1d_power_scale_min.Enabled = txtRx1d_power_scale_max.Enabled = nudRx1d_power_scale_div.Enabled = !((CheckBox)sender).Checked;
		}
		private void chkRx1d_delay_scale_CheckedChanged(object sender, EventArgs e)
		{
			lblRx1d_delay_scale_min.Enabled = lblRx1d_delay_scale_max.Enabled = lblRx1d_delay_scale_div.Enabled =
			txtRx1d_delay_scale_min.Enabled = txtRx1d_delay_scale_max.Enabled = nudRx1d_delay_scale_div.Enabled = !((CheckBox)sender).Checked;
		}

		// Rx2d
		private void chkRx2d_power_scale_CheckedChanged(object sender, EventArgs e)
		{
			lblRx2d_power_scale_min.Enabled = lblRx2d_power_scale_max.Enabled = lblRx2d_power_scale_div.Enabled =
			txtRx2d_power_scale_min.Enabled = txtRx2d_power_scale_max.Enabled = nudRx2d_power_scale_div.Enabled = !((CheckBox)sender).Checked;
		}
		private void chkRx2d_delay_scale_CheckedChanged(object sender, EventArgs e)
		{
			lblRx2d_delay_scale_min.Enabled = lblRx2d_delay_scale_max.Enabled = lblRx2d_delay_scale_div.Enabled =
			txtRx2d_delay_scale_min.Enabled = txtRx2d_delay_scale_max.Enabled = nudRx2d_delay_scale_div.Enabled = !((CheckBox)sender).Checked;
		}

		private void chkComponent1_CheckedChanged(object sender, EventArgs e)
		{
			chkComponent2a.Enabled = chkComponent2.Checked;
		}

		private void chkRx2d_decay_scale_CheckedChanged(object sender, EventArgs e)
		{
			lblRx2d_decay_scale_min.Enabled = lblRx2d_decay_scale_max.Enabled =
			txtRx2d_decay_scale_min.Enabled = txtRx2d_decay_scale_max.Enabled = !((CheckBox)sender).Checked;
		}

		// [バージョン]
		private void mnuVersion_Click(object sender, EventArgs e)
		{
			Version ver = new Version(Application.ProductVersion);
			string msg
				= Application.ProductName + "\n"
				+ "Version " + ver.Major + "." + ver.Minor + "." + ver.Build + "\n"
				+ "2023" + "\n"
				+ "http://emoss.starfree.jp/OpenRTM/";
			MessageBox.Show(msg, Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Information);
		}

		// 最小分解能で丸める
		private double number_resolution(string str)
		{
			Double.TryParse(str, out double d);
			double r = dResolution[iResolution];
			return r * Math.Floor((d / r) + 0.5);
		}

		// 点(x, y)は線分(x1, y1)-(x2, y2)に近いか
		// 物理座標
		private bool nearline(double x, double y, double x1, double y1, double x2, double y2)
		{
			double d = Math.Sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2));
			double s2 = Math.Abs((y - y1) * (x - x2) - (y - y2) * (x - x1));
			double ai = (x1 - x) * (x2 - x) + (y1 - y) * (y2 - y);
			double d1 = Math.Sqrt((x - x1) * (x - x1) + (y - y1) * (y - y1));
			double d2 = Math.Sqrt((x - x2) * (x - x2) + (y - y2) * (y - y2));

			return ((s2 * dScale < NearPixel * d) && (ai < 0)) || ((d1 * dScale) < NearPixel) || ((d2 * dScale) < NearPixel);
		}

		// 点(x, y)は点(x0, y0)に近いか
		// 物理座標
		private bool nearpoint(double x, double y, double x0, double y0)
		{
			double dx = Math.Abs(x - x0);
			double dy = Math.Abs(y - y0);

			return (dx < (PointSize / 2) / dScale) && (dy < (PointSize / 2) / dScale);
		}

		// XY座標の最小・最大
		private void getXYSpan(out double xmin, out double ymin, out double xmax, out double ymax)
		{
			xmin = ymin = +1e10;
			xmax = ymax = -1e10;

			for (int n = 0; n < NPillar; n++)
			{
				for (int i = 0; i < Pillar[n].nvertex; i++)
				{
					double x = Pillar[n].x[i];
					double y = Pillar[n].y[i];
					xmin = Math.Min(xmin, x);
					xmax = Math.Max(xmax, x);
					ymin = Math.Min(ymin, y);
					ymax = Math.Max(ymax, y);
				}
			}

			for (int n = 0; n < NPolygon; n++)
			{
				for (int i = 0; i < Polygon[n].nvertex; i++)
				{
					double x = Polygon[n].x[i];
					double y = Polygon[n].y[i];
					xmin = Math.Min(xmin, x);
					xmax = Math.Max(xmax, x);
					ymin = Math.Min(ymin, y);
					ymax = Math.Max(ymax, y);
				}
			}

			for (int n = 0; n < NTx; n++)
			{
				double x = Tx[n].pos[0];
				double y = Tx[n].pos[1];
				xmin = Math.Min(xmin, x);
				xmax = Math.Max(xmax, x);
				ymin = Math.Min(ymin, y);
				ymax = Math.Max(ymax, y);
			}

			for (int n = 0; n < NRx0d; n++)
			{
				double x = Rx0d[n].pos[0];
				double y = Rx0d[n].pos[1];
				xmin = Math.Min(xmin, x);
				xmax = Math.Max(xmax, x);
				ymin = Math.Min(ymin, y);
				ymax = Math.Max(ymax, y);
			}

			for (int n = 0; n < NRx1d; n++)
			{
				for (int i = 0; i < 2; i++)
				{
					double x = Rx1d[n].pos[i, 0];
					double y = Rx1d[n].pos[i, 1];
					xmin = Math.Min(xmin, x);
					xmax = Math.Max(xmax, x);
					ymin = Math.Min(ymin, y);
					ymax = Math.Max(ymax, y);
				}
			}

			for (int n = 0; n < NRx2d; n++)
			{
				for (int i = 0; i < 4; i++) 
				{
					double x = Rx2d[n].pos[i, 0];
					double y = Rx2d[n].pos[i, 1];
					xmin = Math.Min(xmin, x);
					xmax = Math.Max(xmax, x);
					ymin = Math.Min(ymin, y);
					ymax = Math.Max(ymax, y);
				}
			}
		}


		private void readFile_EEM(string path)
		{
			if (!File.Exists(path)) return;

			Regex regex = new Regex(@"\s+");  // 1個以上の空白文字

			using (var sr = new StreamReader(path))
			{
				try
				{
					// アンテナ(ダミーデータ)
					chkAntennaNo[0].Checked = chkAntennaNo[1].Checked = chkAntennaNo[2].Checked = true;
					cboAntennaType[0].SelectedIndex = 0;
					txtTheta[0].Text = "0";
					txtPhi[0].Text = "0";
					txtBw[0].Text = "0";
					cboAntennaType[1].SelectedIndex = 1;
					txtTheta[1].Text = "0";
					txtPhi[1].Text = "0";
					txtBw[1].Text = "90";
					cboAntennaType[2].SelectedIndex = 2;
					txtTheta[2].Text = "90";
					txtPhi[2].Text = "0";
					txtBwTheta[2].Text = "90";
					txtBwPhi[2].Text = "90";

					// 変数
					string line;
					string[] token;

					// HEADER
					if (sr.ReadLine().Trim() != "EEM-RTM")
					{
						throw new Exception("*** not EEM-RTM data.");
					}
					sr.ReadLine();
					txtTitle.Text = sr.ReadLine();
					txtFrequency.Text = (Double.Parse(sr.ReadLine()) * 1e-9).ToString();

					// MATERIAL
					if (sr.ReadLine() != "====MATERIAL====")
					{
						throw new Exception("*** invalid MATERIAL section.");
					}
					int mat = 0;
					while (true)
					{
						token = regex.Split(sr.ReadLine());
						if (token[0] == "0")
						{
							break;
						}
						else
						{
							Int32.TryParse(token[0], out int mtype);
							chkMaterialNo[mat].Checked = true;
							cboMaterialType[mat].SelectedIndex = mtype - 1;
							if (mtype == 1 && token.Length > 2)
							{
								txtEpsr[mat].Text = token[1];
								txtSigma[mat].Text = token[2];
								txtMaterialComment[mat].Text = (token.Length > 3) ? token[3].Trim('\"') : String.Empty;
							}
							else if (mtype == 2 && token.Length > 3)
							{
								txtEpsr[mat].Text = token[1];
								txtSigma[mat].Text = token[2];
								txtThick[mat].Text = token[3];
								txtMaterialComment[mat].Text = (token.Length > 4) ? token[4].Trim('\"') : String.Empty;
							}
							else if (mtype == 3)
							{
								txtMaterialFile[mat].Text = token[1];
								txtMaterialComment[mat].Text = (token.Length > 2) ? token[2].Trim('\"') : String.Empty;
							}
							mat++;
						}
					}

					// GEOMETRY
					if (sr.ReadLine() != "====GEOMETRY====")
					{
						throw new Exception("*** invalid GEOMETRY section.");
					}
					NPillar = 0;
					NPolygon = 0;
					while (true)
					{
						token = regex.Split(sr.ReadLine());
						if (token[0] == "0")
						{
							break;
						}
						else
						{
							Int32.TryParse(token[0], out int gtype);
							Int32.TryParse(token[1], out int material);
							if (gtype == 1)
							{
								// 多角柱
								token = regex.Split(sr.ReadLine());
								Int32.TryParse(token[0], out int nvertex);
								Pillar[NPillar] = new Pillar_t(nvertex);
								Pillar[NPillar].nvertex = nvertex;
								Pillar[NPillar].material = material;
								for (int iv = 0; iv < nvertex; iv++)
								{
									token = regex.Split(sr.ReadLine());
									Double.TryParse(token[0], out Pillar[NPillar].x[iv]);
									Double.TryParse(token[1], out Pillar[NPillar].y[iv]);
									if (nvertex > 2)
									{
										Pillar[NPillar].x[nvertex] = Pillar[NPillar].x[0];
										Pillar[NPillar].y[nvertex] = Pillar[NPillar].y[0];
									}
								}
								token = regex.Split(sr.ReadLine());
								Double.TryParse(token[0], out Pillar[NPillar].z[0]);
								Double.TryParse(token[1], out Pillar[NPillar].z[1]);
								Pillar[NPillar].defined = true;
								NPillar++;
							}
							else if (gtype == 2)
							{
								// 多角形(=4角形)
								int nvertex = 4;
								Polygon[NPolygon] = new Polygon_t(nvertex);
								Polygon[NPolygon].nvertex = nvertex;
								Polygon[NPolygon].material = material;
								for (int iv = 0; iv < nvertex; iv++)
								{
									token = regex.Split(sr.ReadLine());
									Double.TryParse(token[0], out Polygon[NPolygon].x[iv]);
									Double.TryParse(token[1], out Polygon[NPolygon].y[iv]);
									Double.TryParse(token[2], out Polygon[NPolygon].z[iv]);
								}
								Polygon[NPolygon].defined = true;
								NPolygon++;
							}
						}

					}

					// TX
					if (sr.ReadLine() != "====TX====")
					{
						throw new Exception("*** invalid TX section.");
					}
					NTx = 0;
					while (true)
					{
						token = regex.Split(sr.ReadLine());
						if (token[0] == "0")
						{
							break;
						}
						else
						{
							token = regex.Split(sr.ReadLine());
							Tx[NTx] = new Tx_t();
							Tx[NTx].antenna = 0;
							Double.TryParse(token[0], out Tx[NTx].pos[0]);
							Double.TryParse(token[1], out Tx[NTx].pos[1]);
							Double.TryParse(token[2], out Tx[NTx].pos[2]);
							Double.TryParse(token[3], out Tx[NTx].power);
							Double.TryParse(token[4], out Tx[NTx].phase);
							Tx[NTx].defined = true;
							sr.ReadLine();  // アンテナ:ダミー
							NTx++;
						}
					}

					// RX0
					if (sr.ReadLine() != "====RX0====")
					{
						throw new Exception("*** invalid RX0 section.");
					}
					NRx0d = 0;
					while (true)
					{
						token = regex.Split(sr.ReadLine());
						if (token[0] == "0")
						{
							break;
						}
						else
						{
							Rx0d[NRx0d] = new Rx0d_t();
							token = regex.Split(sr.ReadLine());
							Double.TryParse(token[0], out Rx0d[NRx0d].pos[0]);
							Double.TryParse(token[1], out Rx0d[NRx0d].pos[1]);
							Double.TryParse(token[2], out Rx0d[NRx0d].pos[2]);
							Rx0d[NRx0d].antenna = 0;
							Rx0d[NRx0d].defined = true;
							sr.ReadLine();  // アンテナ:ダミー
							NRx0d++;
						}
					}

					// RX1
					if (sr.ReadLine() != "====RX1====")
					{
						throw new Exception("*** invalid RX1 section.");
					}
					NRx1d = 0;
					while (true)
					{
						token = regex.Split(sr.ReadLine());
						if (token[0] == "0")
						{
							break;
						}
						else
						{
							Rx1d[NRx1d] = new Rx1d_t();
							for (int iv = 0; iv < 2; iv++)
							{
								token = regex.Split(sr.ReadLine());
								Double.TryParse(token[0], out Rx1d[NRx1d].pos[iv, 0]);
								Double.TryParse(token[1], out Rx1d[NRx1d].pos[iv, 1]);
								Double.TryParse(token[2], out Rx1d[NRx1d].pos[iv, 2]);
							}
							token = regex.Split(sr.ReadLine());
							Int32.TryParse(token[0], out Rx1d[NRx1d].div);
							Rx1d[NRx1d].antenna = 0;
							Rx1d[NRx1d].defined = true;
							sr.ReadLine();  // アンテナ:ダミー
							NRx1d++;
						}
					}

					// RX2
					if (sr.ReadLine() != "====RX2====")
					{
						throw new Exception("*** invalid RX2 section.");
					}
					NRx2d = 0;
					while (true)
					{
						token = regex.Split(sr.ReadLine());
						if (token[0] == "0")
						{
							break;
						}
						else
						{
							Rx2d[NRx2d] = new Rx2d_t();
							for (int iv = 0; iv < 4; iv++)
							{
								token = regex.Split(sr.ReadLine());
								Double.TryParse(token[0], out Rx2d[NRx2d].pos[iv, 0]);
								Double.TryParse(token[1], out Rx2d[NRx2d].pos[iv, 1]);
								Double.TryParse(token[2], out Rx2d[NRx2d].pos[iv, 2]);
							}
							token = regex.Split(sr.ReadLine());
							Int32.TryParse(token[0], out Rx2d[NRx2d].div[0]);
							Int32.TryParse(token[1], out Rx2d[NRx2d].div[1]);
							Rx2d[NRx2d].antenna = 0;
							Rx2d[NRx2d].defined = true;
							sr.ReadLine();  // アンテナ:ダミー
							NRx2d++;
						}
					}
				}
				catch (Exception ex)
				{
					MessageBox.Show(ex.Message + "\n" + ex.StackTrace, "readFile_EEM", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
				}
			}
		}
	}

	public class Pillar_t
	{
		public Pillar_t(int nvertex)
		{
			x = new double[(nvertex == 2) ? nvertex : (nvertex + 1)];
			y = new double[(nvertex == 2) ? nvertex : (nvertex + 1)];
		}
		public int nvertex;  // =2/3...
		public double[] x;
		public double[] y;
		public double[] z = new double[2];
		public int material = 1;
		public bool defined = false;
	}

	public class Polygon_t
	{
		public Polygon_t(int nvertex)
		{
			x = new double[nvertex + 1];
			y = new double[nvertex + 1];
			z = new double[nvertex + 1];
		}
		public int nvertex;  // =3/4...
		public double[] x;
		public double[] y;
		public double[] z;
		public int material = 1;
		public bool defined = false;
	}

	public class Tx_t
	{
		public double[] pos = new double[3];
		public double power = 1;
		public double phase = 0;
		public int antenna = -1;
		public bool defined = false;
	}

	public class Rx0d_t
	{
		public double[] pos = new double[3];
		public int antenna = -1;
		public bool defined = false;
	}

	public class Rx1d_t
	{
		public double[,] pos = new double[2, 3];
		public int div = 1;
		public int antenna = -1;
		public bool defined = false;
	}

	public class Rx2d_t
	{
		public double[,] pos = new double[4, 3];
		public int[] div = { 1, 1 };
		public int antenna = -1;
		public bool defined = false;
	}

}
