﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Drawing;
using System.Numerics;
using System.IO;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace OpenRTM
{
	class Utils
	{
		// Fresnel反射係数
		// 入力
		// epsr:比誘電率
		// sigma:導電率[S/m]
		// angle:入射角[度]
		// freq:周波数[Hz]
		// 出力
		// r[2]:電圧反射係数(V/H)(複素数)
		public static void Fresnel(double epsr, double sigma, double angle, double freq, ref Complex[] r)
		{
			if (freq <= 0) return;
			if ((epsr == 1) && (sigma == 0)) return;
			if (angle == 90) { r[0] = +1; r[1] = -1; return; }

			const double C = 2.99792458e8;
			const double MU0 = 4 * Math.PI * 1e-7;
			const double EPS0 = 1 / (C * C * MU0);
			const double DTOR = Math.PI / 180;

			Complex cost = Math.Cos(angle * DTOR);
			Complex sint = Math.Sin(angle * DTOR);
			Complex er = new Complex(epsr, -sigma / (2 * Math.PI * freq * EPS0));  // Er
			Complex root = Complex.Sqrt(er - (sint * sint));  // √(Er - sin2θ)

			r[0] = (root - er * cost) / (root + er * cost);   // V
			r[1] = (cost - root) / (cost + root);             // H
		}

		// 1層誘電体の反射透過係数
		// r[0/1] : 電圧反射係数(V/H)(複素数)
		// t[0/1] : 電圧透過係数(V/H)(複素数)
		public static void SingleLayer(double epsr, double sigma, double thick, double angle, double freq, ref Complex[] r, ref Complex[] t)
		{
			const double C = 2.99792458e8;
			const double MU0 = 4 * Math.PI * 1e-7;
			const double EPS0 = 1 / (C * C * MU0);
			const double DTOR = Math.PI / 180;
			double ETA0 = Math.Sqrt(MU0 / EPS0);

			double k = (2 * Math.PI * freq) / C;
			double sint0 = Math.Sin(angle * DTOR);
			double cost0 = Math.Cos(angle * DTOR);
			Complex cepsr = new Complex(epsr, -sigma / (2 * Math.PI * freq * EPS0));  // Er
			Complex ceps = EPS0 * cepsr;
			Complex cmu = MU0;
			Complex ceta = Complex.Sqrt(cmu / ceps);
			Complex sint = (sint0 / C) / Complex.Sqrt(ceps * cmu);
			Complex cost = Complex.Sqrt(Complex.One - sint * sint);
			Complex d = (C * k * thick) * Complex.Sqrt(ceps * cmu) * cost;

			for (int i = 0; i < 2; i++)
			{
				Complex a1 = (i == 0) ? (ceta * cost) : (ceta / cost);
				Complex m11 = Complex.Cos(d);
				Complex m12 = Complex.ImaginaryOne * Complex.Sin(d) * a1;
				Complex m21 = Complex.ImaginaryOne * Complex.Sin(d) / a1;
				Complex m22 = m11;

				Complex a0 = (i == 0) ? (ETA0 * cost0) : (ETA0 / cost0);
				Complex p = (a0 * m11) + m12;
				Complex q = a0 * ((a0 * m21) + m22);
				r[i] = (p - q) / (p + q);
				t[i] = (2 * a0) / (p + q);
			}
		}

		// 反射透過係数ファイルを読み込む
		public static void readMaterialFile(string path, int maxangle, ref int nangle, ref double[] angle, ref Complex[][] r, ref Complex[][] t)
		{
			using (var sr = new StreamReader(path))
			{
				try
				{
					Regex regex = new Regex(@"\s+");        // 1個以上の空白文字

					nangle = 0;
					string line;
					while ((line = sr.ReadLine()) != null)
					{
						string[] token = regex.Split(line.Trim());
						if (token.Length > 8) {
							// 反射係数
							r[nangle] = new Complex[2];
							Double.TryParse(token[0], out angle[nangle]);
							Double.TryParse(token[1], out double r_v_r);
							Double.TryParse(token[2], out double r_v_i);
							Double.TryParse(token[3], out double r_h_r);
							Double.TryParse(token[4], out double r_h_i);
							r[nangle][0] = new Complex(r_v_r, r_v_i);  // V(P)
							r[nangle][1] = new Complex(r_h_r, r_h_i);  // H(S)

							// 透過係数
							t[nangle] = new Complex[2];
							Double.TryParse(token[5], out double t_v_r);
							Double.TryParse(token[6], out double t_v_i);
							Double.TryParse(token[7], out double t_h_r);
							Double.TryParse(token[8], out double t_h_i);
							t[nangle][0] = new Complex(t_v_r, t_v_i);  // V(P)
							t[nangle][1] = new Complex(t_h_r, t_h_i);  // H(S)

							nangle++;
							if (nangle >= maxangle)
							{
								break;
							}
						}
					}
				}
				catch { }
			}
		}

		// アンテナパターンファイルを読み込む
		// 関数値=正常か
		public static bool readAantennaFile(string path, ref int ntheta, ref int nphi, ref double[,] r)
		{
			if (!File.Exists(path)) return false;

			string[] token;
			string line;
			Regex regex = new Regex(@"\s+");        // 1個以上の空白文字

			using (var sr = new StreamReader(path))
			{
				try
				{
					// コメント(2行)
					sr.ReadLine();
					sr.ReadLine();

					// 本体
					int itheta_max = 0;
					int iphi_max = 0;
					while ((line = sr.ReadLine()) != null)
					{
						token = regex.Split(line.Trim());
						itheta_max = Math.Max(itheta_max, Int32.Parse(token[0]));
						iphi_max = Math.Max(iphi_max, Int32.Parse(token[1]));
					}
					ntheta = itheta_max;
					nphi = iphi_max;
				}
				catch {
					return false;
				}
			}

			r = new double[ntheta + 1, nphi + 1];

			using (var sr = new StreamReader(path))
			{
				try
				{
					// コメント(2行)
					sr.ReadLine();
					sr.ReadLine();

					// 本体
					while ((line = sr.ReadLine()) != null)
					{
						token = regex.Split(line.Trim());
						int itheta = Int32.Parse(token[0]);
						int iphi = Int32.Parse(token[1]);
						itheta = Math.Max(0, Math.Min(ntheta, itheta));
						iphi = Math.Max(0, Math.Min(nphi, iphi));
						Double.TryParse(token[4], out double eabs_db);
						r[itheta, iphi] = Math.Pow(10, eabs_db / 20);
					}
				}
				catch
				{
					return false;
				}
			}
			/*
			// number of data
			token = regex.Split(sr.ReadLine().Trim());
			Int32.TryParse(token[0], out ntheta);
			Int32.TryParse(token[1], out nphi);
			r = new double[ntheta + 1, nphi + 1];

			// comment
			sr.ReadLine();

			// read data
			for (int itheta = 0; itheta <= ntheta; itheta++)
			{
				for (int iphi = 0; iphi <= nphi; iphi++)
				{
					token = regex.Split(sr.ReadLine().Trim());
					Double.TryParse(token[3], out double ath);
					Double.TryParse(token[5], out double aph);
					r[itheta, iphi] = Math.Sqrt(Math.Pow(10, ath / 10) + Math.Pow(10, aph / 10));
				}
			}
			*/

			// normalize
			double rmax = 0;
			for (int itheta = 0; itheta <= ntheta; itheta++)
			{
				for (int iphi = 0; iphi <= nphi; iphi++)
				{
					rmax = Math.Max(rmax, r[itheta, iphi]);
				}
			}
			if (rmax <= 0)
			{
				return false;
			}
			for (int itheta = 0; itheta <= ntheta; itheta++)
			{
				for (int iphi = 0; iphi <= nphi; iphi++)
				{
					r[itheta, iphi] /= rmax;
				}
			}

			return true;
		}

		// 指定した方向のアンテナの相対電界利得(0-1)を計算する
		public static double rantenna(double theta, double phi, int type, double atheta, double aphi, double bw, double bwtheta, double bwphi)
		{
			double r = 1;

			if (type == 0)
			{
				// isotropic
				r = 1;
			}
			else if (type == 1)
			{
				// dipole
				double[] d = new double[3];
				double[] v = new double[3];
				polar2xyz(1, atheta, aphi, d);
				polar2xyz(1, theta, phi, v);
				double cosa = Math.Max(-1, Math.Min(+1, vector_innerprod(d, v)));
				double arg = Math.Acos(cosa);  // [rad]
				arg = (90 / bw) * (Math.PI / 2 - arg);
				arg = Math.Max(-Math.PI / 2, Math.Min(Math.PI / 2, arg));  // no sidelobe
				r = Math.Cos(arg);
			}
			else if (type == 2)
			{
				// beam
				double[] d = new double[3];
				double[] v1 = new double[3];
				double[] v2 = new double[3];
				polar2xyz(1, atheta, aphi, d);
				polar2xyz(1, theta, aphi, v1);
				polar2xyz(1, atheta, phi, v2);
				double cos1 = Math.Max(-1, Math.Min(+1, vector_innerprod(d, v1)));
				double arg1 = Math.Acos(cos1);  // [rad]
				double cos2 = Math.Max(-1, Math.Min(+1, vector_innerprod(d, v2)));
				double arg2 = Math.Acos(cos2);  // [rad]
				arg1 *= 90 / bwtheta;
				arg2 *= 90 / bwphi;
				arg1 = Math.Max(0, Math.Min(Math.PI / 2, arg1));  // no sidelobe
				arg2 = Math.Max(0, Math.Min(Math.PI / 2, arg2));  // no sidelobe
				r = Math.Cos(arg1) * Math.Cos(arg2);
			}
			/*
			else if (type == 3)
			{
				// file : interpolation
				double dtheta = 180.0 / ntheta_file;
				double dphi = 360.0 / nphi_file;
				if (phi < 0) phi += 360;   // -180...+180 -> 0...360
				theta = Math.Max(0, Math.Min(180, theta));
				phi = Math.Max(0, Math.Min(360, phi));
				int itheta = (int)(theta / dtheta);
				int iphi = (int)(phi / dphi);
				itheta = Math.Max(0, Math.Min(ntheta_file - 1, itheta));
				iphi = Math.Max(0, Math.Min(nphi_file - 1, iphi));

				double ftheta = (theta / dtheta) - itheta;
				double fphi = (phi / dphi) - iphi;
				r = (1 - ftheta) * (1 - fphi) * rantenna_file[itheta + 0, iphi + 0]
				  +       ftheta * (1 - fphi) * rantenna_file[itheta + 1, iphi + 0]
				  + (1 - ftheta) *      fphi  * rantenna_file[itheta + 0, iphi + 1]
				  +       ftheta *      fphi  * rantenna_file[itheta + 1, iphi + 1];
			}
			*/

			return r;
		}

		// アンテナ利得[dB]
		public static double gain(int ntheta, int nphi, double[,] r)
		{
			double gsum = 0;
			for (int itheta = 1; itheta < ntheta; itheta++)
			{
				for (int iphi = 0; iphi < nphi; iphi++)
				{
					double theta = Math.PI * itheta / ntheta;
					gsum += r[itheta, iphi] * r[itheta, iphi] * Math.Sin(theta);

				}
			}
			double gain = (4 * Math.PI) / (gsum * (Math.PI / ntheta) * (2 * Math.PI / nphi));
			return 10 * Math.Log10(gain);

		}

		// polar to XYZ
		// theta, phi [deg]
		public static void polar2xyz(double r, double theta, double phi, double[] pos)
		{
			const double dtor = Math.PI / 180;
			pos[0] = r * Math.Sin(theta * dtor) * Math.Cos(phi * dtor);
			pos[1] = r * Math.Sin(theta * dtor) * Math.Sin(phi * dtor);
			pos[2] = r * Math.Cos(theta * dtor);
		}

		// inner product
		public static double vector_innerprod(double[] v1, double[] v2)
		{
			return
			(v1[0] * v2[0]) +
			(v1[1] * v2[1]) +
			(v1[2] * v2[2]);
		}

		// グリッド線を描く
		public static void plotGrid(Graphics g, float x1, float y1, float x2, float y2, int xdiv, int ydiv)
		{
			for (int i = 1; i < xdiv; i++)
			{
				float x = x1 + (x2 - x1) * i / xdiv;
				g.DrawLine(Pens.LightGray, x, y1, x, y2);
			}

			for (int j = 1; j < ydiv; j++)
			{
				float y = y1 + (y2 - y1) * j / ydiv;
				g.DrawLine(Pens.LightGray, x1, y, x2, y);
			}
		}

		// 関数を描く
		public static void plotFunc(Graphics g, int ndata, double[] x, double[] y, double ymin, double ymax, float x1, float y1, float x2, float y2, Pen pen)
		{
			for (int i = 0; i < ndata - 1; i++)
			{
				double xa = x1 + (x2 - x1) * (x[i + 0] - x[0]) / (x[ndata - 1] - x[0]);
				double xb = x1 + (x2 - x1) * (x[i + 1] - x[0]) / (x[ndata - 1] - x[0]);
				double ya = y2 - (y2 - y1) * (y[i + 0] - ymin) / (ymax - ymin);
				double yb = y2 - (y2 - y1) * (y[i + 1] - ymin) / (ymax - ymin);
				ya = Math.Min(y2, Math.Max(ya, y1));
				yb = Math.Min(y2, Math.Max(yb, y1));
				g.DrawLine(pen, (float)xa, (float)ya, (float)xb, (float)yb);
			}
		}
	}
}
