﻿// GML2ort
using System;
using System.Drawing;
using System.Windows.Forms;
using System.IO;
using System.Diagnostics;


namespace GML2ort
{
	public partial class GML2ort : Form
	{
		// 環境設定ファイル
		const string FileIni = "GML2ort.ini";

		// パラメーター(環境設定ファイルに保存する、オプションメニューで変更可)
		int DemMeshID = 2;                     // 地面間引き番号=0...3
		int[] Mesh = { 1, 3, 5, 15 };          // 地面間引き数(LATDIVとLONDIVの公約数)
		double[] BldC = { 3, 0.05 };           // 建物高さパラメーター
		byte[] BldColor = { 0, 0, 255 };       // 建物の色
		byte[] DemColor = { 180, 180, 120 };   // 地面の色
		byte[] ExtColor = { 200, 200, 200 };   // 非選択色(固定)
		bool bLatLon0 = true;                 // 座標原点を指定するか
		double Lat0 = 35;                     // 座標原点の緯度[度]
		double Lon0 = 135;                    // 座標原点の経度[度]
		double Learth = 4e7;                  // 地球の周囲[m]
		double BldNumberFactor = 5;            // 建物のデータ番号を表示する閾値
		bool bDemRegex = false;               // XMLのトークン分解にRegexを用いるか(PLATEAUのDEMのみ)

		// GSI-DEMデータ
		int GSIDemMesh = 3;                   // 2/3 : 2次/3次メッシュ
		int LatDiv = 150;                    // 緯度方向分割数
		int LonDiv = 225;                    // 経度方向分割数
		double LatLng = 1.0 / 120;           // 緯度範囲[度]
		double LonLng = 1.0 / 80;            // 経度範囲[度]
		readonly double GSIDEMSEA = -9999;   // 海の標高

		// 建物データ(PLATEAU)
		int NBldPLA = 0;
		bldPLA_t[] BldPLA;

		// 建物データ(GSI)
		int NBldGSI = 0;
		bldGSI_t[] BldGSI;

		// 標高データ(PLATEAU)
		int NDemTriPLA = 0;
		double[][,] DemTriPLA;

		// 標高データ(GSI)
		int NDemFileGSI = 0;             // DEMファイル数
		demfile_t[] DemFileGSI;          // DEMファイルの元データ

		// 標高データ(PLATEAU/GSI共通、全ファイル合成)
		dem_t[,] Dem;                   // DEMデータ
		int NDemLat = 0, NDemLon = 0;    // 緯度経度の分割数
		double DemLatMin, DemLatMax, DemLonMin, DemLonMax;  // 緯度経度範囲[度]

		// 3D描画
		//int Width3d = 750;
		//int Height3d = 500;
		//int Theta3d = 45;
		//int Phi3d = 270;

		// GUI状態変数
		bool bCut = false;
		bool bDrag = false;
		double Cx = 0, Cy = 0;
		double dFactor = 1;
		double Xold = 0, Yold = 0;
		double Xcut1 = 0, Xcut2 = 0, Ycut1 = 0, Ycut2 = 0;
		int iEdit = -1;

		// OpenRTM用(観測面データ)
		bool bRx2d = false;
		double hRx2d = 1.5;
		int dRx2d = 1;

		// 定数
		readonly double EPS = 1e-4;

		/// <summary>
		/// アプリケーションのメイン エントリ ポイントです。
		/// </summary>
		[STAThread]
		static void Main()
		{
			Application.EnableVisualStyles();
			Application.SetCompatibleTextRenderingDefault(false);
			Application.Run(new GML2ort());
		}

		public GML2ort()
		{
			InitializeComponent();

			// 環境設定ファイル入力
			if (File.Exists(FileIni))
			{
				ReadIni();
			}
		}

		private void mnuFileExit_Click(object sender, EventArgs e)
		{
			this.Close();
		}
		private void GML2ort_FormClosing(object sender, FormClosingEventArgs e)
		{
			// 最小化復元
			if (this.WindowState == FormWindowState.Minimized) this.WindowState = FormWindowState.Normal;

			// 環境設定ファイル出力
			WriteIni();
		}
		private void GML2ort_Resize(object sender, EventArgs e)
		{
			// レイアウト
			int s = 0;
			picGML.Left = 0 + s;
			picGML.Width = this.ClientSize.Width - 2 * s;
			picGML.Top = menuStrip.Height + toolStrip.Height + s;
			picGML.Height = this.ClientSize.Height - toolStrip.Height - statusStrip.Height - 30 - 2 * s;

			// スケール
			GetWinScale();

			// 描画
			picGML.Invalidate();
		}

		// GMLファイルを開く
		private void mnuOpen_Click(object sender, EventArgs e)
		{
			openFileDialog.Filter = "GML/XML files (*.gml *.xml)|*.gml; *.xml|all files (*.*)|*.*";
			openFileDialog.Multiselect = true;
			openFileDialog.FileName = String.Empty;

			if (openFileDialog.ShowDialog() == DialogResult.OK)
			{
				OpenGML(openFileDialog.FileNames);
			}
		}

		// OpenRTMファイル出力
		private void mnuSaveAs_Click(object sender, EventArgs e)
		{
			saveFileDialog.Filter = "OpenRTMデータ (*.ort)|*.ort";
			saveFileDialog.FileName = String.Empty;

			if (saveFileDialog.ShowDialog() == DialogResult.OK)
			{
				string path = saveFileDialog.FileName;
				WriteOpenRTM(path);
			}
		}

		// ファイルを読み込む
		// データを変換する
		// 描画する
		private void OpenGML(string[] fn)
		{
			// カーソル：砂時計
			this.Cursor = Cursors.WaitCursor;

			// GMLファイルを読み込む
			foreach (string path in fn)
			{
				// 拡張子=.xml : GSI
				if (Path.GetExtension(path).ToLower() == ".xml")
				{
					if (path.ToLower().Contains("bldl"))
					{
						// 建物
						GSI.ReadBld(path, ref NBldGSI, ref BldGSI);
					}
					else
					{
						// DEM
						if ((path.ToLower().Contains("dem10") && (GSIDemMesh == 2)) ||
							(path.ToLower().Contains("dem5") && (GSIDemMesh == 3)))
						{
							GSI.ReadDem(path, ref NDemFileGSI, ref DemFileGSI, LatDiv, LonDiv);
						}
						else
						{
							string msg = "ファイル名とDEMメッシュが合っていません。\n[オプション]メニューを確認してください。";
							MessageBox.Show(msg, "GSI DEM", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
							this.Cursor = Cursors.Default;
							return;
						}
					}
				}
				// 拡張子=.xml : PLATEAU
				else if (Path.GetExtension(path).ToLower() == ".gml")
				{
					PLATEAU.ReadGML(path, ref NBldPLA, ref BldPLA, ref NDemTriPLA, ref DemTriPLA, bDemRegex);
				}
			}

			// DEMデータを一つにまとめる
			if (NDemFileGSI > 0)
			{
				SetupDemGSI();
			}
			if (NDemTriPLA > 0)
			{
				SetupDemPLA();
			}

			// 部分選択解除
			ResetCut();

			// データ数表示
			ShowNumberOfData();

			// 緯度経度の中心を計算する
			if (bLatLon0)
			{
				GetLatLon0();
			}

			// 建物・地面の緯度・経度・標高->XYZ座標
			CalcXYZ();

			// スケールを計算する：XY座標をウィンドウ座標に変換する係数
			GetWinScale();

			// 建物の高さを計算する(GSI)
			if (NBldGSI > 0)
			{
				CalcBldGSIHeight();
			}

			// 描画
			picGML.Invalidate();

			// カーソル：標準
			this.Cursor = Cursors.Default;
		}

		// 全体のDEMデータを作成する(PLATEAU)
		private void SetupDemPLA()
		{
			// PLATEAUの三角形の集合は扱いにくいのでGSIと同じ矩形の2D配列に変換する

			// 全体の緯度経度の範囲
			DemLatMin =
			DemLonMin = +1e10;
			DemLatMax =
			DemLonMax = -1e10;
			for (int n = 0; n < NDemTriPLA; n++)
			{
				for (int iv = 0; iv < 3; iv++)
				{
					double lat = DemTriPLA[n][iv, 0];
					double lon = DemTriPLA[n][iv, 1];
					DemLatMin = Math.Min(DemLatMin, lat);
					DemLatMax = Math.Max(DemLatMax, lat);
					DemLonMin = Math.Min(DemLonMin, lon);
					DemLonMax = Math.Max(DemLonMax, lon);

				}
			}

			// 全体の緯度方向と経度方向のセル数
			double dlat = LatLng / LatDiv;
			double dlon = LonLng / LonDiv;
			NDemLat = (int)((DemLatMax - DemLatMin) / dlat + 0.5);
			NDemLon = (int)((DemLonMax - DemLonMin) / dlon + 0.5);

			
			// 各3角形を囲む範囲の数を+1し、標高を加算する
			double[,] hsum = new double[NDemLat + 1, NDemLon + 1];
			int[,] hnum = new int[NDemLat + 1, NDemLon + 1];
			for (int n = 0; n < NDemTriPLA; n++)
			{
				// 三角形の範囲
				double latmin = Math.Min(DemTriPLA[n][0, 0], Math.Min(DemTriPLA[n][1, 0], DemTriPLA[n][2, 0]));
				double latmax = Math.Max(DemTriPLA[n][0, 0], Math.Max(DemTriPLA[n][1, 0], DemTriPLA[n][2, 0]));
				double lonmin = Math.Min(DemTriPLA[n][0, 1], Math.Min(DemTriPLA[n][1, 1], DemTriPLA[n][2, 1]));
				double lonmax = Math.Max(DemTriPLA[n][0, 1], Math.Max(DemTriPLA[n][1, 1], DemTriPLA[n][2, 1]));
				// 三角形を囲む番号
				int ilatmin = (int)((latmin - DemLatMin) / dlat + EPS);
				int ilatmax = (int)((latmax - DemLatMin) / dlat - EPS) + 1;
				int ilonmin = (int)((lonmin - DemLonMin) / dlon + EPS);
				int ilonmax = (int)((lonmax - DemLonMin) / dlon - EPS) + 1;
				ilatmin = Math.Max(0, Math.Min(NDemLat, ilatmin));
				ilatmax = Math.Max(0, Math.Min(NDemLat, ilatmax));
				ilonmin = Math.Max(0, Math.Min(NDemLon, ilonmin));
				ilonmax = Math.Max(0, Math.Min(NDemLon, ilonmax));
				// 三角形の平均標高
				double hgt = (DemTriPLA[n][0, 2] + DemTriPLA[n][1, 2] + DemTriPLA[n][2, 2]) / 3;
				// 集計
				for (int ilat = ilatmin; ilat <= ilatmax; ilat++)
				{
					for (int ilon = ilonmin; ilon <= ilonmax; ilon++)
					{
						hsum[ilat, ilon] += hgt;
						hnum[ilat, ilon]++;
					}
				}
			}

			// 地面標高配列の作成
			// 節点の緯度経度[度]、標高[m]
			// 標高 = 標高の和 / 三角形の数
			// データがないところは海面(=0m)
			int num = 0;
			Dem = new dem_t[NDemLat + 1, NDemLon + 1];
			for (int ilat = 0; ilat <= NDemLat; ilat++)
			{
				for (int ilon = 0; ilon <= NDemLon; ilon++)
				{
					Dem[ilat, ilon].sea = (hnum[ilat, ilon] == 0);
					Dem[ilat, ilon].hgt = (hnum[ilat, ilon] > 0) ? (hsum[ilat, ilon] / hnum[ilat, ilon]) : 0;
					Dem[ilat, ilon].lat = DemLatMin + (ilat * dlat);
					Dem[ilat, ilon].lon = DemLonMin + (ilon * dlon);
					Dem[ilat, ilon].exclude = false;
					num += (hnum[ilat, ilon] > 0) ? 1 : 0;
				}
			}
			Console.WriteLine("{0} {1}", num, (NDemLat + 1) * (NDemLon + 1));
		}

		// 全体のDEMデータを作成する(GSI)
		private void SetupDemGSI()
		{
			// 全体の緯度経度の範囲
			DemLatMin =
			DemLonMin = +1e10;
			DemLatMax =
			DemLonMax = -1e10;
			for (int n = 0; n < NDemFileGSI; n++)
			{
				DemLatMin = Math.Min(DemLatMin, DemFileGSI[n].latmin);
				DemLatMax = Math.Max(DemLatMax, DemFileGSI[n].latmax);
				DemLonMin = Math.Min(DemLonMin, DemFileGSI[n].lonmin);
				DemLonMax = Math.Max(DemLonMax, DemFileGSI[n].lonmax);
			}

			// 全体の緯度経度方向のセル数
			double dlat = LatLng / LatDiv;
			double dlon = LonLng / LonDiv;
			NDemLat = (int)((DemLatMax - DemLatMin) / dlat + 0.5);
			NDemLon = (int)((DemLonMax - DemLonMin) / dlon + 0.5);

			// 地面配列の作成
			// 節点の緯度経度[度]、標高[m]
			Dem = new dem_t[NDemLat + 1, NDemLon + 1];
			for (int ilat = 0; ilat <= NDemLat; ilat++)
			{
				for (int ilon = 0; ilon <= NDemLon; ilon++)
				{
					//Dem[ilat, ilon].hgt = HSEA;  // 海で初期化
					Dem[ilat, ilon].lat = DemLatMin + (ilat * dlat);
					Dem[ilat, ilon].lon = DemLonMin + (ilon * dlon);
					Dem[ilat, ilon].sea = false;
					Dem[ilat, ilon].exclude = false;
				}
			}

			// 標高の計算
			// 全体の作業配列(セル中心):hc
			double[,] hc = new double[NDemLat, NDemLon];
			// 標高を海で初期化
			for (int ilat = 0; ilat < NDemLat; ilat++)
			{
				for (int ilon = 0; ilon < NDemLon; ilon++)
				{
					hc[ilat, ilon] = GSIDEMSEA;
				}
			}
			// 各ファイルの標高を全体の標高に代入する
			for (int n = 0; n < NDemFileGSI; n++)
			{
				// 各ファイルの緯度経度の始点番号
				int ilatmin = (int)((DemFileGSI[n].latmin - DemLatMin) / dlat + 0.5);
				int ilonmin = (int)((DemFileGSI[n].lonmin - DemLonMin) / dlon + 0.5);
				for (int ilat = 0; ilat < LatDiv; ilat++)
				{
					for (int ilon = 0; ilon < LonDiv; ilon++)
					{
						hc[ilatmin + ilat, ilonmin + ilon] = DemFileGSI[n].hgt[ilat, ilon];
						/*if (DemFileGSI[n].hgt[ilat, ilon] < -1000)
						{
							Console.WriteLine(ilat + " " + ilon + " " + DemFileGSI[n].hgt[ilat, ilon]);
						}*/
					}
				}
			}

			// セル中心データを節点データに変換する
			for (int ilat = 0; ilat <= NDemLat; ilat++)
			{
				for (int ilon = 0; ilon <= NDemLon; ilon++)
				{
					int ilat0 = (ilat > 0) ? (ilat - 1) : ilat;
					int ilat1 = (ilat < NDemLat) ? ilat : (ilat - 1);
					int ilon0 = (ilon > 0) ? (ilon - 1) : ilon;
					int ilon1 = (ilon < NDemLon) ? ilon : (ilon - 1);
					double h00 = hc[ilat0, ilon0];
					double h01 = hc[ilat0, ilon1];
					double h10 = hc[ilat1, ilon0];
					double h11 = hc[ilat1, ilon1];
					Dem[ilat, ilon].hgt = (h00 + h01 + h10 + h11) / 4;
					Dem[ilat, ilon].sea = (Dem[ilat, ilon].hgt < GSIDEMSEA / 16);
				}
			}
		}

		// データ数表示
		private void ShowNumberOfData()
		{
			// 建物数(選択/全体)
			int nbld = 0;
			int nbld_in = 0;
			for (int n = 0; n < NBldPLA; n++)
			{
				nbld++;
				nbld_in += !BldPLA[n].exclude ? 1 : 0;
			}
			for (int n = 0; n < NBldGSI; n++)
			{
				nbld++;
				nbld_in += !BldGSI[n].exclude ? 1 : 0;
			}

			// 地面数(選択/全体)
			int m = Mesh[DemMeshID];
			int ndem = 0;
			int ndem_in = 0;
			for (int ilat = 0; ilat <= NDemLat - m; ilat += m)
			{
				for (int ilon = 0; ilon <= NDemLon - m; ilon += m)
				{
					bool sea = Dem[ilat + 0, ilon + 0].sea
							|| Dem[ilat + m, ilon + 0].sea
							|| Dem[ilat + 0, ilon + m].sea
							|| Dem[ilat + m, ilon + m].sea;
					bool exclude = Dem[ilat + 0, ilon + 0].exclude
					            || Dem[ilat + m, ilon + 0].exclude
					            || Dem[ilat + 0, ilon + m].exclude
					            || Dem[ilat + m, ilon + m].exclude;
					ndem += !sea ? 1 : 0;  // 4隅すべてが海でない
					ndem_in += (!sea && !exclude) ? 1 : 0;  // 4隅すべてが海でなく選択されている
				}
			}

			// 表示
			tssBld.Text = "建物数=" + nbld_in.ToString() + "/" + nbld.ToString();
			tssDem.Text = "地面数=" + ndem_in.ToString() + "/" + ndem.ToString();
		}

		// 緯度・経度・標高をXYZ座標に変換する
		private void CalcXYZ()
		{
			// 建物(PLATEAU)
			for (int n = 0; n < NBldPLA; n++)
			{
				// 建物ポリゴン頂点のXYZ座標
				int nvtx = BldPLA[n].nvtx;
				BldPLA[n].x = new double[nvtx];
				BldPLA[n].y = new double[nvtx];
				BldPLA[n].z = new double[nvtx];
				for (int iv = 0; iv < nvtx; iv++)
				{
					geopos2xyz(BldPLA[n].lat[iv], BldPLA[n].lon[iv], BldPLA[n].hgt[iv],
						out BldPLA[n].x[iv], out BldPLA[n].y[iv], out BldPLA[n].z[iv]);
				}
			}

			// 建物(GSI)
			for (int n = 0; n < NBldGSI; n++)
			{
				// 頂点数
				int nvtx = BldGSI[n].nvtx;

				// 建物の緯度経度の中心
				double latsum = 0;
				double lonsum = 0;
				for (int iv = 0; iv < nvtx; iv++)
				{
					latsum += BldGSI[n].lat[iv];
					lonsum += BldGSI[n].lon[iv];
				}
				double lat0 = latsum / nvtx;
				double lon0 = lonsum / nvtx;

				// 建物床の標高
				BldGSI[n].hgt = Elevation(lat0, lon0);

				// 建物床のZ座標
				geopos2xyz(lat0, lon0, BldGSI[n].hgt,
					out double x, out double y, out BldGSI[n].z);

				// 建物断面頂点のXY座標
				BldGSI[n].x = new double[nvtx];
				BldGSI[n].y = new double[nvtx];
				for (int v = 0; v < nvtx; v++)
				{
					geopos2xyz(BldGSI[n].lat[v], BldGSI[n].lon[v], 0,
						out BldGSI[n].x[v], out BldGSI[n].y[v], out double z);
				}
			}

			// 地面(PLATEAU/GSI共通)
			if (NDemLat + NDemLon > 0)
			{
				for (int ilat = 0; ilat <= NDemLat; ilat++)
				{
					for (int ilon = 0; ilon <= NDemLon; ilon++)
					{
						geopos2xyz(Dem[ilat, ilon].lat, Dem[ilat, ilon].lon, Dem[ilat, ilon].hgt,
							out Dem[ilat, ilon].x, out Dem[ilat, ilon].y, out Dem[ilat, ilon].z);
					}
				}
			}
		}

		// 指定した緯度経度[度]の標高[m]
		// 標高データがないときは0
		private double Elevation(double lat, double lon)
		{
			// 一番近い節点
			double dlat = LatLng / LatDiv;
			double dlon = LonLng / LonDiv;
			int ilat = (int)((lat - DemLatMin) / dlat + 0.5);
			int ilon = (int)((lon - DemLonMin) / dlon + 0.5);

			// 標高データの範囲内にあり海でないとき
			return (ilat >= 0) && (ilat < NDemLat) &&
			       (ilon >= 0) && (ilon < NDemLon) &&
			       !Dem[ilat, ilon].sea ? Dem[ilat, ilon].hgt : 0;
		}

		// 建物の高さを計算する(GSIのみ)
		private void CalcBldGSIHeight()
		{
			for (int n = 0; n < NBldGSI; n++)
			{
				// ユーザーが編集していない建物のみ
				if (!BldGSI[n].btall)
				{
					BldGSI[n].tall = BldC[0] + BldC[1] * utils.area(BldGSI[n].nvtx, BldGSI[n].x, BldGSI[n].y);
				}
			}
		}

		// 描画
		private void picGML_Paint(object sender, PaintEventArgs e)
		{
			picGML_Paint(e.Graphics);
		}
		private void picGML_Paint(Graphics g)
		{
			if ((NDemLat <= 0) && (NDemLon <= 0) && (NBldPLA <= 0) && (NBldGSI <= 0)) return;

			float w = picGML.Width;
			float h = picGML.Height;
			float hfont = Font.Height;

			// 色
			Pen bldpen = new Pen(Color.FromArgb(BldColor[0], BldColor[1], BldColor[2]));
			Pen dempen = new Pen(Color.FromArgb(DemColor[0], DemColor[1], DemColor[2]));
			Pen extpen = new Pen(Color.FromArgb(ExtColor[0], ExtColor[1], ExtColor[2]));

			// 地面(PLATEAU/GSI共通)、4角形
			// 地面を建物より先に描画する:建物を優先するため
			if (NDemLat + NDemLon > 0)
			{
				int m = Mesh[DemMeshID];
				PointF[] dpoint = new PointF[4];

				for (int ilat = 0; ilat <= NDemLat - m; ilat += m)
				{
					for (int ilon = 0; ilon <= NDemLon - m; ilon += m)
					{
						xy2winpos(Dem[ilat + 0, ilon + 0].x, Dem[ilat + 0, ilon + 0].y, out float x0, out float y0);
						xy2winpos(Dem[ilat + m, ilon + 0].x, Dem[ilat + m, ilon + 0].y, out float x1, out float y1);
						xy2winpos(Dem[ilat + m, ilon + m].x, Dem[ilat + m, ilon + m].y, out float x2, out float y2);
						xy2winpos(Dem[ilat + 0, ilon + m].x, Dem[ilat + 0, ilon + m].y, out float x3, out float y3);

						bool sea = Dem[ilat + 0, ilon + 0].sea ||
								   Dem[ilat + m, ilon + 0].sea ||
								   Dem[ilat + m, ilon + m].sea ||
								   Dem[ilat + 0, ilon + m].sea;

						// 海でなく画面内にあるセルを表示する
						//if (!sea && (x[0] > 0) && (x[1] < w) && (y[0] > 0) && (y[2] < h))
						if (!sea && (x0 > 0) && (x1 < w) && (y0 > 0) && (y2 < h))
						{
							// 4隅のウィンドウ座標
							dpoint[0].X = x0;
							dpoint[1].X = x1;
							dpoint[2].X = x2;
							dpoint[3].X = x3;
							dpoint[0].Y = y0;
							dpoint[1].Y = y1;
							dpoint[2].Y = y2;
							dpoint[3].Y = y3;
							// 4角形表示、選択/非選択で色を変える
							g.DrawPolygon(select_dem(ilat, ilon, m) ? dempen : extpen, dpoint);
						}
					}
				}

				// DEMファイル周囲線(GSI)
				// DEMファイルに関するループ
				for (int n = 0; n < NDemFileGSI; n++)
				{
					// XY座標
					double z;
					geopos2xyz(DemFileGSI[n].latmin, DemFileGSI[n].lonmin, 0, out double x0, out double y0, out z);
					geopos2xyz(DemFileGSI[n].latmax, DemFileGSI[n].lonmin, 0, out double x1, out double y1, out z);
					geopos2xyz(DemFileGSI[n].latmax, DemFileGSI[n].lonmax, 0, out double x2, out double y2, out z);
					geopos2xyz(DemFileGSI[n].latmin, DemFileGSI[n].lonmax, 0, out double x3, out double y3, out z);
					// 描画座標
					float px, py;
					xy2winpos(x0, y0, out px, out py); dpoint[0].X = px; dpoint[0].Y = py;
					xy2winpos(x1, y1, out px, out py); dpoint[1].X = px; dpoint[1].Y = py;
					xy2winpos(x2, y2, out px, out py); dpoint[2].X = px; dpoint[2].Y = py;
					xy2winpos(x3, y3, out px, out py); dpoint[3].X = px; dpoint[3].Y = py;
					// 外枠
					g.DrawPolygon(Pens.Gray, dpoint);
					// DEM番号
					dpoint[0].Y -= 0.8F * Font.Height;
					g.DrawString(DemFileGSI[n].meshname, Font, Brushes.Magenta, dpoint[0]);
				}
			}

			// 建物(PLATEAU)、ポリゴン
			for (int n = 0; n < NBldPLA; n++)
			{
				int nvtx = BldPLA[n].nvtx;
				PointF[] bldpoint = new PointF[nvtx];

				// 頂点の描画座標
				bool inside = true;
				for (int v = 0; v < nvtx; v++)
				{
					//float x = (float)(+dFactor * BldPLA[n].x[v] + Cx);
					//float y = (float)(-dFactor * BldPLA[n].y[v] + Cy);
					xy2winpos(BldPLA[n].x[v], BldPLA[n].y[v], out float x, out float y);
					// 1点でも画面外なら表示しない
					if ((x < 0) || (x > w) || (y < 0) || (y > h))
					{
						inside = false;
						break;
					}
					bldpoint[v].X = x;
					bldpoint[v].Y = y;
				}

				// 画面内のみ表示
				if (inside)
				{
					// ポリゴン
					g.DrawPolygon((BldPLA[n].exclude ? extpen : bldpen), bldpoint);

					// 属性(拡大時のみ、ポリゴン番号と頂点数)
					if (dFactor > BldNumberFactor)
					{
						string str = (n + 1).ToString() + "(" + nvtx.ToString() + ")";
						g.DrawString(str, Font, Brushes.Black, bldpoint[0].X - 0.3f * hfont, bldpoint[0].Y - 0.5f * hfont);
					}
				}
			}

			// 建物(GSI)、ポリゴン
			for (int n = 0; n < NBldGSI; n++)
			{
				int nvtx = BldGSI[n].nvtx;
				PointF[] bpoint = new PointF[nvtx];

				// 頂点の描画座標
				bool inside = true;
				for (int iv = 0; iv < nvtx; iv++)
				{
					//float x = (float)(+dFactor * BldGSI[n].x[iv] + Cx);
					//float y = (float)(-dFactor * BldGSI[n].y[iv] + Cy);
					xy2winpos(BldGSI[n].x[iv], BldGSI[n].y[iv], out float x, out float y);
					if ((x < 0) || (x > w) || (y < 0) || (y > h))
					{
						inside = false;
						break;
					}
					bpoint[iv].X = x;
					bpoint[iv].Y = y;
				}

				// 画面内のみ表示
				if (inside)
				{
					// ポリゴン
					g.DrawPolygon((BldGSI[n].exclude ? extpen : bldpen), bpoint);

					// 属性(拡大時のみ、建物番号と頂点数)
					if (dFactor > BldNumberFactor)
					{
						string str = (n + 1).ToString() + "(" + nvtx.ToString() + ")";
						g.DrawString(str, Font, Brushes.Black, bpoint[0].X - 0.3f * hfont, bpoint[0].Y - 0.5f * hfont);
					}
				}
			}

			// 切り取り時の矩形
			if (bCut && ((NBldPLA > 0) || (NBldGSI > 0) || (NDemLat + NDemLon > 0)))
			{
				// XY座標→描画座標
				xy2winpos(Xcut1, Ycut1, out float x1, out float y1);
				xy2winpos(Xcut2, Ycut2, out float x2, out float y2);
				// 矩形描画
				g.DrawRectangle(Pens.Magenta, Math.Min(x1, x2), Math.Min(y1, y2), Math.Abs(x1 - x2), Math.Abs(y1 - y2));
				
			}
		}

		// 緯度・経度中心を取得
		private void GetLatLon0()
		{
			// 最小・最大
			double latmin = +1e10;
			double latmax = -1e10;
			double lonmin = +1e10;
			double lonmax = -1e10;

			// 建物
			for (int n = 0; n < NBldPLA; n++)
			{
				// PLATEAU
				int nvtx = BldPLA[n].nvtx;
				for (int v = 0; v < nvtx; v++)
				{
					latmin = Math.Min(latmin, BldPLA[n].lat[v]);
					latmax = Math.Max(latmax, BldPLA[n].lat[v]);
					lonmin = Math.Min(lonmin, BldPLA[n].lon[v]);
					lonmax = Math.Max(lonmax, BldPLA[n].lon[v]);
				}
			}
			for (int n = 0; n < NBldGSI; n++)
			{
				// GSI
				int nvtx = BldGSI[n].nvtx;
				for (int v = 0; v < nvtx; v++)
				{
					latmin = Math.Min(latmin, BldGSI[n].lat[v]);
					latmax = Math.Max(latmax, BldGSI[n].lat[v]);
					lonmin = Math.Min(lonmin, BldGSI[n].lon[v]);
					lonmax = Math.Max(lonmax, BldGSI[n].lon[v]);
				}
			}

			// 標高(PLATEAU/GSI共通)
			if (NDemLat + NDemLon > 0)
			{
				for (int ilat = 0; ilat <= NDemLat; ilat++)
				{
					for (int ilon = 0; ilon <= NDemLon; ilon++)
					{
						latmin = Math.Min(latmin, Dem[ilat, ilon].lat);
						latmax = Math.Max(latmax, Dem[ilat, ilon].lat);
						lonmin = Math.Min(lonmin, Dem[ilat, ilon].lon);
						lonmax = Math.Max(lonmax, Dem[ilat, ilon].lon);
					}
				}
			}

			// 緯度・経度中心
			Lat0 = (latmin + latmax) / 2;
			Lon0 = (lonmin + lonmax) / 2;
		}

		// 描画時の座標中心とスケールを取得する
		private void GetWinScale()
		{
			if ((NDemLat + NDemLon <= 0) && (NBldPLA <= 0) && (NBldGSI <= 0)) return;

			// 最小・最大
			double xmin = +1e10;
			double xmax = -1e10;
			double ymin = +1e10;
			double ymax = -1e10;

			// 建物
			// PLATEAU
			for (int n = 0; n < NBldPLA; n++)
			{
				int nvtx = BldPLA[n].nvtx;
				for (int v = 0; v < nvtx; v++)
				{
					double x = BldPLA[n].x[v];
					double y = BldPLA[n].y[v];
					xmin = Math.Min(xmin, x);
					xmax = Math.Max(xmax, x);
					ymin = Math.Min(ymin, y);
					ymax = Math.Max(ymax, y);
				}
			}
			// GSI
			for (int n = 0; n < NBldGSI; n++)
			{
				int nvtx = BldGSI[n].nvtx;
				for (int v = 0; v < nvtx; v++)
				{
					double x = BldGSI[n].x[v];
					double y = BldGSI[n].y[v];
					xmin = Math.Min(xmin, x);
					xmax = Math.Max(xmax, x);
					ymin = Math.Min(ymin, y);
					ymax = Math.Max(ymax, y);
				}
			}

			// 地面
			if (NDemLat + NDemLon > 0)
			{
				for (int ilat = 0; ilat <= NDemLat; ilat++)
				{
					for (int ilon = 0; ilon <= NDemLon; ilon++)
					{
						double x = Dem[ilat, ilon].x;
						double y = Dem[ilat, ilon].y;
						xmin = Math.Min(xmin, x);
						xmax = Math.Max(xmax, x);
						ymin = Math.Min(ymin, y);
						ymax = Math.Max(ymax, y);
					}
				}
			}

			// XY座標とピクセル座標を変換するための係数を求める
			double x0 = (xmin + xmax) / 2;
			double y0 = (ymin + ymax) / 2;
			dFactor = Math.Min(picGML.Width / (xmax - xmin), picGML.Height / (ymax - ymin)) * 0.9;
			Cx = picGML.Width / 2 - dFactor * x0;
			Cy = picGML.Height / 2 + dFactor * y0;
		}

		// [オプション]メニュー
		private void mnuOption_Click(object sender, EventArgs e)
		{
			Option f = new Option();

			f.chkOrigin.Checked = bLatLon0;
			f.txtOrigin0.Text = Lat0.ToString("0.0###");
			f.txtOrigin1.Text = Lon0.ToString("0.0###");
			f.txtLearth.Text = (Learth / 1000).ToString();
			f.btnBldColor.BackColor = Color.FromArgb(BldColor[0], BldColor[1], BldColor[2]);
			f.btnDemColor.BackColor = Color.FromArgb(DemColor[0], DemColor[1], DemColor[2]);
			//f.nudWidth3D.Value = Width3d;
			//f.nudHeight3D.Value = Height3d;
			//f.nudTheta3D.Value = Theta3d;
			//f.nudPhi3D.Value = Phi3d;
			f.txtGSIBldC0.Text = BldC[0].ToString();
			f.txtGSIBldC1.Text = BldC[1].ToString();
			f.txtBldFactor.Text = BldNumberFactor.ToString();
			f.rbnGSIDemMesh0.Checked = (GSIDemMesh == 2);
			f.rbnGSIDemMesh1.Checked = (GSIDemMesh == 3);
			f.cboDemMesh.SelectedIndex = DemMeshID;
			f.chkDemRegex.Checked = bDemRegex;
			f.chkRx2d.Checked = bRx2d;
			f.txtHRx2d.Text = hRx2d.ToString();
			f.nudDRx2d.Value = dRx2d;

			if (f.ShowDialog() == DialogResult.OK)
			{
				bLatLon0 = f.chkOrigin.Checked;
				Double.TryParse(f.txtOrigin0.Text, out Lat0);
				Double.TryParse(f.txtOrigin1.Text, out Lon0);
				Learth = 1000 * Double.Parse(f.txtLearth.Text);
				BldColor[0] = f.btnBldColor.BackColor.R;
				BldColor[1] = f.btnBldColor.BackColor.G;
				BldColor[2] = f.btnBldColor.BackColor.B;
				DemColor[0] = f.btnDemColor.BackColor.R;
				DemColor[1] = f.btnDemColor.BackColor.G;
				DemColor[2] = f.btnDemColor.BackColor.B;
				//Width3d = (int)f.nudWidth3D.Value;
				//Height3d = (int)f.nudHeight3D.Value;
				//Theta3d = (int)f.nudTheta3D.Value;
				//Phi3d = (int)f.nudPhi3D.Value;
				Double.TryParse(f.txtGSIBldC0.Text, out BldC[0]);
				Double.TryParse(f.txtGSIBldC1.Text, out BldC[1]);
				Double.TryParse(f.txtBldFactor.Text, out BldNumberFactor);
				GSIDemMesh = f.rbnGSIDemMesh0.Checked ? 2 : 3;
				DemMeshID = f.cboDemMesh.SelectedIndex;
				bDemRegex = f.chkDemRegex.Checked;
				bRx2d = f.chkRx2d.Checked;
				Double.TryParse(f.txtHRx2d.Text, out hRx2d);
				dRx2d = (int)f.nudDRx2d.Value;

				// GSI-DEMメッシュ情報
				setGSIDemMesh();

				// データ数表示
				ShowNumberOfData();

				// 緯度経度中心を再計算
				if (bLatLon0)
				{
					GetLatLon0();
				}

				// 建物・地面の緯度・経度・標高->XYZ座標
				CalcXYZ();

				// 建物高さを再計算(GSI)
				if (NBldGSI > 0)
				{
					CalcBldGSIHeight();
				}

				// 再描画
				picGML.Invalidate();
			}
		}

		// ev3ファイル出力
		private void SaveGeometry3d()
		{
			string path = "ev.ev3";
			//string fmt2 = "2 0 {0} {1} {2} {3} {4} {5} {6} {7} {8}";
			string fmt4 = "4 0 {0:0.000} {1:0.000} {2:0.000} {3:0.000} {4:0.000} {5:0.000} {6:0.000} {7:0.000} {8:0.000} {9:0.000} {10:0.000} {11:0.000} {12} {13} {14}";

			using (var sw = new StreamWriter(path))
			{
				try
				{
					sw.WriteLine("-1");
					// 建物
					// PLATEAU
					for (int n = 0; n < NBldPLA; n++)
					{
						if (!BldPLA[n].exclude)
						{
							// 頂点数
							int nvtx = BldPLA[n].nvtx;
							
							// ポリゴン
							sw.Write("{0} 0", nvtx);
							for (int iv = 0; iv < nvtx; iv++)
							{
								sw.Write(" {0:0.000} {1:0.000} {2:0.000}", BldPLA[n].x[iv], BldPLA[n].y[iv], BldPLA[n].z[iv]);
							}
							sw.WriteLine(" {0} {1} {2}", BldColor[0], BldColor[1], BldColor[2]);
						}
					}
					// GSI
					for (int n = 0; n < NBldGSI; n++)
					{
						if (!BldGSI[n].exclude)
						{
							// 頂点数
							int nvtx = BldGSI[n].nvtx;

							// 床と屋根の高さ
							double z1 = BldGSI[n].z;
							double z2 = z1 + BldGSI[n].tall;
							
							// 側面（壁）：4角形の集合
							for (int iv = 0; iv < nvtx - 1; iv++)
							{
								sw.WriteLine(fmt4,
									BldGSI[n].x[iv + 0], BldGSI[n].y[iv + 0], z1,
									BldGSI[n].x[iv + 1], BldGSI[n].y[iv + 1], z1,
									BldGSI[n].x[iv + 1], BldGSI[n].y[iv + 1], z2,
									BldGSI[n].x[iv + 0], BldGSI[n].y[iv + 0], z2,
									BldColor[0], BldColor[1], BldColor[2]);
							}

							// 床と屋根：多角形(ポリゴン)
							for (int m = 0; m < 2; m++)
							{
								double z = (m == 0) ? z1 : z2;
								sw.Write("{0} 0", nvtx);
								for (int iv = 0; iv < nvtx; iv++)
								{
									sw.Write(" {0:0.000} {1:0.000} {2:0.000}", BldGSI[n].x[iv], BldGSI[n].y[iv], z);
								}
								sw.WriteLine(" {0} {1} {2}", BldColor[0], BldColor[1], BldColor[2]);
							}
						}
					}

					// 地面(PLATEAU/GSI共通)、4角形
					if (NDemLat + NDemLon > 0)
					{
						// 標高の最小・最大
						double zmin = +1e10;
						double zmax = -1e10;
						for (int ilat = 0; ilat <= NDemLat; ilat++)
						{
							for (int ilon = 0; ilon <= NDemLon; ilon++)
							{
								if (!Dem[ilat, ilon].sea)  // 海は除く
								{
									zmin = Math.Min(zmin, Dem[ilat, ilon].z);
									zmax = Math.Max(zmax, Dem[ilat, ilon].z);
								}
							}
						}

						int m = Mesh[DemMeshID];        // 間引き間隔
						for (int ilat = 0; ilat <= NDemLat - m; ilat += m)
						{
							for (int ilon = 0; ilon <= NDemLon - m; ilon += m)
							{
								if (select_dem(ilat, ilon, m))
								{
									sw.WriteLine(fmt4,
										Dem[ilat + 0, ilon + 0].x, Dem[ilat + 0, ilon + 0].y, Dem[ilat + 0, ilon + 0].z,
										Dem[ilat + m, ilon + 0].x, Dem[ilat + m, ilon + 0].y, Dem[ilat + m, ilon + 0].z,
										Dem[ilat + m, ilon + m].x, Dem[ilat + m, ilon + m].y, Dem[ilat + m, ilon + m].z,
										Dem[ilat + 0, ilon + m].x, Dem[ilat + 0, ilon + m].y, Dem[ilat + 0, ilon + m].z,
										DemColor[0], DemColor[1], DemColor[2]);
								}
							}
						}
					}
				}
				catch (Exception ex)
				{
					MessageBox.Show(ex.Message + "\n" + ex.StackTrace, "SaveGeometry3d");
				}
			}
		}

		// 3Dプロット
		private void mnuPlot3d_Click(object sender, EventArgs e)
		{
			Process proc = new Process();

			try
			{
				// ev.ev3保存
				SaveGeometry3d();

				// プロセス実行
				proc.StartInfo.FileName = "ev3d_otk.exe";
				proc.StartInfo.Arguments = "ev.ev3";
				proc.Start();
			}
			catch (Exception ex)
			{
				MessageBox.Show(ex.Message, Application.ProductName);
			}
			finally
			{
				proc.Dispose();
			}
		}

		// 切り取り
		private void mnuCut_Click(object sender, EventArgs e)
		{
			//　切り取りモードON
			bCut = true;

			// ツールボタンの背景を赤くする
			tsbCut.BackColor = Color.Red;

			// マウスカーソル=十字
			picGML.Cursor = Cursors.Cross;
		}

		// 切り取り解除
		private void mnuUncut_Click(object sender, EventArgs e)
		{
			ResetCut();
		}
		private void ResetCut()
		{
			// 切り取りモードOFF
			bCut = false;
			Xcut1 = Xcut2 = Ycut1 = Ycut2 = 0;

			// ツールボタンの背景を戻す
			tsbCut.BackColor = Color.Transparent;

			// 全部選択に戻す
			for (int n = 0; n < NBldPLA; n++)
			{
				BldPLA[n].exclude = false;
			}
			for (int n = 0; n < NBldGSI; n++)
			{
				BldGSI[n].exclude = false;
			}
			if (NDemLat + NDemLon > 0)
			{
				for (int ilat = 0; ilat <= NDemLat; ilat++)
				{
					for (int ilon = 0; ilon <= NDemLon; ilon++)
					{
						Dem[ilat, ilon].exclude = false;
					}
				}
			}

			// データ数更新
			ShowNumberOfData();

			// 再描画
			picGML.Invalidate();
		}

		// 建物と地面の配列の切り取りフラグを代入する
		private void CalcCut()
		{
			// 切り取り範囲のXY座標
			double x1 = Math.Min(Xcut1, Xcut2);
			double x2 = Math.Max(Xcut1, Xcut2);
			double y1 = Math.Min(Ycut1, Ycut2);
			double y2 = Math.Max(Ycut1, Ycut2);

			// 建物(PLATEAU)
			for (int n = 0; n < NBldPLA; n++)
			{
				bool exclude = false;
				for (int v = 0; v < BldPLA[n].nvtx; v++)
				{
					double x = BldPLA[n].x[v];
					double y = BldPLA[n].y[v];
					if ((x < x1) || (x > x2) || (y < y1) || (y > y2))
					{
						exclude = true;
						break;
					}
				}
				BldPLA[n].exclude = exclude;
			}

			// 建物(GSI)
			for (int n = 0; n < NBldGSI; n++)
			{
				bool exclude = false;
				for (int v = 0; v < BldGSI[n].nvtx; v++)
				{
					double x = BldGSI[n].x[v];
					double y = BldGSI[n].y[v];
					if ((x < x1) || (x > x2) || (y < y1) || (y > y2))
					{
						exclude = true;
						break;
					}
				}
				BldGSI[n].exclude = exclude;
			}

			// 地面(PLATEAU/GSI共通)
			if (NDemLat + NDemLon > 0)
			{
				for (int ilat = 0; ilat <= NDemLat; ilat++)
				{
					for (int ilon = 0; ilon <= NDemLon; ilon++)
					{
						Dem[ilat, ilon].exclude = (Dem[ilat, ilon].x < x1) || (Dem[ilat, ilon].x > x2) || (Dem[ilat, ilon].y < y1) || (Dem[ilat, ilon].y > y2);
					}
				}
			}
		}

		// 建物高さ編集(GSIの建物のみ有効)
		private void tsmEdit_Click(object sender, EventArgs e)
		{
			if ((iEdit < 0) || (iEdit >= NBldGSI)) return;

			Edit f = new Edit(iEdit);

			f.txtHeight.Text = BldGSI[iEdit].tall.ToString("0.0");

			if (f.ShowDialog() == DialogResult.OK)
			{
				Double.TryParse(f.txtHeight.Text, out BldGSI[iEdit].tall);
				BldGSI[iEdit].btall = true;
			}
		}

		// マウスクリック
		private void picGML_MouseDown(object sender, MouseEventArgs e)
		{
			// マウス位置のXY座標
			//double x = +(e.X - Cx) / dFactor;
			//double y = -(e.Y - Cy) / dFactor;
			winpos2xy(e.X, e.Y, out double x, out double y);

			if (e.Button == MouseButtons.Left)
			{
				// 左クリック:マウスドラッグ開始
				bDrag = true;
				Xold = e.X;
				Yold = e.Y;
				Xcut1 = x;
				Ycut1 = y;
			}
			else if (e.Button == MouseButtons.Right)
			{
				// 右クリック(GSIの建物のみ有効)
				if (NBldGSI > 0)
				{
					iEdit = -1;
					for (int n = 0; n < NBldGSI; n++)
					{
						if (!BldGSI[n].exclude && utils.isInside(x, y, BldGSI[n].x, BldGSI[n].y, BldGSI[n].nvtx))
						{
							iEdit = n;
							break;
						}
					}
					if (iEdit >= 0)
					{
						contextMenuStrip.Show(picGML, new Point(e.X, e.Y));
					}
				}
			}
		}

		// マウスアップ
		private void picGML_MouseUp(object sender, MouseEventArgs e)
		{
			bDrag = false;

			if (bCut)
			{
				// 切り取りフラグ代入
				CalcCut();

				// 描画
				picGML.Invalidate();

				// 切り取りOFF
				bCut = false;

				// ツールバーの色を戻す
				tsbCut.BackColor = Color.Transparent;

				// マウスカーソルを戻す
				picGML.Cursor = Cursors.Default;

				// データ数表示
				ShowNumberOfData();
			}
		}
	
		// マウス移動
		private void picGML_MouseMove(object sender, MouseEventArgs e)
		{
			// 図形を平行移動する
			if (bDrag && !bCut)
			{
				Cx += e.X - Xold;
				Cy += e.Y - Yold;
				picGML.Invalidate();
				Xold = e.X;
				Yold = e.Y;
			}

			// 切り取り範囲の矩形を表示する
			if (bCut && bDrag)
			{
				//Xcut2 = +(e.X - Cx) / dFactor;
				//Ycut2 = -(e.Y - Cy) / dFactor;
				winpos2xy(e.X, e.Y, out Xcut2, out Ycut2);
				picGML.Invalidate();
			}

			// マウス位置表示
			if ((NDemLat + NDemLon > 0) || (NBldPLA > 0) || (NBldGSI > 0))
			{
				//double x = +(e.X - Cx) / dFactor;
				//double y = -(e.Y - Cy) / dFactor;
				winpos2xy(e.X, e.Y, out double x, out double y);
				xy2geopos(x, y, out double lat, out double lon);
				double hgt = Elevation(lat, lon);
				tssLat.Text = "緯度[度]=" + lat.ToString("0.0000");
				tssLon.Text = "経度[度]=" + lon.ToString("0.0000");
				tssEle.Text = "標高[m]=" + hgt.ToString("0.0");
				tssX.Text = "X[m]=" + (int)x;
				tssY.Text = "Y[m]=" + (int)y;
				geopos2xyz(lat, lon, hgt, out x, out y, out double z);
				tssZ.Text = "Z[m]=" + z.ToString("0.0");
			}
			else
			{
				tssLat.Text = "緯度[度]=0";
				tssLon.Text = "経度[度]=0";
				tssEle.Text = "標高[m]=0";
				tssX.Text = "X[m]=0";
				tssY.Text = "Y[m]=0";
				tssZ.Text = "Z[m]=0";
			}
		}

		// マウスホイール
		private void mouseWheel(object sender, MouseEventArgs e)
		{
			// スケール
			const double f = 1.2;
			int d = e.Delta * SystemInformation.MouseWheelScrollLines / 120;
			double ff = (d > 0) ? f : (1 / f);
			dFactor *= ff;

			// 中心
			Cx = ff * Cx + (1 - ff) * e.X;
			Cy = ff * Cy + (1 - ff) * e.Y;

			// 描画
			picGML.Invalidate();
		}

		// データ削除
		private void mnuDelete_Click(object sender, EventArgs e)
		{
			if (sender == mnuDeleteBld)
			{
				// 建物データ削除
				deleteData(0);
			}
			else if (sender == mnuDeleteDem)
			{
				// 地面データ削除
				deleteData(1);
			}
		}

		// [PLATEAU地面3D]メニュー
		private void mnuDemTriPLA_Click(object sender, EventArgs e)
		{
			// PLATEAU地面の三角形を3D表示する(通常使用しない)
			plot3dDemTriPLA();
		}


		// データ削除
		private void deleteData(int id)
		{
			string[] msg = {	"建物データをすべて削除します。", "地面データをすべて削除します。" };
			string[] caption = { "建物データ削除", "地面データ削除" };

			if (MessageBox.Show(msg[id], caption[id], MessageBoxButtons.OKCancel, MessageBoxIcon.Exclamation) == DialogResult.OK)
			{
				// データ数初期化
				if (id == 0)
				{
					NBldPLA = NBldGSI = 0;
				}
				else if (id == 1)
				{
					NDemFileGSI = NDemLat = NDemLon = NDemTriPLA = 0;
				}

				// データ数表示
				ShowNumberOfData();

				// 緯度経度の中心を計算する
				if (bLatLon0)
				{
					GetLatLon0();
				}

				// 建物・地面の緯度・経度・標高->XYZ座標
				CalcXYZ();

				// スケールを計算する：XY座標をウィンドウ座標に変換する係数
				GetWinScale();

				// 図形再描画
				picGML.Invalidate();
			}
		}

		// 環境設定ファイルを読み込む
		private void WriteIni()
		{
			using (var sw = new StreamWriter(FileIni))
			{
				try
				{
					Version ver = new Version(Application.ProductVersion);
					sw.WriteLine("Program={0}", Application.ProductName);
					sw.WriteLine("Version={0},{1}", ver.Major, ver.Minor);
					sw.WriteLine("Window={0},{1},{2},{3}", Math.Max(this.Left, 0), Math.Max(this.Top, 0), this.ClientRectangle.Width, this.ClientRectangle.Height);
					sw.WriteLine("Origin={0},{1},{2}", (bLatLon0 ? 1 : 0), Lat0, Lon0);
					sw.WriteLine("Learth={0}", Learth);
					sw.WriteLine("Color={0},{1},{2},{3},{4},{5}", BldColor[0], BldColor[1], BldColor[2], DemColor[0], DemColor[1], DemColor[2]);
					//sw.WriteLine("View3D={0},{1},{2},{3}", Width3d, Height3d, Theta3d, Phi3d);
					sw.WriteLine("Bld={0},{1},{2}", BldC[0], BldC[1], BldNumberFactor);
					sw.WriteLine("Dem={0},{1},{2}", GSIDemMesh, DemMeshID, (bDemRegex ? 1 : 0));
					sw.WriteLine("OpenRTM={0},{1},{2}", (bRx2d ? 1 : 0), hRx2d, dRx2d);
				}
				catch (Exception ex)
				{
					MessageBox.Show(FileIni + "\n" + ex.Message, "WriteIni");
				}
			}
		}

		// 環境設定ファイルを出力する
		private void ReadIni()
		{
			if (!File.Exists(FileIni)) return;
			int ver = 0;
			char[] delim = { '=', ',' };

			using (var sr = new StreamReader(FileIni))
			{
				try
				{
					string line;
					while ((line = sr.ReadLine()) != null)
					{
						string[] token = line.Trim().Split(delim);
						switch (token[0])
						{
							case "Program":
								if ((token.Length > 1) && (token[1] != Application.ProductName)) return;
								break;
							case "Version":
								if (token.Length > 2)
								{
									int major = Convert.ToInt32(token[1]);
									int minor = Convert.ToInt32(token[2]);
									ver = (10 * major) + minor;
								}
								break;
							case "Window":
								if (token.Length > 4)
								{
									this.Left = Convert.ToInt32(token[1]);
									this.Top = Convert.ToInt32(token[2]);
									this.ClientSize = new Size(Int32.Parse(token[3]), Int32.Parse(token[4]));
								}
								break;
							case "Origin":
								if (token.Length > 3)
								{
									bLatLon0 = (token[1] == "1");
									Double.TryParse(token[2], out Lat0);
									Double.TryParse(token[3], out Lon0);
								}
								break;
							case "Learth":
								if (token.Length > 1)
								{
									Double.TryParse(token[1], out Learth);
								}
								break;
							case "Color":
								if (token.Length > 6)
								{
									Byte.TryParse(token[1], out BldColor[0]);
									Byte.TryParse(token[2], out BldColor[1]);
									Byte.TryParse(token[3], out BldColor[2]);
									Byte.TryParse(token[4], out DemColor[0]);
									Byte.TryParse(token[5], out DemColor[1]);
									Byte.TryParse(token[6], out DemColor[2]);
								}
								break;
							/*
							case "View3D":
								if (token.Length > 4)
								{
									Int32.TryParse(token[1], out Width3d);
									Int32.TryParse(token[2], out Height3d);
									Int32.TryParse(token[3], out Theta3d);
									Int32.TryParse(token[4], out Phi3d);
								}
								break;
							*/
							case "Bld":
								if (token.Length > 3)
								{
									Double.TryParse(token[1], out BldC[0]);
									Double.TryParse(token[2], out BldC[1]);
									Double.TryParse(token[3], out BldNumberFactor);
								}
								break;
							case "Dem":
								if (token.Length > 3)
								{
									Int32.TryParse(token[1], out GSIDemMesh);
									Int32.TryParse(token[2], out DemMeshID);
									bDemRegex = (token[3] == "1");
								}
								break;
							case "OpenRTM":
								if (token.Length > 3)
								{
									bRx2d = (token[1] == "1");
									Double.TryParse(token[2], out hRx2d);
									Int32.TryParse(token[3], out dRx2d);
								}
								break;
							default:
								break;
						}
					}
				}
				catch (Exception ex)
				{
					MessageBox.Show(FileIni + "\n" + ex.Message, "ReadIni");
				}
			}
		}

		// 指定したセル範囲の4隅は選択されており海でもない
		private bool select_dem(int ilat, int ilon, int m)
		{
			dem_t dem1 = Dem[ilat + 0, ilon + 0];
			dem_t dem2 = Dem[ilat + m, ilon + 0];
			dem_t dem3 = Dem[ilat + m, ilon + m];
			dem_t dem4 = Dem[ilat + 0, ilon + m];
			bool sea = dem1.sea || dem2.sea || dem3.sea || dem4.sea;
			//bool sea = dem1.sea && dem2.sea && dem3.sea && dem4.sea;
			bool exclude = dem1.exclude || dem2.exclude || dem3.exclude || dem4.exclude;

			return !sea && !exclude;
		}

		// OpenRTMファイル出力
		private void WriteOpenRTM(string path)
		{
			using (var sw = new StreamWriter(path))
			{
				try
				{
					// ヘッダー
					sw.WriteLine("OpenRTM 1 0");
					sw.WriteLine("title = GML2ort");
					sw.WriteLine("material = 1 5 0.01 建物");  // = 2
					sw.WriteLine("material = 1 3 0.01 地面");  // = 3
					sw.WriteLine("antenna = 1 1");
					sw.WriteLine("antenna = 2 1 0 0 90");

					// 建物: 多角柱
					int mbld = 2;   // 地面の仮の物性値番号
					// PLATEAU
					for (int n = 0; n < NBldPLA; n++)
					{
						if (!BldPLA[n].exclude)
						{
							int nvtx = BldPLA[n].nvtx;
							if (Math.Abs(BldPLA[n].x[0] - BldPLA[n].x[nvtx - 1]) +
								Math.Abs(BldPLA[n].y[0] - BldPLA[n].y[nvtx - 1]) < EPS)
							{
								nvtx--;  // 始点=終点
							}
							sw.Write("polygon = {0}", mbld);
							for (int iv = 0; iv < nvtx; iv++)
							{
								sw.Write(" {0:0.000} {1:0.000} {2:0.000}", BldPLA[n].x[iv], BldPLA[n].y[iv], BldPLA[n].z[iv]);
							}
							sw.WriteLine();
						}
					}
					// GSI
					for (int n = 0; n < NBldGSI; n++)
					{
						if (!BldGSI[n].exclude)
						{
							int nvtx = BldGSI[n].nvtx;
							if (Math.Abs(BldGSI[n].x[0] - BldGSI[n].x[nvtx - 1]) +
								Math.Abs(BldGSI[n].y[0] - BldGSI[n].y[nvtx - 1]) < EPS)
							{
								nvtx--;  // 始点=終点
							}
							sw.Write("pillar = {0} {1:0.000} {2:0.000}", mbld, BldGSI[n].z, BldGSI[n].z + BldGSI[n].tall);
							for (int iv = 0; iv < nvtx; iv++)
							{
								sw.Write(" {0:0.000} {1:0.000}", BldGSI[n].x[iv], BldGSI[n].y[iv]);
							}
							sw.WriteLine();
						}
					}

					// 地面: 四角形
					if (NDemLat + NDemLon > 0)
					{
						int m = Mesh[DemMeshID];       // 間引き間隔
						int mdem = 3;           // 地面の仮の物性値番号
						for (int ilat = 0; ilat <= NDemLat - m; ilat += m)
						{
							for (int ilon = 0; ilon <= NDemLon - m; ilon += m)
							{
								if (select_dem(ilat, ilon, m))
								{
									double x1 = Dem[ilat + 0, ilon + 0].x;
									double y1 = Dem[ilat + 0, ilon + 0].y;
									double z1 = Dem[ilat + 0, ilon + 0].z;
									double x2 = Dem[ilat + m, ilon + 0].x;
									double y2 = Dem[ilat + m, ilon + 0].y;
									double z2 = Dem[ilat + m, ilon + 0].z;
									double x3 = Dem[ilat + m, ilon + m].x;
									double y3 = Dem[ilat + m, ilon + m].y;
									double z3 = Dem[ilat + m, ilon + m].z;
									double x4 = Dem[ilat + 0, ilon + m].x;
									double y4 = Dem[ilat + 0, ilon + m].y;
									double z4 = Dem[ilat + 0, ilon + m].z;
									sw.WriteLine("polygon = {0} {1:0.000} {2:0.000} {3:0.000} {4:0.000} {5:0.000} {6:0.000} {7:0.000} {8:0.000} {9:0.000} {10:0.000} {11:0.000} {12:0.000}",
										mdem, x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4);
								}
							}
						}
					}

					// 観測面
					if (bRx2d)
					{
						int m = Mesh[DemMeshID];       // 間引き間隔
						int rxant = 1;
						for (int ilat = 0; ilat <= NDemLat - m; ilat += m)
						{
							for (int ilon = 0; ilon <= NDemLon - m; ilon += m)
							{
								if (select_dem(ilat, ilon, m))
								{
									double x1 = Dem[ilat + 0, ilon + 0].x;
									double y1 = Dem[ilat + 0, ilon + 0].y;
									double z1 = Dem[ilat + 0, ilon + 0].z + hRx2d;
									double x2 = Dem[ilat + m, ilon + 0].x;
									double y2 = Dem[ilat + m, ilon + 0].y;
									double z2 = Dem[ilat + m, ilon + 0].z + hRx2d;
									double x3 = Dem[ilat + m, ilon + m].x;
									double y3 = Dem[ilat + m, ilon + m].y;
									double z3 = Dem[ilat + m, ilon + m].z + hRx2d;
									double x4 = Dem[ilat + 0, ilon + m].x;
									double y4 = Dem[ilat + 0, ilon + m].y;
									double z4 = Dem[ilat + 0, ilon + m].z + hRx2d;
									sw.WriteLine("rx2d = {0} {1:0.000} {2:0.000} {3:0.000} {4:0.000} {5:0.000} {6:0.000} {7:0.000} {8:0.000} {9:0.000} {10:0.000} {11:0.000} {12:0.000} {13} {14}",
										rxant, x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4, dRx2d, dRx2d);
								}
							}
						}
					}

					// 計算条件
					sw.WriteLine("frequency = 1e9");
					sw.WriteLine("maxpath = 30");
					sw.WriteLine("maxref = 3");
					sw.WriteLine("ndivlaunch = 90");
					sw.WriteLine("component = 1 1 2 0");
					//sw.WriteLine("ndivantenna = 36");
					//sw.WriteLine("log = 0 0");
					//sw.WriteLine("plot3d = 1 0 0 0");

					sw.WriteLine("end");
				}
				catch (Exception ex)
				{
					MessageBox.Show(path + "\n" + ex.Message, "WriteOpenRTM");
				}
			}
		}

		// 出力データのXY座標の範囲を取得する
		private void GetXYMinMax(out double xmin, out double xmax, out double ymin, out double ymax)
		{
			xmin = +1e10;
			xmax = -1e10;
			ymin = +1e10;
			ymax = -1e10;

			// 建物(PLATEAU)
			for (int n = 0; n < NBldPLA; n++)
			{
				if (!BldPLA[n].exclude)
				{
					int nvtx = BldPLA[n].nvtx;
					for (int v = 0; v < nvtx; v++)
					{
						double x = BldPLA[n].x[v];
						double y = BldPLA[n].y[v];
						xmin = Math.Min(xmin, x);
						xmax = Math.Max(xmax, x);
						ymin = Math.Min(ymin, y);
						ymax = Math.Max(ymax, y);
					}
				}
			}
			// 建物(GSI)
			for (int n = 0; n < NBldGSI; n++)
			{
				if (!BldGSI[n].exclude)
				{
					int nvtx = BldGSI[n].nvtx;
					for (int v = 0; v < nvtx; v++)
					{
						double x = BldGSI[n].x[v];
						double y = BldGSI[n].y[v];
						xmin = Math.Min(xmin, x);
						xmax = Math.Max(xmax, x);
						ymin = Math.Min(ymin, y);
						ymax = Math.Max(ymax, y);
					}
				}
			}

			// 地面(PLATEAU/GSI共通)
			if (NDemLat + NDemLon > 0)
			{
				for (int ilat = 0; ilat <= NDemLat; ilat++)
				{
					for (int ilon = 0; ilon <= NDemLon; ilon++)
					{
						if (!Dem[ilat, ilon].exclude)
						{
							double x = Dem[ilat, ilon].x;
							double y = Dem[ilat, ilon].y;
							xmin = Math.Min(xmin, x);
							xmax = Math.Max(xmax, x);
							ymin = Math.Min(ymin, y);
							ymax = Math.Max(ymax, y);
						}
					}
				}
			}
		}

		// GSI DEM のメッシュ情報
		private void setGSIDemMesh()
		{
			if (GSIDemMesh == 2)
			{
				// 2次メッシュ(10m)
				LatDiv = 750;
				LonDiv = 1125;
				LatLng = 1.0 / 12;
				LonLng = 1.0 / 8;
			}
			else
			{
				// 3次メッシュ(5m)
				LatDiv = 150;
				LonDiv = 225;
				LatLng = 1.0 / 120;
				LonLng = 1.0 / 80;
			}
		}

		// 緯度[度],経度[度],標高[m]をX,Y,Z座標[m]に変換する
		// Learth : 地球の周囲[m]
		// Lat0, Lon0 : 中心の緯度・経度[度]
		private void geopos2xyz(double lat, double lon, double hgt, out double x, out double y, out double z)
		{
			const double dtor = Math.PI / 180;
			double rearth = Learth / (2 * Math.PI);

			x = rearth * (lon - Lon0) * dtor * Math.Cos(lat * dtor);
			y = rearth * (lat - Lat0) * dtor;
			z = hgt - ((x * x) + (y * y)) / (2 * rearth);
		}

		// X,Y座標[m]を緯度,経度[度]に変換する
		// Learth : 地球の周囲[m]
		// Lat0, Lon0 : 中心の緯度・経度[度]
		private void xy2geopos(double x, double y, out double lat, out double lon)
		{
			const double dtor = Math.PI / 180;
			double rearth = Learth / (2 * Math.PI);

			lat = Lat0 + y / (rearth * dtor);
			lon = Lon0 + x / (rearth * dtor * Math.Cos(lat * dtor));
		}

		// PLATEAU地面標高の元データを3D表示する
		private void plot3dDemTriPLA()
		{
			if (NDemTriPLA == 0)
			{
				MessageBox.Show("PLATEAU地面標高データがありません。", mnuPLATEAUDEM3D.Text);
				return;
			}

			string fmt = "3 0 {0:0.000} {1:0.000} {2:0.000} {3:0.000} {4:0.000} {5:0.000} {6:0.000} {7:0.000} {8:0.000} {9} {10} {11}";

			this.Cursor = Cursors.WaitCursor;

			using (var sw = new StreamWriter("ev.ev3"))
			{
				try
				{
					sw.WriteLine("-1");
					for (int n = 0; n < NDemTriPLA; n++)
					{
						geopos2xyz(DemTriPLA[n][0, 0], DemTriPLA[n][0, 1], DemTriPLA[n][0, 2], out double x1, out double y1, out double z1);
						geopos2xyz(DemTriPLA[n][1, 0], DemTriPLA[n][1, 1], DemTriPLA[n][1, 2], out double x2, out double y2, out double z2);
						geopos2xyz(DemTriPLA[n][2, 0], DemTriPLA[n][2, 1], DemTriPLA[n][2, 2], out double x3, out double y3, out double z3);
						double xmin = Math.Min(x1, Math.Min(x2, x3));
						double xmax = Math.Max(x1, Math.Max(x2, x3));
						double ymin = Math.Min(y1, Math.Min(y2, y3));
						double ymax = Math.Max(y1, Math.Max(y2, y3));
						double xcutmin = Math.Min(Xcut1, Xcut2);
						double xcutmax = Math.Max(Xcut1, Xcut2);
						double ycutmin = Math.Min(Ycut1, Ycut2);
						double ycutmax = Math.Max(Ycut1, Ycut2);
						bool bcut = (xcutmax - xcutmin > 1) && (ycutmax - ycutmin > 1);  // 切り取り済み?
						// 切り取りなしのときはすべて、切り取りありのときは切り取り範囲内を表示する
						if (!bcut ||	((xcutmin < xmin) && (xmax < xcutmax) && (ycutmin < ymin) && (ymax < ycutmax)))
						{
							sw.WriteLine(fmt, x1, y1, z1, x2, y2, z2, x3, y3, z3, DemColor[0], DemColor[1], DemColor[2]);
						}
					}
				}
				catch { }
			}
			
			Process proc = new Process();
			try
			{
				proc.StartInfo.FileName = "ev3d_otk.exe";
				proc.StartInfo.Arguments = "ev.ev3";
				proc.Start();
			}
			catch { }

			this.Cursor = Cursors.Default;
		}

		// XY座標[m]をピクセル座標に変換する
		// ピクセル座標は左上が原点
		private void xy2winpos(double x, double y, out float px, out float py)
		{
			px = (float)(+dFactor * x + Cx);
			py = (float)(-dFactor * y + Cy);
		}

		// ピクセル座標をXY座標[m]に変換する
		// ピクセル座標は左上が原点
		private void winpos2xy(float px, float py, out double x, out double y)
		{
			x = +(px - Cx) / dFactor;
			y = -(py - Cy) / dFactor;
		}

		// [バージョン]
		private void mnuVersion_Click(object sender, EventArgs e)
		{
			Version ver = new Version(Application.ProductVersion);
			string msg
			= Application.ProductName + "\n"
			+ "Version " + ver.Major + "." + ver.Minor + "\n"
			+ "2023";
			MessageBox.Show(msg, Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Information);
		}

	}

	// PLATEAU用建物データ：ポリゴン
	struct bldPLA_t
	{
		// 元データ(標高あり)
		public int nvtx;               // 頂点数
		public double[] lat, lon, hgt;  // 頂点の緯度・経度[度]、標高[m]
		// 加工データ
		public double[] x, y, z;       // 頂点のXYZ座標[m]
		public bool exclude;           // 切り取り用フラグ
	}

	// GSI用建物データ：多角柱
	struct bldGSI_t
	{
		// 元データ(標高なし)
		public int nvtx;                // 頂点数
		public double[] lat, lon;       // 頂点の緯度・経度[度]
		// 加工データ
		public double hgt;              // 床中心の標高[m]（demから算出）
		public double[] x, y;           // 頂点のXY座標[m]
		public double z;                // 床中心のZ座標[m]
		public bool btall;              // 高さを編集済み
		public double tall;             // 地面からの高さ[m]
		public bool exclude;            // 切り取り用フラグ
	}   
	
	// DEMファイルデータ(GSI)
	struct demfile_t 
	{
		public double[,] hgt;				// 標高(セル中心)
		public double latmin, latmax, lonmin, lonmax;   // 緯度経度範囲
		public string meshname; 

		public demfile_t(int maxlat, int maxlon)
		{
			hgt = new double[maxlat, maxlon];
			for (int ilat = 0; ilat < maxlat; ilat++)
			{
				for (int ilon = 0; ilon < maxlon; ilon++)
				{
					hgt[ilat, ilon] = -9999;		// 海面で初期化
				}
			}
			latmin =
			lonmin = +1e10;
			latmax =
			lonmax = -1e10;
			meshname = String.Empty;
		}
	}

	// 地面標高データ(PLATEAU/GSI共通、節点)
	struct dem_t
	{
		public double lat, lon, hgt;	// 緯度・経度[度]、標高[m]
		public double x, y, z;      // XYZ座標[m]
		public bool sea;            // 海であるか
		public bool exclude;			// 切り取り用フラグ(除外されたか)
	}
}
