/*
dlspa: FDTD (2D)
*/

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <time.h>

#ifdef _OPENMP
#include <omp.h>
#endif

#include "alloc.h"

extern void make_mesh(int, int, float, float, float, float *, float *, float *, float *, float *, float *, float *);
extern void txrx(int, int, int, int, int, const int [], int, int *, int *, int, int *, int *);
extern void geometry(int, int, int, int, int, int, const int [], const int [][3], const int [], float, float, float, const float [], const float [], int64_t, float ***, float ***);
extern void geometry_test(int, int, int, int, int, int, float, float, const float [], const float [], float ***, float ***);
extern void solve(int, int, int, int, int, const float [], float, int, int *, int *, int, int *, int *, float **, float **, float **, float **, float **, float *, float *, float *, float *, float *, float *, float **, float **, float, float, float **, float **, float **, float **, float ***, float ***, int, int, float **, float **, float **);
extern void writedata(const char [], int, int, int, int, int, int, int, const float [], float ***, float ***, float ****, float ****, int, int, int);
extern void plot2d_g(int, int, int, int, int, int, int, int, float ***, int, int, int, int *, int *, int, int *, int *, const char []);
extern void plot2d_e(int, int, int, int, int, int, int, int, float ****, int, int, int *, int *, const char []);
extern void plot2d_s(int, int, int, int, int, int, int, float ****, float ****, int, int, const char []);
extern void log_s1(int, int, int, int, float ****, float ****);
extern void log_s2(int, int, int, int, float ****, float ****);
extern int memory_size(int, int, int, int, int, int, int, int, int);
extern int file_size(int, int, int, int, int, int, int, int, int);
extern void mpi_init(int, char *[], int *, int *);
extern void mpi_close(void);
extern void comm_S(int, int, int, int, int, int, int, float ****, float ****);
extern double cputime(void);

const float C = 2.99792458e8f;  // 光速

int main(int argc, char *argv[])
{
	// パラメーター
	const int   Itest  = 0;               // 0=学習, 1=テスト（通常0）
	const int   Ndata  = 1000;            // データ数
	const int   Nout   = 4;               // 外部領域セル数
	const int   Lant   = 2;               // アンテナ位置（対象領域境界からのセル数）
	const int   ABC    = 4;               // PML層数（0のときはMur一次）
	const int   Niter  = 600;             // タイムステップ数
	const int   Nant   = 2;               // アンテナ辺数（1/2）
	const int   Nsmot  = 1;               // 形状平滑化回数（0以上）
	const int   Mean[] = {1, 3};          // 形状平均個数（学習用:最小,最大,1/2/3/4）
	const int   Sdif   = 1;               // S行列差分（0/1）
	const int   Mx     = 20;              // Xセル数（内部）
	const int   My     = 20;              // Yセル数（内部）
	const float Dx     = 0.005f;          // Xセルサイズ
	const float Dy     = 0.005f;          // Yセルサイズ
	const int   Pant[] = {0, 1, 21};      // アンテナ位置（開始,間隔,個数）
	const int   Nobj[] = {1, 5};          // 誘電体数（学習用:最小,最大）
	const int   Lobj[][3] = {{1, 8, 0}, {1, 8, 0}};  // 誘電体XYセル数（学習用:最小,最大,余白）
	const int   NFreq  = 1;               // 周波数数
	const float Freq0  = 3.0e9f;          // 開始周波数
	const float Freq1  = 3.0e9f;          // 終了周波数
	const float Eps0   = 1;               // 背景媒質比誘電率
	const float Sig0   = 0;               // 背景媒質導電率[S/m]
	const float Eps[]  = {1.5f, 2.5f};    // 比誘電率（最小,最大）
	const float Sig[]  = {0.02f, 0.08f};  // 導電率[S/m]（最小,最大）

	// 定数
	const int   Nx     = Mx + (2 * Nout);
	const int   Ny     = My + (2 * Nout);
	const float gRatio = 0.5f;     // 長方形/(長方形+楕円) （通常0.5）
	const int   size1  = 2;        // image出力サイズ(1/2/4バイト, 通常2)
	const int   size2  = 2;        // label出力サイズ(1/2/4バイト, 通常2)
	const int64_t Seed = 10000;    // 乱数の種
	char outfile[BUFSIZ] = "fdtd.bin";   // 出力ファイル名

	// debug
	const int log1 = 1;  // (数値出力) 計算条件とS行列平均(0/1, 通常1)
	const int log2 = 0;  // (数値出力) 収束状況(0/1, 通常0)
	const int log3 = 0;  // (数値出力) S行列(0/1, 通常0)
	const int fig1 = 1;  // (図形出力) 誘電率/導電率分布(0/1, 通常0, -1:debug)
	const int fig2 = 1;  // (図形出力) 電界分布(0,1,2..., 複数ページ対応, 通常0)
	const int fig3 = 1;  // (図形出力) S行列(0/1, 通常0)
	const char fig1ev2[] = "fig1.ev2";  // $ ev2d.exe fig1.ev2
	const char fig2ev2[] = "fig2.ev2";  // $ ev2d.exe fig2.ev2
	const char fig3ev2[] = "fig3.ev2";  // $ ev2d.exe fig3.ev2

	// check
	assert((Itest == 0) || (Itest == 1));
	assert((Mx > 0) && (My > 0));
	assert((Nout > Lant) && (Lant > 0));
	assert((Nant == 1) || (Nant == 2));
	assert(ABC >= 0);
	assert(NFreq > 0);
	assert((Mean[1] >= Mean[0]) && (Mean[0] > 0));
	assert((Sdif == 0) || (Sdif == 1));
	assert((size1 == 1) || (size1 == 2) || (size1 == 4));
	assert((size2 == 1) || (size2 == 2) || (size2 == 4));

	// MPI
	int commsize, commrank;
	mpi_init(argc, argv, &commsize, &commrank);

	// 引数: スレッド数、出力ファイル
	int nthread = 1;
	if (argc > 1) {
		nthread = atoi(argv[1]);
	}
	if (argc > 2) {
		strcpy(outfile, argv[2]);
	}

	// OpenMP
#ifdef _OPENMP
	omp_set_num_threads(nthread);
#endif

	assert((Nx > 0) && (Ny > 0) && (Mx > 0) && (My > 0));
	assert(NFreq > 0);
	assert(Nout > Lant);
	assert(ABC >= 0);

	// 開始時刻
	const double tstart = cputime();

	// メッシュ作成
	float Dt;
	float *Xn, *Yn, *dXn, *dYn, *dXc, *dYc;
	alloc1d(float, Xn,  Nx + 1)
	alloc1d(float, Yn,  Ny + 1)
	alloc1d(float, dXn, Nx + 1)
	alloc1d(float, dYn, Ny + 1)
	alloc1d(float, dXc, Nx + 0)
	alloc1d(float, dYc, Ny + 0)
	make_mesh(Nx, Ny, Dx, Dy, Eps0, &Dt, Xn, Yn, dXn, dYn, dXc, dYc);

	// 送受信点座標
	int NTx, NRx;
	NTx = NRx = Nant * Pant[2];
	int *iTx, *jTx, *iRx, *jRx;
	alloc1d(int, iTx, NTx)
	alloc1d(int, jTx, NTx)
	alloc1d(int, iRx, NRx)
	alloc1d(int, jRx, NRx)
	txrx(Nant, Nx, Ny, Nout, Lant, Pant, NTx, iTx, jTx, NRx, iRx, jRx);

	// 周波数
	float *Freq;
	alloc1d(float, Freq, NFreq)
	const float dfreq = (NFreq > 1) ? (Freq1 - Freq0) / (NFreq - 1) : 0;
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		Freq[ifreq] = Freq0 + ifreq * dfreq;
	}

	// 計算条件表示
	if (log1 && (commrank == 0)) {
		const int mem = memory_size(nthread, Ndata, Nx, Ny, NFreq, NTx, NRx, ABC, fig2);
		const int filesize = file_size(Ndata, Mx, My, 1, NFreq, NTx, NRx, size1, size2);
		printf("data=%d Nx=%d Ny=%d Mx=%d My=%d Tx=%d Rx=%d F=%d S=%d Sdif=%d\n",
			Ndata, Nx, Ny, Mx, My, NTx, NRx, NFreq, Nsmot, Sdif);
		printf("process=%d thread=%d memory=%dMB ABC=%d iter=%d output=%s(%dMB)\n",
			commsize, nthread, mem, ABC, Niter, outfile, filesize);
		fflush(stdout);
	}

	// 誘電率、導電率配列(セル中心, xデータ数)
	float ***Epsr, ***Sigm;
	alloc3d(float, Epsr, Ndata, Nx, Ny)
	alloc3d(float, Sigm, Ndata, Nx, Ny)

	// 誘電率、導電率配列代入
	if      (Itest == 0) {
		geometry(Ndata, Nx, Ny, Nout, Sdif, Nsmot, Nobj, Lobj, Mean, gRatio, Eps0, Sig0, Eps, Sig, Seed, Epsr, Sigm);
	}
	else if (Itest == 1) {
		assert(Ndata == 5 + Sdif);  // データ数5限定
		geometry_test(Ndata, Nx, Ny, Nout, Sdif, Nsmot, Eps0, Sig0, Eps, Sig, Epsr, Sigm);
	}

	// debug, 誘電率,導電率分布図
	if (((fig1 == 1) || (fig1 == -1)) && (commrank == 0)) {
		plot2d_g(0, 4, 2, 4, Ndata, Nx, Ny, Nout, Epsr, 1, 0, NTx, iTx, jTx, NRx, iRx, jRx, fig1ev2);
		//plot2d_g(0, 4, 1, 4, Ndata, Nx, Ny, Nout, Sigm, 1, 0, NTx, iTx, jTx, NRx, iRx, jRx, fig1ev2);
	}
	if (fig1 < 0) {
		exit(0);  // debug: 計算せずに終了する
	}

	// debug, 全領域電界絶対値, 図形出力用, セル中心, 表示データ数x送信点xセル数の配列
	float ****Ea = NULL;
	if (fig2 && (commrank == 0)) {
		alloc4d(float, Ea, fig2, NTx, Nx, Ny)
	}

	// S行列（データ数x送信点数x受信点数）
	float ****S_r = NULL, ****S_i = NULL;
	alloc4d(float, S_r, Ndata, NFreq, NTx, NRx)
	alloc4d(float, S_i, Ndata, NFreq, NTx, NRx)

	// プロセスごとのデータ番号の範囲（ブロック分割）
	const int ndata = (Ndata + (commsize - 1)) / commsize;
	int idata0 = commrank * ndata;
	if (idata0 > Ndata - 1) {
		idata0 = Ndata - 1;
	}
	int idata1 = idata0 + ndata;
	if (idata1 > Ndata) {
		idata1 = Ndata;
	}

	// OpenMP並列領域開始
#ifdef _OPENMP
#pragma omp parallel
#endif
{
	// FDTD係数(スレッド変数)
	float **C1Ez = NULL, **C2Ez = NULL;
	alloc2d(float, C1Ez, Nx + 1, Ny + 1)
	alloc2d(float, C2Ez, Nx + 1, Ny + 1)

	// ABC変数(スレッド変数)
	float **Ezmurx = NULL, **Ezmury = NULL;
	float **Ezx = NULL, **Ezy = NULL;
	if (ABC == 0) {
		alloc2d(float, Ezmurx, 2, Ny + 1)
		alloc2d(float, Ezmury, 2, Nx + 1)
	}
	else {
		alloc2d(float, Ezx, Nx + 1 + 2 * ABC, Ny + 1 + 2 * ABC)
		alloc2d(float, Ezy, Nx + 1 + 2 * ABC, Ny + 1 + 2 * ABC)
	}

	// 電磁界配列(スレッド変数)
	float **Ez = NULL, **Hx = NULL, **Hy = NULL;
	alloc2d(float, Ez, Nx + 1 + 2 * ABC, Ny + 1 + 2 * ABC)
	alloc2d(float, Hx, Nx + 1 + 2 * ABC, Ny + 0 + 2 * ABC)
	alloc2d(float, Hy, Nx + 0 + 2 * ABC, Ny + 1 + 2 * ABC)

	// debug, 全領域調和界(スレッド変数, 節点)
	float **Ez_r = NULL, **Ez_i = NULL;
	if (fig2 && (commrank == 0)) {
		alloc2d(float, Ez_r, Nx + 1, Ny + 1)
		alloc2d(float, Ez_i, Nx + 1, Ny + 1)
	}

	// スレッド番号
	const int ithread = omp_get_thread_num();

	// データに関するループ
	for (int idata = idata0 + ithread; idata < idata1; idata += nthread) {
		const int ilog = log2 && (commrank == 0);  // 収束状況
		const int iout = (idata < fig2) && (commrank == 0);  // 電界分布
		if (log1 && (commrank == 0) && (idata % 1000 == 0)) {printf("%d %d\n", idata, ithread); fflush(stdout);}  // 経過出力
		for (int itx = 0; itx < NTx; itx++) {
			solve(
				itx, Niter, Nx, Ny, NFreq, Freq, Dt, ABC,
				iTx, jTx, NRx, iRx, jRx,
				Ez, Hx, Hy, Ezx, Ezy,
				Xn, Yn, dXn, dYn, dXc, dYc,
				Epsr[idata], Sigm[idata], Eps0, Sig0,
				C1Ez, C2Ez,
				Ezmurx, Ezmury,
				S_r[idata], S_i[idata],
				ilog, iout, Ez_r, Ez_i, (iout ? Ea[idata][itx] : NULL));
		}
	}
}
	// OpenMP並列領域終了

	// S行列をrootに集める
	if (commsize > 1) {
		comm_S(commsize, commrank, idata0, idata1, NFreq, NTx, NRx, S_r, S_i);
	}

	// 出力:root
	if (commrank == 0) {
		// ファイル出力
		writedata(outfile, Ndata, Nx, Ny, Nout, NTx, NRx, NFreq, Freq, Epsr, Sigm, S_r, S_i, Sdif, size1, size2);

		// debug, S行列平均数値出力
		if (log1) {
			log_s1(Ndata, NFreq, NTx, NRx, S_r, S_i);
		}

		// debug, S行列数値出力
		if (log3) {
			log_s2(Ndata, NFreq, NTx, NRx, S_r, S_i);
		}

		// debug, 電界分布図形出力
		if (fig2) {
			const int edata = (fig2 < Ndata) ? fig2 : Ndata;
			plot2d_e(3, 2, 1, 6, edata, NTx, Nx, Ny, Ea, 40, 0, iTx, jTx, fig2ev2);
		}

		// debug, S行列図形出力
		if (fig3) {
			plot2d_s(0, 4, 4, 6, Ndata, NTx, NRx, S_r, S_i, 50, 0, fig3ev2);
		}
	}

	// 計算時間
	const double tstop = cputime();
	if (log1 && (commrank == 0)) {
		printf("CPU=%.3f[sec]\n", tstop - tstart);
		fflush(stdout);
	}

	// MPI
	mpi_close();

	return 0;
}
