/*
plot2d.c
fdtd2d図形出力, 2D, debug用
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include "ev.h"
#include "alloc.h"

extern void getminmax3f(int, int, int, int, int, int, int, float ***, double *, double *);
extern void getminmax4f(int, int, int, int, int, int, int, int, int, float ****, double *, double *);
extern void plot2d_contour(double, double, int, int, double **, double, double, int, int);

// 誘電率分布, 送受信点, 全データ1ページ
// idata0<=idata<idata1, idata=0,1,2,,,,
// mx : 1行の図数
// space : 余白
void plot2d_g(int idata0, int idata1, int mx, int space, int Ndata, int Nx, int Ny, int Nout,
	float ***f, int color, int grid,
	int NTx, int *iTx, int *jTx, int NRx, int *iRx, int *jRx, const char fn[])
{
	assert((idata0 < idata1) && (mx > 0) && (Nx > 0) && (Ny > 0) && (NTx > 0) && (NRx > 0));

	char str[BUFSIZ], fmt[BUFSIZ];
	const double dm = 0.3;

	// 最小・最大
	double dmin, dmax;
	getminmax3f(0, Ndata, 0, Nx, 0, Ny, 0, f, &dmin, &dmax);

	// 作業配列
	double **d;
	alloc2d(double, d, Nx, Ny)

	// 開始
	const int nfig = idata1 - idata0;
	const int my = nfig / mx + (nfig % mx > 0);
	const double w = (mx * Nx) + ((mx + 1) * space);
	const double h = (my * Ny) + ((my + 2) * space);
	const double hgt = h / 60;
	ev2d_init(w, h);
	ev2d_newPage();

	// 等高線図, 1データ=1図
	int ifig = 0;
	for (int py = my - 1; py >= 0; py--) {
	for (int px = 0; px < mx; px++) {
		if ((ifig < idata1 - idata0) && (idata0 + ifig < Ndata)) {
			// 原点
			const double x0 = (px * Nx) + ((px + 1) * space);
			const double y0 = (py * Ny) + ((py + 2) * space);

			// XY配列
			for (int i = 0; i < Nx; i++) {
			for (int j = 0; j < Ny; j++) {
				d[i][j] = f[idata0 + ifig][i][j];
			}
			}

			// 等高線図
			plot2d_contour(x0, y0, Nx, Ny, d, dmin, dmax, color, grid);

			// 内部領域（灰）
			ev2d_setColor(160, 160, 160);
			ev2d_drawRectangle(x0 + Nout, y0 + Nout, x0 + Nx - Nout, y0 + Ny - Nout);

			// データ番号
			ev2d_setColor(0, 0, 0);
			sprintf(str, "%d", idata0 + ifig + 1);
			ev2d_drawString(x0, y0 + Ny + 0.2 * hgt, hgt, str);

			// 送信点（赤）
			ev2d_setColor(255, 0, 0);
			for (int n = 0; n < NTx; n++) {
				ev2d_fillRectangle(x0 + iTx[n] - dm, y0 + jTx[n] - dm, x0 + iTx[n] + dm, y0 + jTx[n] + dm);
			}

			// 受信点（緑）
			ev2d_setColor(0, 255, 0);
			for (int n = 0; n < NRx; n++) {
				ev2d_fillRectangle(x0 + iRx[n] - dm, y0 + jRx[n] - dm, x0 + iRx[n] + dm, y0 + jRx[n] + dm);
			}

			// 図番号++
			ifig++;
		}
	}
	}
	free2d(d, Nx)

	// 最小・最大
	strcpy(fmt, "data=%d fig=%d Nx=%d Ny=%d Mx=%d My=%d Tx=%d Rx=%d min=%.4g max=%.4g");
	sprintf(str, fmt, Ndata, idata1 - idata0, Nx, Ny, Nx - 2 * Nout, Ny - 2 * Nout, NTx, NRx, dmin, dmax);
	ev2d_setColor(0, 0, 0);
	ev2d_drawString(space, 1.0 * hgt, 1.5 * hgt, str);

	// 出力
	ev2d_file(1, fn);
	ev2d_output();
}


// 電界分布
void plot2d_e(int mx, int my, int mstep, int space, int ndata, int NTx, int Nx, int Ny,
	float ****e, int dbspan, int grid, int *iTx, int *jTx, const char fn[])
{
	assert((mx > 0) && (my > 0) && (mstep > 0) && (NTx > 0) && (Nx > 0) && (Ny > 0) && (ndata > 0));

	char str[BUFSIZ], fmt[BUFSIZ];
	const double eps = 1e-12;
	const double dm = 0.3;

	// 最小・最大
	double dmin, dmax;
	getminmax4f(0, ndata, 0, NTx, 0, Nx, 0, Ny, dbspan, e, &dmin, &dmax);
	dmin = dbspan ? (dmax - dbspan) : 0;

	// 作業配列
	double **d;
	alloc2d(double, d, Nx, Ny)

	// 開始
	const double w = (mx * Nx) + ((mx + 1) * space);
	const double h = (my * Ny) + ((my + 2) * space);
	const double hgt = h / 60;
	ev2d_init(w, h);

	// 1データ=1ページ
	for (int idata = 0; idata < ndata; idata++) {
		// 改ページ
		ev2d_newPage();

		// 等高線図(カラー塗りつぶし), 1送信点=1図
		int num = 0;
		for (int py = my - 1; py >= 0; py--) {
		for (int px = 0; px < mx; px++) {
			// 送信点番号
			const int itx = (num + 1) * mstep - 1;

			if (itx < NTx) {
				// XY配列
				for (int i = 0; i < Nx; i++) {
				for (int j = 0; j < Ny; j++) {
					d[i][j] = e[idata][itx][i][j];
				}
				}

				// dB
				if (dbspan) {
					for (int i = 0; i < Nx; i++) {
					for (int j = 0; j < Ny; j++) {
						d[i][j] = 20 * log10(fmax(d[i][j], eps));
					}
					}
				}

				// 原点
				const double x0 = (px * Nx) + ((px + 1) * space);
				const double y0 = (py * Ny) + ((py + 2) * space);

				// 等高線図(カラー)
				plot2d_contour(x0, y0, Nx, Ny, d, dmin, dmax, 1, grid);

				// 送信点
				ev2d_setColor(0, 0, 0);
				ev2d_fillRectangle(x0 + iTx[itx] - dm, y0 + jTx[itx] - dm, x0 + iTx[itx] + dm, y0 + jTx[itx] + dm);

				// データ番号
				ev2d_setColor(0, 0, 0);
				sprintf(str, "%d", itx + 1);
				ev2d_drawString(x0, y0 + Ny + 0.2 * hgt, hgt, str);

				// 図++
				num++;
			}
		}
		}

		// 最小・最大
		strcpy(fmt, dbspan ? "Tx=%d fig=%d step=%d Nx=%d Ny=%d min=%.3fdB max=%.3fdB"
		                   : "Tx=%d fig=%d step=%d Nx=%d Ny=%d min=%.6f max=%.6f");
		sprintf(str, fmt, NTx, num, mstep, Nx, Ny, dmin, dmax);
		ev2d_setColor(0, 0, 0);
		ev2d_drawString(space, 1.0 * hgt, 1.5 * hgt, str);
	}
	free2d(d, Nx)

	// 出力
	ev2d_file(1, fn);
	ev2d_output();
}
