/*
plotutils.c
fdtd2d/fdtd3d共通
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#include "ev.h"
#include "alloc.h"

#define MAX(x, y) ((x) > (y) ? (x) : (y))

extern void getminmax3f(int, int, int, int, int, int, int, float ***, double *, double *);
extern void getminmax3d(int, int, int, int, int, int, int, double ***, double *, double *);

// 等高線図
void plot2d_contour(double x0, double y0, int nx, int ny,
	double **d, double dmin, double dmax, int color, int grid)
{
	// 等高線図（塗りつぶし）
	for (int i = 0; i < nx; i++) {
	for (int j = 0; j < ny; j++) {
		const double v = (d[i][j] - dmin) / (dmax - dmin);
		ev2d_setColorV(v, (color > 0));
		ev2d_fillRectangle(x0 + i, y0 + j, x0 + i + 1, y0 + j + 1);
	}
	}

	// 外枠
	if (color == 0) {
		ev2d_setColor(0, 0, 0);
		ev2d_drawRectangle(x0, y0, x0 + nx, y0 + ny);
	}

	// グリッド
	if (grid) {
		ev2d_setColor(160, 160, 160);
		for (int i = 1; i < nx; i++) {
			ev2d_drawLine(x0 + i, y0, x0 + i, y0 + ny);
		}
		for (int j = 1; j < ny; j++) {
			ev2d_drawLine(x0, y0 + j, x0 + nx, y0 + j);
		}
	}
}


// S行列等高線図, 全データ1ページ
void plot2d_s(int idata0, int idata1, int mx, int space, int Ndata, int NTx, int NRx,
	float ****S_r, float ****S_i, int dbspan, int grid, const char fn[])
{
	assert((idata0 < idata1) && (mx > 0) && (NTx > 0) && (NRx > 0));
	if (idata1 > Ndata) {
		idata1 = Ndata;
	}

	const int ifreq = 0;  // 第1周波数のみ
	const int ndata = idata1 - idata0;
	char str[BUFSIZ], fmt[BUFSIZ];

	// S行列絶対値
	double ***d;
	alloc3d(double, d, ndata, NTx, NRx)
	for (int n = 0; n < ndata; n++) {
		for (int itx = 0; itx < NTx; itx++) {
		for (int irx = 0; irx < NRx; irx++) {
			const double s_r = S_r[idata0 + n][ifreq][itx][irx];
			const double s_i = S_i[idata0 + n][ifreq][itx][irx];
			const double s = sqrt((s_r * s_r) + (s_i * s_i));
			d[n][itx][irx] = !dbspan ? s : 20 * log10(MAX(s, 1e-10));
		}
		}
	}

	// 最小・最大
	double dmin, dmax;
	getminmax3d(0, ndata, 0, NTx, 0, NRx, 0, d, &dmin, &dmax);
	dmin = dbspan ? (dmax - dbspan) : 0;

	// 開始
	const int nfig = idata1 - idata0;
	const int my = nfig / mx + (nfig % mx > 0);
	const double w = (mx * NTx) + ((mx + 1) * space);
	const double h = (my * NRx) + ((my + 2) * space);
	const double hgt = h / 60;
	ev2d_init(w, h);
	ev2d_newPage();

	// 等高線図, 1データ=1図
	int ifig = 0;
	for (int py = my - 1; py >= 0; py--) {
	for (int px = 0; px < mx; px++) {
		if (ifig < ndata) {
			// 原点
			const double x0 = (px * NTx) + ((px + 1) * space);
			const double y0 = (py * NRx) + ((py + 2) * space);

			// 等高線図
			plot2d_contour(x0, y0, NTx, NRx, d[ifig], dmin, dmax, 1, grid);

			// データ番号
			ev2d_setColor(0, 0, 0);
			sprintf(str, "%d", ifig + idata0 + 1);
			ev2d_drawString(x0, y0 + NRx + 0.2 * hgt, hgt, str);

			// 図番号++
			ifig++;
		}
	}
	}
	free(d);

	// 最小・最大
	strcpy(fmt, !dbspan
		? "fig=%d Tx=%d Rx=%d min=%.6f max=%.6f"
		: "fig=%d Tx=%d Rx=%d min=%.3fdB max=%.3fdB");
	sprintf(str, fmt, idata1 - idata0, NTx, NRx, dmin, dmax);
	ev2d_setColor(0, 0, 0);
	ev2d_drawString(space, 1.0 * hgt, 1.5 * hgt, str);

	// 出力
	ev2d_file(1, fn);
	ev2d_output();
}


// 3D配列のX面等高線図
void plot2d_X(int mx, int my, int istart, int istep, int space, int nx, int ny, int nz, float ***f, const char fn[])
{
	assert((mx > 0) && (my > 0) && (istart >= 0) && (istep > 0));
	assert((nx > 0) & (ny > 0) && (nz > 0));
	char str[BUFSIZ], fmt[BUFSIZ];

	// 最小・最大
	double dmin, dmax;
	getminmax3f(0, nx, 0, ny, 0, nz, 0, f, &dmin, &dmax);

	// 開始
	const double w = (mx * ny) + ((mx + 1) * space);
	const double h = (my * nz) + ((my + 3) * space);
	const double hgt = h / 60;
	ev2d_init(w, h);
	ev2d_newPage();

	// 等高線図, 1データ=1図
	int num = 0;
	int i = istart;
	for (int py = my - 1; py >= 0; py--) {
	for (int px = 0; px < mx; px++) {
		if (i < nx) {
			// 原点
			const double x0 = (px * ny) + ((px + 1) * space);
			const double y0 = (py * nz) + ((py + 3) * space);

			// 等高線図
			for (int j = 0; j < ny; j++) {
			for (int k = 0; k < nz; k++) {
				const double v = (f[i][j][k] - dmin) / (dmax - dmin);
				ev2d_setColorV(v, 1);
				ev2d_fillRectangle(x0 + j, y0 + k, x0 + j + 1, y0 + k + 1);
			}
			}

			// データ番号
			ev2d_setColor(0, 0, 0);
			sprintf(str, "%d", i + 1);
			ev2d_drawString(x0, y0 + nz + 0.2 * hgt, hgt, str);

			// 図++
			num++;
			i += istep;
		}
	}
	}

	// 最小・最大
	strcpy(fmt, "fig=%d nx=%d ny=%d nz=%d min=%.6f max=%.6f");
	sprintf(str, fmt, num, nx, ny, nz, dmin, dmax);
	ev2d_setColor(0, 0, 0);
	ev2d_drawString(space, 1.0 * hgt, 1.5 * hgt, str);

	// 出力
	ev2d_file(1, fn);
	ev2d_output();
}


// 3D配列のY面等高線図
void plot2d_Y(int mx, int my, int jstart, int jstep, int space, int nx, int ny, int nz, float ***f, const char fn[])
{
	assert((mx > 0) && (my > 0) && (jstart >= 0) && (jstep > 0));
	assert((nx > 0) & (ny > 0) && (nz > 0));
	char str[BUFSIZ], fmt[BUFSIZ];

	// 最小・最大
	double dmin, dmax;
	getminmax3f(0, nx, 0, ny, 0, nz, 0, f, &dmin, &dmax);

	// 開始
	const double w = (mx * ny) + ((mx + 1) * space);
	const double h = (my * nz) + ((my + 3) * space);
	const double hgt = h / 60;
	ev2d_init(w, h);
	ev2d_newPage();

	// 等高線図, 1データ=1図
	int num = 0;
	int j = jstart;
	for (int py = my - 1; py >= 0; py--) {
	for (int px = 0; px < mx; px++) {
		if (j < ny) {
			// 原点
			const double x0 = (px * ny) + ((px + 1) * space);
			const double y0 = (py * nz) + ((py + 3) * space);

			// 等高線図
			for (int i = 0; i < nx; i++) {
			for (int k = 0; k < nz; k++) {
				const double v = (f[i][j][k] - dmin) / (dmax - dmin);
				ev2d_setColorV(v, 1);
				ev2d_fillRectangle(x0 + i, y0 + k, x0 + i + 1, y0 + k + 1);
			}
			}

			// データ番号
			ev2d_setColor(0, 0, 0);
			sprintf(str, "%d", j + 1);
			ev2d_drawString(x0, y0 + nz + 0.2 * hgt, hgt, str);

			// 図++
			num++;
			j += jstep;
		}
	}
	}

	// 最小・最大
	strcpy(fmt, "fig=%d nx=%d ny=%d nz=%d min=%.6f max=%.6f");
	sprintf(str, fmt, num, nx, ny, nz, dmin, dmax);
	ev2d_setColor(0, 0, 0);
	ev2d_drawString(space, 1.0 * hgt, 1.5 * hgt, str);

	// 出力
	ev2d_file(1, fn);
	ev2d_output();
}


// 3D配列のZ面等高線図
void plot2d_Z(int mx, int my, int kstart, int kstep, int space, int nx, int ny, int nz, float ***f, const char fn[])
{
	assert((mx > 0) && (my > 0) && (kstart >= 0) && (kstep > 0));
	assert((nx > 0) & (ny > 0) && (nz > 0));
	char str[BUFSIZ], fmt[BUFSIZ];

	// 最小・最大
	double dmin, dmax;
	getminmax3f(0, nx, 0, ny, 0, nz, 0, f, &dmin, &dmax);

	// 開始
	const double w = (mx * nx) + ((mx + 1) * space);
	const double h = (my * ny) + ((my + 2) * space);
	const double hgt = h / 60;
	ev2d_init(w, h);
	ev2d_newPage();

	// 等高線図, 1データ=1図
	int num = 0;
	int k = kstart;
	for (int py = my - 1; py >= 0; py--) {
	for (int px = 0; px < mx; px++) {
		if (k < nz) {
			// 原点
			const double x0 = (px * nx) + ((px + 1) * space);
			const double y0 = (py * ny) + ((py + 2) * space);

			// 等高線図
			//plot2d_contour(x0, y0, NTx, NRx, d[idata], dmin, dmax, 1, grid);
			for (int i = 0; i < nx; i++) {
			for (int j = 0; j < ny; j++) {
				const double v = (f[i][j][k] - dmin) / (dmax - dmin);
				ev2d_setColorV(v, 1);
				ev2d_fillRectangle(x0 + i, y0 + j, x0 + i + 1, y0 + j + 1);
			}
			}

			// データ番号
			ev2d_setColor(0, 0, 0);
			sprintf(str, "%d", k + 1);
			ev2d_drawString(x0, y0 + ny + 0.2 * hgt, hgt, str);

			// 図++
			num++;
			k += kstep;
		}
	}
	}

	// 最小・最大
	strcpy(fmt, "fig=%d nx=%d ny=%d nz=%d min=%.6f max=%.6f");
	sprintf(str, fmt, num, nx, ny, nz, dmin, dmax);
	ev2d_setColor(0, 0, 0);
	ev2d_drawString(space, 1.0 * hgt, 1.5 * hgt, str);

	// 出力
	ev2d_file(1, fn);
	ev2d_output();
}
