/*
utils.c
fdtd2d/fdtd3d共通
*/

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <math.h>
#include <assert.h>

extern double urand(int, int64_t *);

// 給電電圧（微分ガウス）
float vfeed(float t, float freq)
{
	const double tw = 1.27 / freq;
	const double arg = (t - tw) / (tw / 4);
	const double v = sqrt(2) * exp(0.5) * arg * exp(-arg * arg);

	return (float)v;
}


// DFT係数
void dftfactor(int Niter, float freq, float Dt, float *fcos, float *fsin)
{
	const double pi = 4 * atan(1);
	const double omega = 2 * pi * freq;

	// 分母
	double sum_r = 0;
	double sum_i = 0;;
	for (int iter = 0; iter <= Niter; iter++) {
		const float t = iter * Dt;
		const float v = vfeed(t, freq);
		sum_r += cos(omega * t) * v;
		sum_i -= sin(omega * t) * v;
	}
	const float fnorm = (float)((sum_r * sum_r) + (sum_i * sum_i));

	// DFT係数
	for (int iter = 0; iter <= Niter; iter++) {
		const float t = iter * Dt;
		fcos[iter] = (float)cos(omega * t) / fnorm;
		fsin[iter] = (float)sin(omega * t) / fnorm;
	}
}


// PML減衰係数
void pmlfactor(int lpml, float *f)
{
	// PMLパラメーター
	const float pml_m = 2;
	const float pml_r0 = 1e-8f;

	const float c = (float)((pml_m + 1) / (2.0 * lpml) * log(1.0 / pml_r0));
	f[0] = 0;
	for (int n = 1; n < 2 * lpml + 1; n++) {
		f[n] = (float)(c * pow((n + 0) / (2.0 * lpml), pml_m));
	}
}


// ランダムに区間を取り出す
// n : 全体の大きさ
// nout : 両端の余白
// lmin, lmax : 長さの最小, 最大
// n0, n1 : 区間の下限, 上限
void getspan(int n, int nout, int lmin, int lmax, int64_t *seed, int *n0, int *n1)
{
	const double r1 = urand(3, seed);
	const double r2 = urand(3, seed);

	const int ls = (int)(lmin + r1 * (lmax - lmin) + 0.5);
	const int ds = n - 2 * nout - ls;
	const double pc = nout + (0.5 * ls) + (r2 * ds);

	*n0 = (int)(pc - (0.5 * ls) + 0.5);
	*n1 = (int)(pc + (0.5 * ls) + 0.5);

	assert((*n0 >= nout) && (*n0 <= n - nout) && (*n1 >= nout) && (*n1 <= n - nout));
}


// 最小・最大(3D配列, float)
void getminmax3f(int i0, int i1, int j0, int j1, int k0, int k1, int db, float ***d, double *dmin, double *dmax)
{
	*dmin = +1e12;
	*dmax = -1e12;
	for (int i = i0; i < i1; i++) {
	for (int j = j0; j < j1; j++) {
	for (int k = k0; k < k1; k++) {
		const double f = d[i][j][k];
		if (f < *dmin) {
			*dmin = f;
		}
		if (f > *dmax) {
			*dmax = f;
		}
	}
	}
	}

	// dB
	if (db) {
		const double eps = 1e-12;
		*dmax = 20 * log10(fmax(*dmax, eps));
		*dmin = *dmax - db;
	}
}


// 最小・最大(3D配列, double)
void getminmax3d(int i0, int i1, int j0, int j1, int k0, int k1, int db, double ***d, double *dmin, double *dmax)
{
	*dmin = +1e12;
	*dmax = -1e12;
	for (int i = i0; i < i1; i++) {
	for (int j = j0; j < j1; j++) {
	for (int k = k0; k < k1; k++) {
		const double f = d[i][j][k];
		if (f < *dmin) {
			*dmin = f;
		}
		if (f > *dmax) {
			*dmax = f;
		}
	}
	}
	}

	// dB
	if (db) {
		const double eps = 1e-12;
		*dmax = 20 * log10(fmax(*dmax, eps));
		*dmin = *dmax - db;
	}
}


// 最小・最大(4D配列, float)
void getminmax4f(int n0, int n1, int i0, int i1, int j0, int j1, int k0, int k1, int db, float ****d, double *dmin, double *dmax)
{
	*dmin = +1e12;
	*dmax = -1e12;
	for (int n = n0; n < n1; n++) {
	for (int i = i0; i < i1; i++) {
	for (int j = j0; j < j1; j++) {
	for (int k = k0; k < k1; k++) {
		const double f = d[n][i][j][k];
		if (f < *dmin) {
			*dmin = f;
		}
		if (f > *dmax) {
			*dmax = f;
		}
	}
	}
	}
	}

	// dB
	if (db) {
		const double eps = 1e-12;
		*dmax = 20 * log10(fmax(*dmax, eps));
		*dmin = *dmax - db;
	}
}


// 最小・最大(4D配列, double)
void getminmax4d(int n0, int n1, int i0, int i1, int j0, int j1, int k0, int k1, int db, double ****d, double *dmin, double *dmax)
{
	*dmin = +1e12;
	*dmax = -1e12;
	for (int n = n0; n < n1; n++) {
	for (int i = i0; i < i1; i++) {
	for (int j = j0; j < j1; j++) {
	for (int k = k0; k < k1; k++) {
		const double f = d[n][i][j][k];
		if (f < *dmin) {
			*dmin = f;
		}
		if (f > *dmax) {
			*dmax = f;
		}
	}
	}
	}
	}

	// dB
	if (db) {
		const double eps = 1e-12;
		*dmax = 20 * log10(fmax(*dmax, eps));
		*dmin = *dmax - db;
	}
}


// debug, S行列平均数値出力
void log_s1(int Ndata, int NFreq, int NTx, int NRx, float ****S_r, float ****S_i)
{
	double sum = 0;
	for (int idata = 0; idata < Ndata; idata++) {
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
	for (int itx = 0; itx < NTx; itx++) {
	for (int irx = 0; irx < NRx; irx++) {
		sum += fabs(S_r[idata][ifreq][itx][irx])
		     + fabs(S_i[idata][ifreq][itx][irx]);
	}
	}
	}
	}
	if (Ndata * NFreq * NTx * NRx > 0) {
		sum /= Ndata * NFreq * NTx * NRx;
	}

	printf("Save=%.8e\n", sum);
	fflush(stdout);
}


// debug, S行列数値出力, 見やすくするために1000倍
void log_s2(int Ndata, int NFreq, int NTx, int NRx, float ****S_r, float ****S_i)
{
	if ((Ndata == 0) || (NFreq == 0) || (NTx == 0) || (NRx == 0)) return;
	const int idata = 0;  // 最初のデータのみ
	const int ifreq = 0;  // 最初の周波数のみ

	for (int itx = 0; itx < NTx; itx++) {
	for (int irx = 0; irx < NRx; irx++) {
		printf("%4d %4d %10.4f %10.4f\n", itx, irx,
			1000 * S_r[idata][ifreq][itx][irx],
			1000 * S_i[idata][ifreq][itx][irx]);
	}
	}
	fflush(stdout);
}


// 出力ファイルサイズ[MB]
int file_size(int Ndata, int Mx, int My, int Mz, int NFreq, int NTx, int NRx, int size1, int size2)
{
	size_t size = 0;

	size += (size_t)Ndata * 2 * NFreq * NTx * NRx * size1;  // image
	size += (size_t)Ndata * 2 * Mx * My * Mz * size2;       // label

	return (int)(size * 1e-6) + 1;

}


// 複素数割り算
void cdiv(float a_r, float a_i, float b_r, float b_i, float *c_r, float *c_i)
{
	const float denom = (b_r * b_r) + (b_i * b_i);
	*c_r = ((a_r * b_r) + (a_i * b_i)) / denom;
	*c_i = ((a_i * b_r) - (a_r * b_i)) / denom;
}
