/*
misc.c
*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <assert.h>

#include "../fdtd2d/alloc.h"

extern float C;

// メッシュ
void make_mesh(int Nx, int Ny, int Nz, float Dx, float Dy, float Dz, float Eps0, float *Dt,
	float *Xn, float *Yn, float *Zn,
	float *dXn, float *dYn, float *dZn, float *dXc, float *dYc, float *dZc)
{
	// 格子点
	for (int i = 0; i <= Nx; i++) {
		Xn[i] = i * Dx;
	}
	for (int j = 0; j <= Ny; j++) {
		Yn[j] = j * Dy;
	}
	for (int k = 0; k <= Nz; k++) {
		Zn[k] = k * Dz;
	}
/*
	for (int i = 0; i <= Nx; i++) printf("%d %f\n", i, Xn[i]);
	for (int j = 0; j <= Ny; j++) printf("%d %f\n", j, Yn[j]);
	for (int k = 0; k <= Nz; k++) printf("%d %f\n", k, Zn[k]);
	for (int i = 0; i <  Nx; i++) printf("%d %f\n", i, Xc[i]);
	for (int j = 0; j <  Ny; j++) printf("%d %f\n", j, Yc[j]);
	for (int k = 0; k <  Nz; k++) printf("%d %f\n", k, Zc[k]);
*/
	// タイムステップ
	const float v = C / (float)sqrt(Eps0);
	*Dt = (float)(1 / sqrt(1 / (Dx * Dx) + 1 / (Dy * Dy) + 1 / (Dz * Dz)) / v);

	// C * Dt
	const float cdt = C * (*Dt);

	// 格子点に関する因子
	for (int i = 1; i < Nx; i++) {
		dXn[i] = cdt / ((Xn[i + 1] - Xn[i - 1]) / 2);
	}
	for (int j = 1; j < Ny; j++) {
		dYn[j] = cdt / ((Yn[j + 1] - Yn[j - 1]) / 2);
	}
	for (int k = 1; k < Nz; k++) {
		dZn[k] = cdt / ((Zn[k + 1] - Zn[k - 1]) / 2);
	}
	dXn[0]  = cdt / (Xn[1]  - Xn[0]);
	dYn[0]  = cdt / (Yn[1]  - Yn[0]);
	dZn[0]  = cdt / (Zn[1]  - Zn[0]);
	dXn[Nx] = cdt / (Xn[Nx] - Xn[Nx - 1]);
	dYn[Ny] = cdt / (Yn[Ny] - Yn[Ny - 1]);
	dZn[Nz] = cdt / (Zn[Nz] - Zn[Nz - 1]);

	// セル中心に関する因子
	for (int i = 0; i < Nx; i++) {
		dXc[i] = cdt / (Xn[i + 1] - Xn[i]);
	}
	for (int j = 0; j < Ny; j++) {
		dYc[j] = cdt / (Yn[j + 1] - Yn[j]);
	}
	for (int k = 0; k < Nz; k++) {
		dZc[k] = cdt / (Zn[k + 1] - Zn[k]);
	}
/*
	for (int i = 0; i <= Nx; i++) printf("%d %.5f\n", i, dXn[i]);
	for (int j = 0; j <= Ny; j++) printf("%d %.5f\n", j, dYn[j]);
	for (int k = 0; k <= Nz; k++) printf("%d %.5f\n", k, dZn[k]);
	for (int i = 0; i <  Nx; i++) printf("%d %.5f\n", i, dXc[i]);
	for (int j = 0; j <  Ny; j++) printf("%d %.5f\n", j, dYc[j]);
	for (int k = 0; k <  Nz; k++) printf("%d %.5f\n", k, dZc[k]);
*/
}


// 平滑化(内部領域, 7点平均, 複数回可)
void smooth(int Ndata, int Nx, int Ny, int Nz, int Nout, int Nsmot, float ****Epsr, float ****Sigm)
{
	// 作業配列
	float ***e, ***s;
	alloc3d(float, e, Nx, Ny, Nz)
	alloc3d(float, s, Nx, Ny, Nz)

	// データに関するループ
	for (int idata = 0; idata < Ndata; idata++) {
		// 複数回
		for (int m = 0; m < Nsmot; m++) {
			// 作業配列にコピー
			for (int i = 0; i < Nx; i++) {
			for (int j = 0; j < Ny; j++) {
			for (int k = 0; k < Nz; k++) {
				e[i][j][k] = Epsr[idata][i][j][k];
				s[i][j][k] = Sigm[idata][i][j][k];
			}
			}
			}
			// 更新
			for (int i = Nout; i < Nx - Nout; i++) {
			for (int j = Nout; j < Ny - Nout; j++) {
			for (int k = Nout; k < Nz - Nout; k++) {
				Epsr[idata][i][j][k] = (e[i][j][k]
					+ e[i - 1][j    ][k    ] + e[i + 1][j    ][k    ]
					+ e[i    ][j - 1][k    ] + e[i    ][j + 1][k    ]
					+ e[i    ][j    ][k - 1] + e[i    ][j    ][k + 1]) / 7;
				Sigm[idata][i][j][k] = (s[i][j][k]
					+ s[i - 1][j    ][k    ] + s[i + 1][j    ][k    ]
					+ s[i    ][j - 1][k    ] + s[i    ][j + 1][k    ]
					+ s[i    ][j    ][k - 1] + s[i    ][j    ][k + 1]) / 7;
			}
			}
			}
		}
	}

	// free
	free3d(e, Nx, Ny)
	free3d(s, Nx, Ny)
}


// メモリーサイズ[MB]
int memory_size(int nthread, int Ndata, int Nx, int Ny, int Nz, int NFreq, int NTx, int NRx, int ABC, int fig2)
{
	size_t mem = 0;

	mem += Ndata * Nx * Ny * Nz * 2 * sizeof(float);         // Epsr, Sigm
	mem += Ndata * NFreq * NTx * NRx * 2 * sizeof(float);    // S_r, S_i
	mem += nthread * Nx * Ny * Nz * 6 * sizeof(float);       // C1Ex, C2Ex, C1Ey, C2Ey, C1Ez, C2Ez
	mem += nthread * (Nx + 2 * ABC) * (Ny + 2 * ABC) * (Nz + 2 * ABC) * 6 * sizeof(float);  // Ex, Ey, Ez, Hx, Hy, Hz
	mem += (ABC > 0) * nthread * (Nx + 2 * ABC) * (Ny + 2 * ABC) * (Nz + 2 * ABC) * 12 * sizeof(float);  // Exy, Exz, Eyz, Eyx, Ezx, Ezy, Hxy, Hxz, Hyz, Hyx, Hzx, Hzy
	mem += (fig2 > 0) * Nx * Ny * Nz * 6 * sizeof(float);    // Ex_r, Ex_i, Ey_r, Ey_i, Ez_r, Ez_i
	mem += (fig2 > 0) * NTx * Nx * Ny * Nz * sizeof(float);  // Ea

	return (int)(mem * 1e-6) + 1;
}
