/*
comm.c
fdtd2d/fdtd3d共通
*/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <assert.h>

#ifdef _MPI
#include <mpi.h>
#endif

// init
void mpi_init(int argc, char *argv[], int *commsize, int *commrank)
{
#ifdef _MPI
	MPI_Init(&argc, &argv);
	MPI_Comm_size(MPI_COMM_WORLD, commsize);
	MPI_Comm_rank(MPI_COMM_WORLD, commrank);
#else
	argc = argc;
	argv = argv;
	*commsize = 1;
	*commrank = 0;
#endif
}


// close
void mpi_close(void)
{
#ifdef _MPI
	MPI_Finalize();
#endif
}


// S行列をrootに集める
void comm_S(int commsize, int commrank, int idata0, int idata1, int NFreq, int NTx, int NRx, float ****S_r, float ****S_i)
{
#ifdef _MPI
	MPI_Barrier(MPI_COMM_WORLD);  // 必須ではないが念のため

	if (commrank > 0) {
		// 送信(root以外)
		int i_sendbuf[2];
		i_sendbuf[0] = idata0;
		i_sendbuf[1] = idata1;
		MPI_Send(i_sendbuf, 2, MPI_INT, 0, 0, MPI_COMM_WORLD);
		const int count = 2 * (idata1 - idata0) * NFreq * NTx * NRx;
		float *f_sendbuf = (float *)malloc(count * sizeof(float));
		int num = 0;
		for (int idata = idata0; idata < idata1; idata++) {
		for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		for (int itx = 0; itx < NTx; itx++) {
		for (int irx = 0; irx < NRx; irx++) {
			f_sendbuf[num++] = S_r[idata][ifreq][itx][irx];
			f_sendbuf[num++] = S_i[idata][ifreq][itx][irx];
		}
		}
		}
		}
		assert(num == count);
		MPI_Send(f_sendbuf, count, MPI_FLOAT, 0, 0, MPI_COMM_WORLD);
		free(f_sendbuf);
	}
	else {
		// 受信(root)
		MPI_Status status;
		int i_recvbuf[2];
		for (int rank = 1; rank < commsize; rank++) {
			MPI_Recv(i_recvbuf, 2, MPI_INT, rank, 0, MPI_COMM_WORLD, &status);
			const int i_data0 = i_recvbuf[0];
			const int i_data1 = i_recvbuf[1];
			const int count = 2 * (i_data1 - i_data0) * NFreq * NTx * NRx;
			float *f_recvbuf = (float *)malloc(count * sizeof(float));
			MPI_Recv(f_recvbuf, count, MPI_FLOAT, rank, 0, MPI_COMM_WORLD, &status);
			int num = 0;
			for (int idata = i_data0; idata < i_data1; idata++) {
			for (int ifreq = 0; ifreq < NFreq; ifreq++) {
			for (int itx = 0; itx < NTx; itx++) {
			for (int irx = 0; irx < NRx; irx++) {
				S_r[idata][ifreq][itx][irx] = f_recvbuf[num++];
				S_i[idata][ifreq][itx][irx] = f_recvbuf[num++];
			}
			}
			}
			}
			assert(num == count);
			free(f_recvbuf);
		}
	}
#else
	// dummy
	commsize = commsize;
	commrank = commrank;
	idata0 = idata0;
	idata1 = idata1;
	NFreq = NFreq;
	NTx = NTx;
	NRx = NRx;
	S_r = S_r;
	S_i = S_i;
#endif
}


// CPU時間
double cputime(void)
{
#ifdef _MPI
	MPI_Barrier(MPI_COMM_WORLD);
	return MPI_Wtime();
#else
#ifdef _WIN32
	return (double)clock() / CLOCKS_PER_SEC;
#else
	struct timespec ts;
	clock_gettime(CLOCK_REALTIME, &ts);
	return ts.tv_sec + (ts.tv_nsec * 1e-9);
#endif  // _WIN32
#endif  // _MPI
}
