/*
geometry.c
誘電率、導電率分布
*/

#include <stdio.h>
#include <stdint.h>
#include <math.h>
#include <assert.h>

#include "alloc.h"

extern double urand(int, int64_t *);
extern void getspan(int, int, int, int, int64_t *, int *, int *);
extern void smooth(int, int, int, int, int, float ***, float ***);

// 学習データ
void geometry(int Ndata, int Nx, int Ny, int Nout, int Sdif, int Nsmot,
	const int Nobj[], const int Lobj[][3], const int Mean[], float gRatio,
	float Eps0, float Sig0, const float Eps[], const float Sig[], int64_t Seed,
	float ***Epsr, float ***Sigm)
{
	assert((Ndata >= 0) && (Nx > 0) && (Ny > 0) && (Nout > 0) && (Seed > 0));

	// 全体を背景媒質で初期化
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < Nx; i++) {
		for (int j = 0; j < Ny; j++) {
			Epsr[idata][i][j] = Eps0;
			Sigm[idata][i][j] = Sig0;
		}
		}
	}

	// 誘電率、導電率代入
	// データに関するループ
	for (int idata = Sdif; idata < Ndata; idata++) {
		// 誘電体に関するループ
		const double r1 = urand(3, &Seed);
		const int nobj = Nobj[0] + (int)(r1 * (Nobj[1] - Nobj[0] + 1));
		for (int iobj = 0; iobj < nobj; iobj++) {
			// 誘電率、導電率
			const double r2 = urand(3, &Seed);
			const double r3 = urand(3, &Seed);
			const float ep = Eps[0] + (Eps[1] - Eps[0]) * (float)r2;
			const float si = Sig[0] + (Sig[1] - Sig[0]) * (float)r3;
			// 誘電体セル座標（複数個平均:中央に集める）
			int i0a = 0, i1a = 0, j0a = 0, j1a = 0;
			int i0b = 0, i1b = 0, j0b = 0, j1b = 0;
			int i0c = 0, i1c = 0, j0c = 0, j1c = 0;
			int i0d = 0, i1d = 0, j0d = 0, j1d = 0;
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0a, &i1a);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0a, &j1a);
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0b, &i1b);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0b, &j1b);
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0c, &i1c);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0c, &j1c);
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0d, &i1d);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0d, &j1d);
			const double r4 = urand(3, &Seed);
			const int mean = Mean[0] + (int)(r4 * (Mean[1] - Mean[0] + 1));
			double ri0, ri1, rj0, rj1;
			if      (mean == 1) {
				ri0 = i0a;
				ri1 = i1a;
				rj0 = j0a;
				rj1 = j1a;
			}
			else if (mean == 2) {
				ri0 = (i0a + i0b) / 2.0;
				ri1 = (i1a + i1b) / 2.0;
				rj0 = (j0a + j0b) / 2.0;
				rj1 = (j1a + j1b) / 2.0;
			}
			else if (mean == 3) {
				ri0 = (i0a + i0b + i0c) / 3.0;
				ri1 = (i1a + i1b + i1c) / 3.0;
				rj0 = (j0a + j0b + j0c) / 3.0;
				rj1 = (j1a + j1b + j1c) / 3.0;
			}
			else {
				ri0 = (i0a + i0b + i0c + i0d) / 4.0;
				ri1 = (i1a + i1b + i1c + i1d) / 4.0;
				rj0 = (j0a + j0b + j0c + j0d) / 4.0;
				rj1 = (j1a + j1b + j1c + j1d) / 4.0;
			}
			const int i0 = (int)(ri0 + 0.5);
			const int i1 = (int)(ri1 + 0.5);
			const int j0 = (int)(rj0 + 0.5);
			const int j1 = (int)(rj1 + 0.5);
			assert((i0 < i1) && (j0 < j1));
			// 代入
			const double r5 = urand(3, &Seed);
			if (r5 < gRatio) {
				// 長方形内部
				for (int i = i0; i < i1; i++) {
				for (int j = j0; j < j1; j++) {
					Epsr[idata][i][j] = ep;
					Sigm[idata][i][j] = si;
				}
				}
			}
			else {
				// 楕円内部
				const double xc = (i0 + i1) / 2.0;
				const double yc = (j0 + j1) / 2.0;
				const double xr = (i1 - i0) / 2.0;
				const double yr = (j1 - j0) / 2.0;
				for (int i = i0; i < i1; i++) {
				for (int j = j0; j < j1; j++) {
					const double x = i + 0.5;
					const double y = j + 0.5;
					const double ex = (x - xc) / xr;
					const double ey = (y - yc) / yr;
					if ((ex * ex) + (ey * ey) < 1) {
						Epsr[idata][i][j] = ep;
						Sigm[idata][i][j] = si;
					}
				}
				}
			}
		}
	}

	// 平滑化
	if (Nsmot) {
		smooth(Ndata, Nx, Ny, Nout, Nsmot, Epsr, Sigm);
	}
/*
	// debug
	double esum = 0, ssum = 0;
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < Nx; i++) {
		for (int j = 0; j < Ny; j++) {
			esum += fabs(Epsr[idata][i][j]);
			ssum += fabs(Sigm[idata][i][j]);
		}
		}
	}
	printf("%d %d %d %e %e\n", Ndata, Nx, Ny, esum / (Ndata * Nx * Ny), ssum / (Ndata * Nx * Ny));
*/
}


// 長方形（テストデータ）
static void rectangle(int Nout, int i0, int j0, int i1, int j1, float **e, float **s, float epsr, float sigm)
{
	for (int i = Nout + i0; i < Nout + i1; i++) {
	for (int j = Nout + j0; j < Nout + j1; j++) {
		e[i][j] = epsr;
		s[i][j] = sigm;
	}
	}
}


// 三角形（テストデータ）
static void triangle(int Nout, int i0, int j0, int i1, int j1, float **e, float **s, float epsr, float sigm)
{
	const double eps = 1e-6;
	for (int i = i0; i < i1; i++) {
	for (int j = j0; j < j1; j++) {
		const double h = (j + 0.5 - j0) / (j1 - j0);
		if ((h > -eps) && (h < 1 + eps)) {
			const double di = (i1 - i0) / 2.0 * (1 - h);
			if (fabs(i + 0.5 - (i0 + i1) / 2.0) < di * (1 + eps)) {
				e[i + Nout][j + Nout] = epsr;
				s[i + Nout][j + Nout] = sigm;
			}
		}
	}
	}
}


// 楕円（テストデータ）
static void ellipse(int Nout, int i0, int j0, int i1, int j1, float **e, float **s, float epsr, float sigm)
{
	const double eps = 1e-6;
	const double cx = (i0 + i1) / 2.0;
	const double cy = (j0 + j1) / 2.0;
	const double rx = (i1 - i0) / 2.0;
	const double ry = (j1 - j0) / 2.0;
	for (int i = i0; i < i1; i++) {
	for (int j = j0; j < j1; j++) {
		const double x = i + 0.5;
		const double y = j + 0.5;
		const double dx = x - cx;
		const double dy = y - cy;
		if ((dx * dx) / (rx * rx) + (dy * dy) / (ry * ry) < (1 + eps)) {
			e[i + Nout][j + Nout] = epsr;
			s[i + Nout][j + Nout] = sigm;
		}
	}
	}
}


// テストデータ作成
void geometry_test(int Ndata, int Nx, int Ny, int Nout, int Sdif, int Nsmot,
	 float Eps0, float Sig0, const float Eps[], const float Sig[], float ***Epsr, float ***Sigm)
{
	assert((Nx > 0) && (Ny > 0) && (Nout > 0) && (Nsmot >= 0));
	const double eps = 1e-6;
	int i0, j0, i1, j1;

	// 全体を背景媒質で初期化
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < Nx; i++) {
		for (int j = 0; j < Ny ; j++) {
			Epsr[idata][i][j] = Eps0;
			Sigm[idata][i][j] = Sig0;
		}
		}
	}

	// 内部セル数
	const int mx = Nx - (2 * Nout);
	const int my = Ny - (2 * Nout);

	// 比誘電率, 導電率（中間値）
	const float epsr = (Eps[0] + Eps[1]) / 2;
	const float sigm = (Sig[0] + Sig[1]) / 2;

	// データ番号
	int idata = Sdif;

	// (1) 長方形1個
	i0 = (int)(0.3 * mx + 0.5 - eps);
	j0 = (int)(0.3 * my + 0.5 - eps);
	i1 = (int)(0.7 * mx + 0.5 + eps);
	j1 = (int)(0.7 * my + 0.5 + eps);
	rectangle(Nout, i0, j0, i1, j1, Epsr[idata], Sigm[idata], epsr, sigm);
	idata++;

	// (2) 長方形2個
	// 1/2
	i0 = (int)(0.2 * mx + 0.5 - eps);
	j0 = (int)(0.2 * my + 0.5 - eps);
	i1 = (int)(0.6 * mx + 0.5 + eps);
	j1 = (int)(0.6 * my + 0.5 + eps);
	rectangle(Nout, i0, j0, i1, j1, Epsr[idata], Sigm[idata], epsr, sigm);
	// 2/2
	i0 = (int)(0.4 * mx + 0.5 - eps);
	j0 = (int)(0.4 * my + 0.5 - eps);
	i1 = (int)(0.8 * mx + 0.5 + eps);
	j1 = (int)(0.8 * my + 0.5 + eps);
	rectangle(Nout, i0, j0, i1, j1, Epsr[idata], Sigm[idata], epsr, sigm);
	idata++;

	// (3) 三角形
	i0 = (int)(0.2 * mx + 0.5 - eps);
	j0 = (int)(0.2 * my + 0.5 - eps);
	i1 = (int)(0.8 * mx + 0.5 + eps);
	j1 = (int)(0.8 * my + 0.5 + eps);
	triangle(Nout, i0, j0, i1, j1, Epsr[idata], Sigm[idata], epsr, sigm);
	idata++;

	// (4) 楕円
	i0 = (int)(0.2 * mx + 0.5 - eps);
	j0 = (int)(0.2 * my + 0.5 - eps);
	i1 = (int)(0.8 * mx + 0.5 + eps);
	j1 = (int)(0.8 * my + 0.5 + eps);
	ellipse(Nout, i0, j0, i1, j1, Epsr[idata], Sigm[idata], epsr, sigm);
	idata++;

	// (5) 長方形1個+楕円1個
	i0 = (int)(0.3 * mx + 0.5 - eps);
	j0 = (int)(0.1 * my + 0.5 - eps);
	i1 = (int)(0.7 * mx + 0.5 + eps);
	j1 = (int)(0.5 * my + 0.5 + eps);
	rectangle(Nout, i0, j0, i1, j1, Epsr[idata], Sigm[idata], epsr, sigm);
	i0 = (int)(0.3 * mx + 0.5 - eps);
	j0 = (int)(0.5 * my + 0.5 - eps);
	i1 = (int)(0.7 * mx + 0.5 + eps);
	j1 = (int)(0.9 * my + 0.5 + eps);
	ellipse(Nout, i0, j0, i1, j1, Epsr[idata], Sigm[idata], epsr, sigm);
	idata++;

	assert(idata == Ndata);

	// 平滑化
	if (Nsmot) {
		smooth(Ndata, Nx, Ny, Nout, Nsmot, Epsr, Sigm);
	}
}
