/*
geometry2.c
誘電率、導電率分布 (2)
数値ファントム
*/

#include <stdio.h>
#include <stdint.h>
#include <math.h>
#include <assert.h>

#include "alloc.h"

extern void phantom(const char [], int, int, int, int ***);
extern double urand(int, int64_t *);
extern void getspan(int, int, int, int, int64_t *, int *, int *);
extern void smooth(int, int, int, int, int, float ***, float ***);
//extern void plot2d_X(int, int, int, int, int, int, int, int, float ***, const char []);
//extern void plot2d_Y(int, int, int, int, int, int, int, int, float ***, const char []);
//extern void plot2d_Z(int, int, int, int, int, int, int, int, float ***, const char []);

// 教師データ作成（誘電率、導電率）
void geometry2(int Ndata, int Nx, int Ny, int Nout, int Sdif, int Nsmot,
	const int Nobj[], const int Lobj[][3], const int Mean[], float gRatio, float mRatio,
	float Eps0, float Sig0, const float Eps[], const float Sig[], int64_t Seed,
	float ***Epsr, float ***Sigm)
{
	assert((Ndata >= 0) && (Nx > 0) && (Ny > 0) && (Nout > 0));

	// 半円の中心と半径, この外は背景媒質
	const double xrc = Nx / 2.0;
	const double yrc = Ny / 2.0;
	const double rad = (Nx + Ny) / 4.0 - Nout;

	// 全体を背景媒質で初期化
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < Nx; i++) {
		for (int j = 0; j < Ny; j++) {
			Epsr[idata][i][j] = Eps0;
			Sigm[idata][i][j] = Sig0;
		}
		}
	}

	// 誘電率、導電率代入
	// データに関するループ
	for (int idata = Sdif; idata < Ndata; idata++) {
		// 誘電体に関するループ
		const double r1 = urand(3, &Seed);
		const int nobj = Nobj[0] + (int)(r1 * (Nobj[1] - Nobj[0] + 1));
		for (int iobj = 0; iobj < nobj; iobj++) {
			// 誘電率、導電率
			const double r2 = urand(3, &Seed);
			const int tid = (r2 < mRatio) ? 2 : 4;  // 乳腺(=2)とがん(=4)に配分
			const float ep = Eps[tid];
			const float si = Sig[tid];
			// 誘電体セル座標（複数個平均:中央に集める）
			int i0a = 0, i1a = 0, j0a = 0, j1a = 0;
			int i0b = 0, i1b = 0, j0b = 0, j1b = 0;
			int i0c = 0, i1c = 0, j0c = 0, j1c = 0;
			int i0d = 0, i1d = 0, j0d = 0, j1d = 0;
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0a, &i1a);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0a, &j1a);
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0b, &i1b);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0b, &j1b);
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0c, &i1c);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0c, &j1c);
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0d, &i1d);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0d, &j1d);
			const double r3 = urand(3, &Seed);
			const int mean = Mean[0] + (int)(r3 * (Mean[1] - Mean[0] + 1));
			double ri0, ri1, rj0, rj1;
			if      (mean == 1) {
				ri0 = i0a;
				ri1 = i1a;
				rj0 = j0a;
				rj1 = j1a;
			}
			else if (mean == 2) {
				ri0 = (i0a + i0b) / 2.0;
				ri1 = (i1a + i1b) / 2.0;
				rj0 = (j0a + j0b) / 2.0;
				rj1 = (j1a + j1b) / 2.0;
			}
			else if (mean == 3) {
				ri0 = (i0a + i0b + i0c) / 3.0;
				ri1 = (i1a + i1b + i1c) / 3.0;
				rj0 = (j0a + j0b + j0c) / 3.0;
				rj1 = (j1a + j1b + j1c) / 3.0;
			}
			else {
				ri0 = (i0a + i0b + i0c + i0d) / 4.0;
				ri1 = (i1a + i1b + i1c + i1d) / 4.0;
				rj0 = (j0a + j0b + j0c + j0d) / 4.0;
				rj1 = (j1a + j1b + j1c + j1d) / 4.0;
			}
			const int i0 = (int)(ri0 + 0.5);
			const int i1 = (int)(ri1 + 0.5);
			const int j0 = (int)(rj0 + 0.5);
			const int j1 = (int)(rj1 + 0.5);
			assert((i0 < i1) && (j0 < j1));
			// 代入
			const double r4 = urand(3, &Seed);
			if (r4 < gRatio) {
				// 長方形内部 かつ 半円内部
				for (int i = i0; i < i1; i++) {
				for (int j = j0; j < j1; j++) {
					const double x = i + 0.5;
					const double y = j + 0.5;
					const double dx = x - xrc;
					const double dy = y - yrc;
					if ((dx * dx) + (dy * dy) < (rad * rad)) {
						Epsr[idata][i][j] = ep;
						Sigm[idata][i][j] = si;
					}
				}
				}
			}
			else {
				// 楕円内部 かつ 半円内部
				const double xc = (i0 + i1) / 2.0;
				const double yc = (j0 + j1) / 2.0;
				const double xr = (i1 - i0) / 2.0;
				const double yr = (j1 - j0) / 2.0;
				for (int i = i0; i < i1; i++) {
				for (int j = j0; j < j1; j++) {
					const double x = i + 0.5;
					const double y = j + 0.5;
					const double dx = x - xrc;
					const double dy = y - yrc;
					const double ex = (x - xc) / xr;
					const double ey = (y - yc) / yr;
					if (((ex * ex) + (ey * ey) < 1) &&
					    ((dx * dx) + (dy * dy) < (rad * rad))) {
						Epsr[idata][i][j] = ep;
						Sigm[idata][i][j] = si;
					}
				}
				}
			}
		}
	}

	// 平滑化
	if (Nsmot) {
		smooth(Ndata, Nx, Ny, Nout, Nsmot, Epsr, Sigm);
	}
/*
	// debug
	double esum = 0, ssum = 0;
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < Nx; i++) {
		for (int j = 0; j < Ny; j++) {
			esum += fabs(Epsr[idata][i][j]);
			ssum += fabs(Sigm[idata][i][j]);
		}
		}
	}
	printf("%d %d %d %e %e\n", Ndata, Nx, Ny, esum / (Ndata * Nx * Ny), ssum / (Ndata * Nx * Ny));
*/
}


// 数値ファントム
void geometry2_test(int kstart, int kstep, int Mprec, int Ndata, int Nx, int Ny, int Nout, int Sdif, int Nsmot,
	float Eps0, float Sig0, const float Eps[], const float Sig[], float ***Epsr, float ***Sigm)
{
	// ファイル名
	const char fn[] = "../phantom/mediumLabelList_Reso1mm_Case1.txt";
	//const char fn[] = "../phantom/mediumLabelList_Reso1mm_Case2.txt";
	//const char fn[] = "../phantom/mediumLabelList_Reso1mm_Case3.txt";

	// ファイルのセル数（固定）
	const int nx = 100;
	const int ny = 100;
	const int nz = 50;

	// 数値ファントムを読み込む
	int ***map;
	alloc3d(int, map, nx, ny, nz)
	phantom(fn, nx, ny, nz, map);

	// 組織IDを電気定数に変換する
	float ***epsr, ***sigm;
	alloc3d(float, epsr, nx, ny, nz)
	alloc3d(float, sigm, nx, ny, nz)
	for (int i = 0; i < nx; i++) {
	for (int j = 0; j < ny; j++) {
	for (int k = 0; k < nz; k++) {
		const int id = map[i][j][k];
		assert((id >= 0) && (id <= 5));
		epsr[i][j][k] = Eps[id];
		sigm[i][j][k] = Sig[id];
	}
	}
	}

	// 全体を背景媒質で初期化
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < Nx; i++) {
		for (int j = 0; j < Ny; j++) {
			Epsr[idata][i][j] = Eps0;
			Sigm[idata][i][j] = Sig0;
		}
		}
	}

	// 間引き
	const int ms = Mprec;  // 間引き
	const int mx = Nx - (2 * Nout);
	const int my = Ny - (2 * Nout);
	assert((mx == nx / ms) && (my == ny / ms));
	for (int idata = Sdif; idata < Ndata; idata++) {
		const int k = kstart + (idata - Sdif) * kstep;
		assert((k >= 0) && (k < nz));
		for (int i = 0; i < mx; i++) {
		for (int j = 0; j < my; j++) {
			float esum = 0, ssum = 0;
			for (int is = 0; is < ms; is++) {
			for (int js = 0; js < ms; js++) {
				const int im = (i * ms) + is;
				const int jm = (j * ms) + js;
				assert((im >= 0) && (im < nx) && (jm >= 0) && (jm < ny));
				esum += epsr[im][jm][k];
				ssum += sigm[im][jm][k];
			}
			}
			Epsr[idata][Nout + i][Nout + j] = esum / (ms * ms);
			Sigm[idata][Nout + i][Nout + j] = ssum / (ms * ms);
		}
		}
	}

	// 平滑化
	if (Nsmot) {
		smooth(Ndata, Nx, Ny, Nout, Nsmot, Epsr, Sigm);
	}
/*
	// debug
	double esum = 0, ssum = 0;
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < mx; i++) {
		for (int j = 0; j < my; j++) {
			esum += Epsr[idata][i + Nout][j + Nout];
			ssum += Sigm[idata][i + Nout][j + Nout];
		}
		}
	}
	printf("%d %d %e %e\n", mx, my, esum / (Ndata * mx * my), ssum / (Ndata * mx * my));
*/
	// 図形表示
	//plot2d_X(10, 10, 0, 1, 8, nx, ny, nz, epsr, "X.ev2");
	//plot2d_X(10, 10, 0, 1, 8, nx, ny, nz, sigm, "X.ev2");
	//plot2d_Y(10, 10, 0, 1, 8, nx, ny, nz, epsr, "Y.ev2");
	//plot2d_Y(10, 10, 0, 1, 8, nx, ny, nz, sigm, "Y.ev2");
	//plot2d_Z(10, 5, 0, 1, 8, nx, ny, nz, epsr, "Z.ev2");
	//plot2d_Z(10, 5, 0, 1, 8, nx, ny, nz, sigm, "Z.ev2");

	//plot2d_X(1, 1, 1, 1, 8, 1, Nx, Ny, Epsr, "Xe.ev2");
	//plot2d_X(1, 1, 1, 1, 8, 1, Nx, Ny, Sigm, "Xs.ev2");

	// free
	free3d(map, nx, ny)
	free3d(epsr, nx, ny)
	free3d(sigm, nx, ny)
}
