/*
misc.c
*/

#include <stdio.h>
#include <math.h>

#include "alloc.h"

extern float C;

// メッシュ
void make_mesh(int Nx, int Ny, float Dx, float Dy, float Eps0, float *Dt,
	float *Xn, float *Yn,
	float *dXn, float *dYn, float *dXc, float *dYc)
{
	// 格子点
	for (int i = 0; i <= Nx; i++) {
		Xn[i] = i * Dx;
	}
	for (int j = 0; j <= Ny; j++) {
		Yn[j] = j * Dy;
	}

	// タイムステップ
	const float v = C / (float)sqrt(Eps0);
	*Dt = 1 / (float)(sqrt(1 / (Dx * Dx) + 1 / (Dy * Dy)) * v);

	// C * Dt
	const float cdt = C * (*Dt);

	// 格子点に関する因子
	for (int i = 1; i < Nx; i++) {
		dXn[i] = cdt / ((Xn[i + 1] - Xn[i - 1]) / 2);
	}
	for (int j = 1; j < Ny; j++) {
		dYn[j] = cdt / ((Yn[j + 1] - Yn[j - 1]) / 2);
	}
	dXn[0]  = cdt / (Xn[1]  - Xn[0]);
	dYn[0]  = cdt / (Yn[1]  - Yn[0]);
	dXn[Nx] = cdt / (Xn[Nx] - Xn[Nx - 1]);
	dYn[Ny] = cdt / (Yn[Ny] - Yn[Ny - 1]);

	// セル中心に関する因子
	for (int i = 0; i < Nx; i++) {
		dXc[i] = cdt / (Xn[i + 1] - Xn[i]);
	}
	for (int j = 0; j < Ny; j++) {
		dYc[j] = cdt / (Yn[j + 1] - Yn[j]);
	}
}


// 平滑化(内部領域, 5点平均, 複数回可)
void smooth(int Ndata, int Nx, int Ny, int Nout, int Nsmot, float ***Epsr, float ***Sigm)
{
	// 作業配列
	float **e, **s;
	alloc2d(float, e, Nx, Ny)
	alloc2d(float, s, Nx, Ny)

	// データに関するループ
	for (int idata = 0; idata < Ndata; idata++) {
		// 複数回
		for (int m = 0; m < Nsmot; m++) {
			// 作業配列にコピー
			for (int i = 0; i < Nx; i++) {
			for (int j = 0; j < Ny; j++) {
				e[i][j] = Epsr[idata][i][j];
				s[i][j] = Sigm[idata][i][j];
			}
			}
			// 更新
			for (int i = Nout; i < Nx - Nout; i++) {
			for (int j = Nout; j < Ny - Nout; j++) {
				Epsr[idata][i][j] = (e[i][j]
					+ e[i - 1][j    ] + e[i + 1][j    ]
					+ e[i    ][j - 1] + e[i    ][j + 1]) / 5;
				Sigm[idata][i][j] = (s[i][j]
					+ s[i - 1][j    ] + s[i + 1][j    ]
					+ s[i    ][j - 1] + s[i    ][j + 1]) / 5;
			}
			}
		}
	}

	// free
	free2d(e, Nx)
	free2d(s, Nx)
}


// メモリーサイズ[MB]
int memory_size(int nthread, int Ndata, int Nx, int Ny, int NFreq, int NTx, int NRx, int ABC, int fig2)
{
	size_t mem = 0;

	mem += Ndata * Nx * Ny * 2 * sizeof(float);            // Epsr, Sigm
	mem += Ndata * NFreq * NTx * NRx * 2 * sizeof(float);  // S_r, S_i
	mem += nthread * Nx * Ny * 2 * sizeof(float);          // C1Ez, C2Ez
	mem += nthread * (Nx + 2 * ABC) * (Ny + 2 * ABC) * 3 * sizeof(float);  // Ez, Hx, Hy
	mem += (ABC > 0) * nthread * (Nx + 2 * ABC) * (Ny + 2 * ABC) * 2 * sizeof(float);  // Ezx, Ezy
	mem += fig2 * NTx * Nx * Ny * sizeof(float);           // Ea

	return (int)(mem * 1e-6) + 1;
}
