/*
solve.c
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#include "alloc.h"

static void setupFactor(int, int, float, float **, float **, float, float, float **, float **);
static void initfield(int, int, int, int, int, float **, float **, float **, float **, float **, float **, float **, float *, float *, float **, float **, int, float **, float **);
static void dft1(int, int, int, int, int *, int *, float, float **, float **, float **, float *, float *, float **, float **);
static void dft2(int, int, int, float **, float, float, float **, float **);
static void average(int, int, int, float **, float **, float **, float []);
static void efield(int, int, float **, float **, float **);

extern void dftfactor(int, float, float, float *, float *);
extern void updateEz(int, int, int, float **, float **, float **, float *, float *, float **, float **);
extern void updateHx(int, int, int, float **, float **, float *);
extern void updateHy(int, int, int, float **, float **, float *);
extern void murEz(int, int, float, float *, float *, float **, float **, float **, float);
extern void pmlEz(int, int, int, float **, float **, float **, float **, float **, float *, float *, float *, float *);
extern void pmlHx(int, int, int, float **, float **, float *, float *);
extern void pmlHy(int, int, int, float **, float **, float *, float *);
extern void pmlfactor(int, float *);
extern void setupPmlEz(int, int, int, float, float, float *, float *, float *, float *, float *, float *, float *, float *, float *);
extern void setupPmlHx(int, int, float, float, float *, float *, float *, float *, float *);
extern void setupPmlHy(int, int, float, float, float *, float *, float *, float *, float *);
extern float vfeed(float, float);
extern void cdiv(float, float, float, float, float *, float *);

extern float C;

void solve(
	int itx, int Niter, int Nx, int Ny, int NFreq, const float Freq[], float Dt, int ABC,
	int *iTx, int *jTx, int NRx, int *iRx, int *jRx,
	float **Ez, float **Hx, float **Hy, float **Ezx, float **Ezy,
	float *Xn, float *Yn, float *dXn, float *dYn, float *dXc, float *dYc,
	float **epsr, float **sigm, float Eps0, float Sig0,
	float **C1Ez, float **C2Ez,
	float **Ezmurx, float **Ezmury,
	float ***s_r, float ***s_i,
	int ilog, int iout, float **Ez_r, float **Ez_i, float **Ea)
{
	// DFT係数
	float **fcos, **fsin;
	alloc2d(float, fcos, NFreq, Niter + 1)
	alloc2d(float, fsin, NFreq, Niter + 1)
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		dftfactor(Niter, Freq[ifreq], Dt, fcos[ifreq], fsin[ifreq]);
	}

	// 送受信電界
	float *etx_r, *etx_i;
	float **erx_r, **erx_i;
	alloc1d(float, etx_r, NFreq)
	alloc1d(float, etx_i, NFreq)
	alloc2d(float, erx_r, NFreq, NRx)
	alloc2d(float, erx_i, NFreq, NRx)

	// FDTD係数
	setupFactor(Nx, Ny, Dt, epsr, sigm, Eps0, Sig0, C1Ez, C2Ez);

	// PML係数
	float *fpml, *fezx, *fezy, *fhx, *fhy, *dezx, *dezy, *dhx, *dhy;
	alloc1d(float, fpml,      1 + 2 * ABC)
	alloc1d(float, fezx, Nx + 1 + 2 * ABC)
	alloc1d(float, fezy, Ny + 1 + 2 * ABC)
	alloc1d(float, fhx,  Ny + 0 + 2 * ABC)
	alloc1d(float, fhy,  Nx + 0 + 2 * ABC)
	alloc1d(float, dezx, Nx + 1 + 2 * ABC)
	alloc1d(float, dezy, Ny + 1 + 2 * ABC)
	alloc1d(float, dhx,  Ny + 0 + 2 * ABC)
	alloc1d(float, dhy,  Nx + 0 + 2 * ABC)
	if (ABC) {
		pmlfactor(ABC, fpml);
		setupPmlEz(ABC, Nx, Ny, Dt, Eps0, Xn, Yn, dXn, dYn, fpml, fezx, fezy, dezx, dezy);
		setupPmlHx(ABC, Ny, Dt, Eps0, Yn, dYc, fpml, fhx, dhx);
		setupPmlHy(ABC, Nx, Dt, Eps0, Xn, dXc, fpml, fhy, dhy);
	}

	// 電磁界初期化
	initfield(ABC, Nx, Ny, NRx, NFreq, Ez, Hx, Hy, Ezmurx, Ezmury, Ezx, Ezy, etx_r, etx_i, erx_r, erx_i, iout, Ez_r, Ez_i);

	// FDTD反復計算
	float t = 0;
	for (int iter = 0; iter <= Niter; iter++) {
		// Ez更新
		updateEz(ABC, Nx, Ny, Ez, Hx, Hy, dXn, dYn, C1Ez, C2Ez);

		// 給電点
		const float freq = (Freq[0] + Freq[NFreq - 1]) / 2;
		const float etx = vfeed(t, freq);
		Ez[iTx[itx] + ABC][jTx[itx] + ABC] = etx;

		// ABC
		if (ABC > 0) {
			pmlEz(ABC, Nx, Ny, Ez, Hx, Hy, Ezx, Ezy, fezx, fezy, dezx, dezy);
		}
		else {
			murEz(Nx, Ny, Dt, Xn, Yn, Ez, Ezmurx, Ezmury, Eps0);
		}

		// Hx/Hy更新
		updateHx(ABC, Nx, Ny, Ez, Hx, dYc);
		updateHy(ABC, Nx, Ny, Ez, Hy, dXc);

		// ABC
		if (ABC > 0) {
			pmlHx(ABC, Nx, Ny, Ez, Hx, fhx, dhx);
			pmlHy(ABC, Nx, Ny, Ez, Hy, fhy, dhy);
		}

		// DFT
		dft1(iter, ABC, NFreq, NRx, iRx, jRx, etx, Ez, fcos, fsin, etx_r, etx_i, erx_r, erx_i);
		if (iout) {
			const int ifreq = 0;  // 第1周波数のみ
			dft2(ABC, Nx, Ny, Ez, fcos[ifreq][iter], fsin[ifreq][iter], Ez_r, Ez_i);
		}

		// 時間ステップ
		t += Dt;

		// debug, 平均電磁界出力
		if (ilog && (iter % 20 == 0)) {
			float fsum[2];
			average(ABC, Nx, Ny, Ez, Hx, Hy, fsum);
			printf("%5d %.6f %.6f\n", iter, fsum[0], fsum[1]);
		}
	}

	// S行列 = Erx / Etx
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		for (int irx = 0; irx < NRx; irx++) {
			cdiv(erx_r[ifreq][irx], erx_i[ifreq][irx], etx_r[ifreq], etx_i[ifreq], &s_r[ifreq][itx][irx], &s_i[ifreq][itx][irx]);
		}
	}

	// debug, 電界絶対値, セル中心
	if (iout) {
		efield(Nx, Ny, Ez_r, Ez_i, Ea);
	}


	// free
	free(fpml);
	free(fezx);
	free(fezy);
	free(fhx);
	free(fhy);
	free(dezx);
	free(dezy);
	free(dhx);
	free(dhy);

	// free
	free2d(fcos, NFreq)
	free2d(fsin, NFreq)
	free(etx_r);
	free(etx_i);
	free2d(erx_r, NFreq)
	free2d(erx_i, NFreq)
}


// 電磁界初期化
static void initfield(int ABC, int Nx, int Ny, int NRx, int NFreq,
	float **Ez, float **Hx, float **Hy,
	float **Ezmurx, float **Ezmury, float **Ezx, float **Ezy,
	float *etx_r, float *etx_i, float **erx_r, float **erx_i,
	int iout, float **Ez_r, float **Ez_i)
{
	assert((Nx > 0) && (Ny > 0) && (ABC >= 0) && (NRx > 0) && (NFreq > 0));

	for (int i = - ABC; i < Nx + 1 + ABC; i++) {
		memset(Ez[i + ABC], 0, (Ny + 1 + 2 * ABC) * sizeof(float));
	}
	for (int i = - ABC; i < Nx + 1 + ABC; i++) {
		memset(Hx[i + ABC], 0, (Ny + 0 + 2 * ABC) * sizeof(float));
	}
	for (int i = - ABC; i < Nx + 0 + ABC; i++) {
		memset(Hy[i + ABC], 0, (Ny + 1 + 2 * ABC) * sizeof(float));
	}

	// ABC
	if (ABC == 0) {
		for (int m = 0; m < 2; m++) {
			memset(Ezmurx[m], 0, (Ny + 1) * sizeof(float));
			memset(Ezmury[m], 0, (Nx + 1) * sizeof(float));
		}
	}
	else {
		for (int i = - ABC; i < Nx + 1 + ABC; i++) {
			memset(Ezx[i + ABC], 0, (Ny + 1 + 2 * ABC) * sizeof(float));
			memset(Ezy[i + ABC], 0, (Ny + 1 + 2 * ABC) * sizeof(float));
		}
	}

	// DFT
	memset(etx_r, 0, NFreq * sizeof(float));
	memset(etx_i, 0, NFreq * sizeof(float));
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		memset(erx_r[ifreq], 0, NRx * sizeof(float));
		memset(erx_i[ifreq], 0, NRx * sizeof(float));
	}

	// debug
	if (iout) {
		for (int i = 0; i < Nx + 1; i++) {
			memset(Ez_r[i], 0, (Ny + 1) * sizeof(float));
			memset(Ez_i[i], 0, (Ny + 1) * sizeof(float));
		}
	}
}


// FDTD係数
static void setupFactor(int Nx, int Ny, float Dt, float **epsr, float **sigm, float Eps0, float Sig0, float **C1Ez, float **C2Ez)
{
	const float pi = (float)(4 * atan(1));
	const float eta = C * 4 * pi * 1e-7f;

	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 1; j++) {
		float eps, sig;
		if ((i == 0) || (i == Nx) || (j == 0) || (j == Ny)) {
			// 境界
			eps = Eps0;
			sig = Sig0;
		}
		else {
			// 内部:4点の平均
			eps = (epsr[i - 1][j - 1] + epsr[i - 1][j] + epsr[i][j - 1] + epsr[i][j]) / 4;
			sig = (sigm[i - 1][j - 1] + sigm[i - 1][j] + sigm[i][j - 1] + sigm[i][j]) / 4;
		}
		C2Ez[i][j] = 1 / (eps + (sig * eta * C * Dt));
		C1Ez[i][j] = C2Ez[i][j] * eps;
	}
	}
}


// 平均電磁界
static void average(int ABC, int Nx, int Ny, float **Ez, float **Hx, float **Hy, float fsum[])
{
	// <E>
	float sume = 0;
	int nume = 0;
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 1; j++) {
		sume += (float)fabs(Ez[i + ABC][j + ABC]);
		nume++;
	}
	}
	fsum[0] = sume / nume;

	// <H>
	float sumh = 0;
	int numh = 0;
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 0; j++) {
		sumh += (float)fabs(Hx[i + ABC][j + ABC]);
		numh++;
	}
	}
	for (int i = 0; i < Nx + 0; i++) {
	for (int j = 0; j < Ny + 1; j++) {
		sumh += (float)fabs(Hy[i + ABC][j + ABC]);
		numh++;
	}
	}
	fsum[1] = sumh / numh;
}


// DFT加算, 受信点
static void dft1(int iter, int ABC, int NFreq, int NRx, int *iRx, int *jRx,
	float etx, float **Ez, float **fcos, float **fsin, float *etx_r, float *etx_i, float **erx_r, float **erx_i)
{
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		const float gcos = fcos[ifreq][iter];
		const float gsin = fsin[ifreq][iter];
		etx_r[ifreq] += gcos * etx;
		etx_i[ifreq] -= gsin * etx;
		for (int irx = 0; irx < NRx; irx++) {
			const float erx = Ez[iRx[irx] + ABC][jRx[irx] + ABC];
			erx_r[ifreq][irx] += gcos * erx;
			erx_i[ifreq][irx] -= gsin * erx;
		}
	}
}


// debug, DFT加算, 全領域調和界, 計算時間が増えるので必要なときのみ
static void dft2(int ABC, int Nx, int Ny, float **Ez, float fcos, float fsin, float **Ez_r, float **Ez_i)
{
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 1; j++) {
		Ez_r[i][j] += fcos * Ez[i + ABC][j + ABC];
		Ez_i[i][j] -= fsin * Ez[i + ABC][j + ABC];
	}
	}
}


// debug, 電界分布
static void efield(int Nx, int Ny, float **Ez_r, float **Ez_i, float **Ea)
{
	for (int i = 0; i < Nx; i++) {
	for (int j = 0; j < Ny; j++) {
		// セル平均
		const float e_r = (Ez_r[i][j] + Ez_r[i + 1][j] + Ez_r[i][j + 1] + Ez_r[i + 1][j + 1]) / 4;
		const float e_i = (Ez_i[i][j] + Ez_i[i + 1][j] + Ez_i[i][j + 1] + Ez_i[i + 1][j + 1]) / 4;
		Ea[i][j] = (float)sqrt((e_r * e_r) + (e_i * e_i));
	}
	}
}
