/*
geometry2.c
誘電率、導電率分布 (2)
数値ファントム
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <math.h>
#include <assert.h>

#include "../fdtd2d/alloc.h"
#include "../fdtd2d/ev.h"

#define MIN(x, y) ((x) < (y) ? (x) : (y))

static void plot3d_phantom(int, int, int, int ***, const char []);

extern void phantom(const char [], int, int, int, int ***);
extern double urand(int, int64_t *);
extern void getspan(int, int, int, int, int64_t *, int *, int *);
extern void smooth(int, int, int, int, int, int, float ****, float ****);

// 教師データ作成（誘電率、導電率）
void geometry2(int Ndata, int Nx, int Ny, int Nz, int Nout, int Sdif, int Nsmot,
	const int Nobj[], const int Lobj[][3], const int Mean[], float gRatio, float mRatio,
	float Eps0, float Sig0, const float Eps[], const float Sig[], int64_t Seed,
	float ****Epsr, float ****Sigm)
{
	assert((Ndata >= 0) && (Nx > 0) && (Ny > 0) && (Nz > 0) && (Nout > 0) && (Seed > 0));

	// 半球の中心と半径, この外は背景媒質
	const double xrc = Nx / 2.0;
	const double yrc = Ny / 2.0;
	const double zrc = Nout;
	const double rad = (Nx + Ny) / 4.0 - Nout;

	// 全体を背景媒質で初期化
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < Nx; i++) {
		for (int j = 0; j < Ny; j++) {
		for (int k = 0; k < Nz; k++) {
			Epsr[idata][i][j][k] = Eps0;
			Sigm[idata][i][j][k] = Sig0;
		}
		}
		}
	}

	// 誘電率、導電率代入
	// データに関するループ
	for (int idata = Sdif; idata < Ndata; idata++) {
		// 誘電体に関するループ
		const double r1 = urand(3, &Seed);
		const int nobj = Nobj[0] + (int)(r1 * (Nobj[1] - Nobj[0] + 1));
		for (int iobj = 0; iobj < nobj; iobj++) {
			// 誘電率、導電率
			const double r2 = urand(3, &Seed);
			const int tid = (r2 < mRatio) ? 2 : 4;  // 乳腺とがんに配分
			const float ep = Eps[tid];
			const float si = Sig[tid];
			// XY方向（複数個平均:中央に集める）
			int i0a = 0, i0b = 0, i0c = 0, i0d = 0;
			int i1a = 0, i1b = 0, i1c = 0, i1d = 0;
			int j0a = 0, j0b = 0, j0c = 0, j0d = 0;
			int j1a = 0, j1b = 0, j1c = 0, j1d = 0;
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0a, &i1a);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0a, &j1a);
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0b, &i1b);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0b, &j1b);
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0c, &i1c);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0c, &j1c);
			getspan(Nx, Nout + Lobj[0][2], Lobj[0][0], Lobj[0][1], &Seed, &i0d, &i1d);
			getspan(Ny, Nout + Lobj[1][2], Lobj[1][0], Lobj[1][1], &Seed, &j0d, &j1d);
			const double r3 = urand(3, &Seed);
			const int mean = Mean[0] + (int)(r3 * (Mean[1] - Mean[0] + 1));
			double ri0, ri1, rj0, rj1;
			if      (mean == 1) {
				ri0 = i0a;
				ri1 = i1a;
				rj0 = j0a;
				rj1 = j1a;
			}
			else if (mean == 2) {
				ri0 = (i0a + i0b) / 2.0;
				ri1 = (i1a + i1b) / 2.0;
				rj0 = (j0a + j0b) / 2.0;
				rj1 = (j1a + j1b) / 2.0;
			}
			else if (mean == 3) {
				ri0 = (i0a + i0b + i0c) / 3.0;
				ri1 = (i1a + i1b + i1c) / 3.0;
				rj0 = (j0a + j0b + j0c) / 3.0;
				rj1 = (j1a + j1b + j1c) / 3.0;
			}
			else {
				ri0 = (i0a + i0b + i0c + i0d) / 4.0;
				ri1 = (i1a + i1b + i1c + i1d) / 4.0;
				rj0 = (j0a + j0b + j0c + j0d) / 4.0;
				rj1 = (j1a + j1b + j1c + j1d) / 4.0;
			}
			const int i0 = (int)(ri0 + 0.5);
			const int i1 = (int)(ri1 + 0.5);
			const int j0 = (int)(rj0 + 0.5);
			const int j1 = (int)(rj1 + 0.5);
			// Z方向, 平均化しない
			int k0, k1;
			getspan(Nz, Nout + Lobj[2][2], Lobj[2][0], Lobj[2][1], &Seed, &k0, &k1);
			assert((i0 < i1) && (j0 < j1) && (k0 < k1));
			// 代入
			const double r4 = urand(3, &Seed);
			if (r4 < gRatio) {
				// 直方体内部 かつ 半球内部
				for (int i = i0; i < i1; i++) {
				for (int j = j0; j < j1; j++) {
				for (int k = k0; k < k1; k++) {
					const double x = i + 0.5;
					const double y = j + 0.5;
					const double z = k + 0.5;
					const double dx = x - xrc;
					const double dy = y - yrc;
					const double dz = z - zrc;
					if ((dx * dx) + (dy * dy) + (dz * dz) < (rad * rad)) {
						Epsr[idata][i][j][k] = ep;
						Sigm[idata][i][j][k] = si;
					}
				}
				}
				}
			}
			else {
				// 楕円体内部 かつ 半球内部
				const double xc = (i0 + i1) / 2.0;
				const double yc = (j0 + j1) / 2.0;
				const double zc = (k0 + k1) / 2.0;
				const double xr = (i1 - i0) / 2.0;
				const double yr = (j1 - j0) / 2.0;
				const double zr = (k1 - k0) / 2.0;
				for (int i = i0; i < i1; i++) {
				for (int j = j0; j < j1; j++) {
				for (int k = k0; k < k1; k++) {
					const double x = i + 0.5;
					const double y = j + 0.5;
					const double z = k + 0.5;
					const double dx = x - xrc;
					const double dy = y - yrc;
					const double dz = z - zrc;
					const double ex = (x - xc) / xr;
					const double ey = (y - yc) / yr;
					const double ez = (z - zc) / zr;
					if (((ex * ex) + (ey * ey) + (ez * ez) < 1) &&
					    ((dx * dx) + (dy * dy) + (dz * dz) < (rad * rad))) {
						Epsr[idata][i][j][k] = ep;
						Sigm[idata][i][j][k] = si;
					}
				}
				}
				}
			}
		}
	}

	// 平滑化
	if (Nsmot) {
		smooth(Ndata, Nx, Ny, Nz, Nout, Nsmot, Epsr, Sigm);
	}
/*
	// debug
	const int mx = Nx - (2 * Nout);
	const int my = Ny - (2 * Nout);
	const int mz = Nz - (2 * Nout);
	double esum = 0, ssum = 0;
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < mx; i++) {
		for (int j = 0; j < my; j++) {
		for (int k = 0; k < mz; k++) {
			esum += Epsr[idata][i + Nout][j + Nout][k + Nout];
			ssum += Sigm[idata][i + Nout][j + Nout][k + Nout];
		}
		}
		}
	}
	printf("%d %d %d %e %e\n", mx, my, mz, esum / (Ndata * mx * my * mz), ssum / (Ndata * mx * my * mz));
*/
}


// 数値ファントム
void geometry2_test(int Mprec, int Ndata, int Nx, int Ny, int Nz, int Nout, int Sdif, int Nsmot,
	float Eps0, float Sig0, const float Eps[], const float Sig[], float ****Epsr, float ****Sigm)
{
	// ファイル名
	const char fn[] = "../phantom/mediumLabelList_Reso1mm_Case1.txt";
	//const char fn[] = "../phantom/mediumLabelList_Reso1mm_Case2.txt";
	//const char fn[] = "../phantom/mediumLabelList_Reso1mm_Case3.txt";

	// ファイルのセル数（固定）
	const int nx = 100;
	const int ny = 100;
	const int nz = 50;

	// 数値ファントムを読み込む
	int ***map;
	alloc3d(int, map, nx, ny, nz)
	phantom(fn, nx, ny, nz, map);

	// 組織IDを電気定数に変換する
	// 0=背景, 1=脂肪, 2=乳腺, 3=容器, 4=がん, 5=皮膚
	float ***epsr, ***sigm;
	alloc3d(float, epsr, nx, ny, nz)
	alloc3d(float, sigm, nx, ny, nz)
	for (int i = 0; i < nx; i++) {
	for (int j = 0; j < ny; j++) {
	for (int k = 0; k < nz; k++) {
		const int id = map[i][j][k];
		assert((id >= 0) && (id <= 5));
		epsr[i][j][k] = Eps[id];
		sigm[i][j][k] = Sig[id];
	}
	}
	}

	// 全体を背景媒質で初期化
	for (int idata = 0; idata < Ndata; idata++) {
		for (int i = 0; i < Nx; i++) {
		for (int j = 0; j < Ny; j++) {
		for (int k = 0; k < Nz; k++) {
			Epsr[idata][i][j][k] = Eps0;
			Sigm[idata][i][j][k] = Sig0;
		}
		}
		}
	}

	// 誘電率, 導電率（平均化）
	const int ms = Mprec;  // 間引き間隔
	const int mx = Nx - (2 * Nout);
	const int my = Ny - (2 * Nout);
	const int mz = Nz - (2 * Nout);
	assert((ms > 0) && (mx == nx / ms) && (my == ny / ms) && (mz == nz / ms));  // 切り捨て
	for (int idata = Sdif; idata < Ndata; idata++) {
		for (int i = 0; i < mx; i++) {
		for (int j = 0; j < my; j++) {
		for (int k = 0; k < mz; k++) {
			// 平均化
			float esum = 0;
			float ssum = 0;
			for (int is = 0; is < ms; is++) {
			for (int js = 0; js < ms; js++) {
			for (int ks = 0; ks < ms; ks++) {
				esum += epsr[(i * ms) + is][(j * ms) + js][(k * ms) + ks];
				ssum += sigm[(i * ms) + is][(j * ms) + js][(k * ms) + ks];
			}
			}
			}
			Epsr[idata][Nout + i][Nout + j][Nout + k] = esum / (ms * ms * ms);
			Sigm[idata][Nout + i][Nout + j][Nout + k] = ssum / (ms * ms * ms);
		}
		}
		}
	}

	// 平滑化
	if (Nsmot) {
		smooth(Ndata, Nx, Ny, Nz, Nout, Nsmot, Epsr, Sigm);
	}

	// 図形表示
	plot3d_phantom(nx, ny, nz, map, "phantom.ev3");

	// free
	free3d(map, nx, ny)
	free3d(epsr, nx, ny)
	free3d(sigm, nx, ny)
}


// 数値ファントム図形表示(3D), debug
static void plot3d_phantom(int nx, int ny, int nz, int ***map, const char fn[])
{
	// 0=背景, 1=脂肪, 2=乳腺, 3=容器, 4=がん, 5=皮膚
	const unsigned char rgb[][3] = {
	{  0,   0, 255},
	{  0,   0, 255},
	{  0, 255,   0},
	{  0,   0,   0},
	{255,   0,   0},
	{  0,   0,   0}};

	// 開始
	ev3d_init();
	ev3d_newPage();

	// 描画
	for (int i = 0; i < nx; i++) {
	for (int j = 0; j < ny; j++) {
	for (int k = 0; k < nz; k++) {
		const int c = map[i][j][k];
		// 乳腺とがんのみ
		if ((c == 2) || (c == 4)) {
			ev3d_setColor(rgb[c][0], rgb[c][1], rgb[c][2]);
			//ev3d_fillRectangle('X', i, j, k, j + 1, k + 1);
			//ev3d_fillRectangle('Y', j, k, i, k + 1, i + 1);
			//ev3d_fillRectangle('Z', k, i, j, i + 1, j + 1);
			ev3d_drawLine(i, j, k, i + 1, j,     k    );
			ev3d_drawLine(i, j, k, i,     j + 1, k    );
			ev3d_drawLine(i, j, k, i,     j,     k + 1);
		}
	}
	}
	}

	// 領域
	ev3d_setColor(160, 160, 160);
	ev3d_drawBox(0, nx, 0, ny, 0, nz);

	// 出力
	ev3d_file(1, fn, 0);
	ev3d_output();
}
