/*
plot2d.c
2D図形出力, debug用
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#include "../fdtd2d/alloc.h"
#include "../fdtd2d/ev.h"

extern void getminmax4f(int, int, int, int, int, int, int, int, int, float ****, double *, double *);
extern void plot2d_contour(double, double, int, int, double **, double, double, int, int);

// 誘電率分布, 送受信点
// dir = 'X'/'Y'/'Z' : Y-Z, X-Z, X-Y
// idata0<=idata<idata1, idata=0,1,2,...
// mx : 1行の図数
// space : 余白
void plot2d_g(char dir, int idata0, int idata1, int mx, int space, int Nx, int Ny, int Nz, int Nout,
	float ****d, int color, int grid,
	int NTx, int *iTx, int *jTx, int *kTx,
	int NRx, int *iRx, int *jRx, int *kRx, const char fn[])
{
	assert((Nx > 0) && (Ny > 0) && (Nz > 0));
	assert ((idata0 >= 0) && (idata1 > idata0));

	char str[BUFSIZ], fmt[BUFSIZ];
	const double dm = 0.3;

	// 面で場合分け
	int nfig = 0, n1 = 0, n2 = 0;
	if      (dir == 'X') {
		nfig = Nx;
		n1 = Ny;
		n2 = Nz;
	}
	else if (dir == 'Y') {
		nfig = Ny;
		n1 = Nx;
		n2 = Nz;
	}
	else if (dir == 'Z') {
		nfig = Nz;
		n1 = Nx;
		n2 = Ny;
	}

	// 2D配列
	double **f;
	alloc2d(double, f, n1, n2)

	// 最小・最大
	double dmin, dmax;
	getminmax4f(idata0, idata1, 0, Nx, 0, Ny, 0, Nz, 0, d, &dmin, &dmax);

	// 開始
	const int my = nfig / mx + (nfig % mx > 0);
	const double w = (mx * n1) + ((mx + 1) * space);
	const double h = (my * n2) + ((my + 2) * space);
	const double hgt = h / 60;
	ev2d_init(w, h);

	// 1データ=1ページ
	for (int idata = idata0; idata < idata1; idata++) {
		ev2d_newPage();

		// 等高線図, 全Z面の集合
		int ifig = 0;
		for (int py = my - 1; py >= 0; py--) {
		for (int px = 0; px < mx; px++) {
			if (ifig < nfig) {
				// 原点
				const double x0 = (px * n1) + ((px + 1) * space);
				const double y0 = (py * n2) + ((py + 2) * space);

				// XYデータ
				for (int i1 = 0; i1 < n1; i1++) {
				for (int i2 = 0; i2 < n2; i2++) {
					if      (dir == 'X') {
						f[i1][i2] = d[idata][ifig][i1][i2];
					}
					else if (dir == 'Y') {
						f[i1][i2] = d[idata][i1][ifig][i2];
					}
					else if (dir == 'Z') {
						f[i1][i2] = d[idata][i1][i2][ifig];
					}
				}
				}

				// 等高線図
				plot2d_contour(x0, y0, n1, n2, f, dmin, dmax, color, grid);

				// 内部領域
				ev2d_setColor(160, 160, 160);
				ev2d_drawRectangle(x0 + Nout, y0 + Nout, x0 + n1 - Nout, y0 + n2 - Nout);

				// データ番号
				ev2d_setColor(0, 0, 0);
				sprintf(str, "%d", ifig + 1);
				ev2d_drawString(x0, y0 + n2 + 0.3 * hgt, hgt, str);

				// 送信点（面内にあるとき）
				ev2d_setColor(255, 0, 0);
				for (int n = 0; n < NTx; n++) {
					const int p0 = (dir == 'X') ? iTx[n] : (dir == 'Y') ? jTx[n] : kTx[n];
					const int p1 = (dir == 'X') ? jTx[n] : (dir == 'Y') ? iTx[n] : iTx[n];
					const int p2 = (dir == 'X') ? kTx[n] : (dir == 'Y') ? kTx[n] : jTx[n];
					if (ifig == p0) {
						ev2d_fillRectangle(x0 + p1 - dm, y0 + p2 - dm, x0 + p1 + dm, y0 + p2 + dm);
					}
				}

				// 受信点（面内にあるとき）
				ev2d_setColor(0, 255, 0);
				for (int n = 0; n < NRx; n++) {
					const int p0 = (dir == 'X') ? iRx[n] : (dir == 'Y') ? jRx[n] : kRx[n];
					const int p1 = (dir == 'X') ? jRx[n] : (dir == 'Y') ? iRx[n] : iRx[n];
					const int p2 = (dir == 'X') ? kRx[n] : (dir == 'Y') ? kRx[n] : jRx[n];
					if (ifig == p0) {
						ev2d_fillRectangle(x0 + p1 - dm, y0 + p2 - dm, x0 + p1 + dm, y0 + p2 + dm);
					}
				}

				// 図++
				ifig++;
			}
		}
		}

		// 最小・最大
		strcpy(fmt, "%c data=%d Nx=%d Ny=%d Nz=%d Mx=%d My=%d Mz=%d Tx=%d Rx=%d min=%.4g max=%.4g");
		sprintf(str, fmt, dir, idata + 1, Nx, Ny, Nz, Nx - 2 * Nout, Ny - 2 * Nout, Nz - 2 * Nout, NTx, NRx, dmin, dmax);
		ev2d_setColor(0, 0, 0);
		ev2d_drawString(space, 1.0 * hgt, 1.5 * hgt, str);

	}
	free2d(f, n1)

	// 出力
	ev2d_file(1, fn);
	ev2d_output();
}


// 電界分布
void plot2d_e(int nstep, int space, int NTx, int Nx, int Ny, int Nz, int Nout,
	float ****d, int db, int grid, const char fn[])
{
	assert((NTx > 0) && (nstep > 0) && (Nx > 0) && (Ny > 0) && (Nz > 0));

	char str[BUFSIZ], fmt[BUFSIZ];
	const float eps = 1e-12f;

	// 最小・最大(ZXYに注意)
	double dmin, dmax;
	getminmax4f(0, NTx, 0, Nz, 0, Nx, 0, Ny, db, d, &dmin, &dmax);

	// XY配列
	double **f;
	alloc2d(double, f, Nx, Ny)

	// 開始
	int my = (int)sqrt(Nz) - 1;
	if (my < 1) my = 1;
	const int mx = Nz / my + 1;
	const double w = (mx * Nx) + ((mx + 1) * space);
	const double h = (my * Ny) + ((my + 2) * space);
	const double hgt = h / 40;
	ev2d_init(w, h);

	// 1送信点=1ページ
	for (int itx = 0; itx < NTx; itx += nstep) {
		// 改ページ
		ev2d_newPage();

		// 等高線図, 全Z面の集合
		int k = 0;
		for (int py = my - 1; py >= 0; py--) {
		for (int px = 0; px < mx; px++) {
			if (k < Nz) {
				// 原点
				const double x0 = (px * Nx) + ((px + 1) * space);
				const double y0 = (py * Ny) + ((py + 2) * space);

				// XYデータ
				for (int i = 0; i < Nx; i++) {
				for (int j = 0; j < Ny; j++) {
					f[i][j] = d[itx][k][i][j];
				}
				}

				// dB
				if (db) {
					for (int i = 0; i < Nx; i++) {
					for (int j = 0; j < Ny; j++) {
						f[i][j] = 20 * (float)log10(fmax(f[i][j], eps));
					}
					}
				}

				// 等高線図(カラー)
				plot2d_contour(x0, y0, Nx, Ny, f, dmin, dmax, 1, grid);

				// 内部領域
				ev2d_setColor(160, 160, 160);
				ev2d_drawRectangle(x0 + Nout, y0 + Nout, x0 + Nx - Nout, y0 + Ny - Nout);

				// データ番号
				ev2d_setColor(0, 0, 0);
				sprintf(str, "%d", k + 1);
				ev2d_drawString(x0, y0 + Ny + 0.2 * hgt, hgt, str);

				// 図++
				k++;
			}
		}
		}

		// 最小・最大
		strcpy(fmt, db ? "Tx=%d/%d Nx=%d Ny=%d Nz=%d min=%.3fdB max=%.3fdB"
		               : "Tx=%d/%d Nx=%d Ny=%d Nz=%d min=%.6f max=%.6f");
		sprintf(str, fmt, itx + 1, NTx, Nx, Ny, Nz, dmin, dmax);
		ev2d_setColor(0, 0, 0);
		ev2d_drawString(space, 1.0 * hgt, 1.5 * hgt, str);
	}
	free2d(f, Nx)

	// 出力
	ev2d_file(1, fn);
	ev2d_output();
}
