/*
solve.c
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#include "../fdtd2d/alloc.h"

static void setupFactor(int, int, int, float, float ***, float ***, float, float, float ***, float ***, float ***, float ***, float ***, float ***);
static void initfield(int, int, int, int, int, int, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float *, float *, float **, float **, int, float ***, float ***, float ***, float ***, float ***, float ***);
static void dft1(int, int, int, float, int, int *, int *, int *, int *, float ***, float ***, float ***, float **, float **, float *, float *, float **, float **);
static void dft2(int, int, int, int, float, float, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***, float ***);
static void average(int, int, int, int, float ***, float ***, float ***, float ***, float ***, float ***, float []);
static void efield(int, int, int, float ***, float ***, float ***, float ***, float ***, float ***, float ***);

extern void dftfactor(int, float, float, float *, float *);
extern void updateEx(int, int, int, int, float ***, float ***, float ***, float *, float *, float ***, float ***);
extern void updateEy(int, int, int, int, float ***, float ***, float ***, float *, float *, float ***, float ***);
extern void updateEz(int, int, int, int, float ***, float ***, float ***, float *, float *, float ***, float ***);
extern void updateHx(int, int, int, int, float ***, float ***, float ***, float *, float *);
extern void updateHy(int, int, int, int, float ***, float ***, float ***, float *, float *);
extern void updateHz(int, int, int, int, float ***, float ***, float ***, float *, float *);
extern void murEx(int, int, int, float, float *, float *, float ***, float ***, float ***, float);
extern void murEy(int, int, int, float, float *, float *, float ***, float ***, float ***, float);
extern void murEz(int, int, int, float, float *, float *, float ***, float ***, float ***, float);
extern void pmlEx(int, int, int, int, float ***, float ***, float ***, float ***, float ***, float *, float *, float *, float *);
extern void pmlEy(int, int, int, int, float ***, float ***, float ***, float ***, float ***, float *, float *, float *, float *);
extern void pmlEz(int, int, int, int, float ***, float ***, float ***, float ***, float ***, float *, float *, float *, float *);
extern void pmlHx(int, int, int, int, float ***, float ***, float ***, float ***, float ***, float *, float *, float *, float *);
extern void pmlHy(int, int, int, int, float ***, float ***, float ***, float ***, float ***, float *, float *, float *, float *);
extern void pmlHz(int, int, int, int, float ***, float ***, float ***, float ***, float ***, float *, float *, float *, float *);
extern void setupPmlEx(int, int, int, float, float, float *, float *, float *, float *, float *, float *, float *, float *, float *);
extern void setupPmlEy(int, int, int, float, float, float *, float *, float *, float *, float *, float *, float *, float *, float *);
extern void setupPmlEz(int, int, int, float, float, float *, float *, float *, float *, float *, float *, float *, float *, float *);
extern void setupPmlHx(int, int, int, float, float, float *, float *, float *, float *, float *, float *, float *, float *, float *);
extern void setupPmlHy(int, int, int, float, float, float *, float *, float *, float *, float *, float *, float *, float *, float *);
extern void setupPmlHz(int, int, int, float, float, float *, float *, float *, float *, float *, float *, float *, float *, float *);
extern void pmlfactor(int, float *);
extern float vfeed(float, float);
extern void cdiv(float, float, float, float, float *, float *);

extern float C;

void solve(
	int itx, int Niter, int Nx, int Ny, int Nz, int NFreq, const float Freq[], float Dt, int ABC,
	int *iTx, int *jTx, int *kTx, int *mTx, int NRx, int *iRx, int *jRx, int *kRx, int *mRx,
	float *Xn, float *Yn, float *Zn, float *dXn, float *dYn, float *dZn, float *dXc, float *dYc, float *dZc,
	float ***epsr, float ***sigm, float Eps0, float Sig0,
	float ***Ex, float ***Ey, float ***Ez, float ***Hx, float ***Hy, float ***Hz,
	float ***C1Ex, float ***C2Ex, float ***C1Ey, float ***C2Ey, float ***C1Ez, float ***C2Ez,
	float ***Exmury, float ***Exmurz, float ***Eymurz, float ***Eymurx, float ***Ezmurx, float ***Ezmury,
	float ***Exy, float ***Exz, float ***Eyz, float ***Eyx, float ***Ezx, float ***Ezy,
	float ***Hxy, float ***Hxz, float ***Hyz, float ***Hyx, float ***Hzx, float ***Hzy,
	float ***s_r, float ***s_i,
	int ilog, int iout, float ***Ex_r, float ***Ex_i, float ***Ey_r, float ***Ey_i, float ***Ez_r, float ***Ez_i, float ***Ea)
{
	// DFT係数
	float **fcos, **fsin;
	alloc2d(float, fcos, NFreq, Niter + 1)
	alloc2d(float, fsin, NFreq, Niter + 1)
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		dftfactor(Niter, Freq[ifreq], Dt, fcos[ifreq], fsin[ifreq]);
	}

	// 送信電界
	float *etx_r, *etx_i;
	alloc1d(float, etx_r, NFreq)
	alloc1d(float, etx_i, NFreq)

	// 受信電界
	float **erx_r, **erx_i;
	alloc2d(float, erx_r, NFreq, NRx)
	alloc2d(float, erx_i, NFreq, NRx)

	// FDTD係数
	setupFactor(Nx, Ny, Nz, Dt, epsr, sigm, Eps0, Sig0, C1Ex, C2Ex, C1Ey, C2Ey, C1Ez, C2Ez);

	// PML係数
	float *fpml;
	float *fexy, *fexz, *feyz, *feyx, *fezx, *fezy;
	float *dexy, *dexz, *deyz, *deyx, *dezx, *dezy;
	float *fhxy, *fhxz, *fhyz, *fhyx, *fhzx, *fhzy;
	float *dhxy, *dhxz, *dhyz, *dhyx, *dhzx, *dhzy;
	alloc1d(float, fpml,      1 + 2 * ABC)
	alloc1d(float, fexy, Ny + 1 + 2 * ABC)
	alloc1d(float, fexz, Nz + 1 + 2 * ABC)
	alloc1d(float, dexy, Ny + 1 + 2 * ABC)
	alloc1d(float, dexz, Nz + 1 + 2 * ABC)
	alloc1d(float, feyz, Nz + 1 + 2 * ABC)
	alloc1d(float, feyx, Nx + 1 + 2 * ABC)
	alloc1d(float, deyz, Nz + 1 + 2 * ABC)
	alloc1d(float, deyx, Nx + 1 + 2 * ABC)
	alloc1d(float, fezx, Nx + 1 + 2 * ABC)
	alloc1d(float, fezy, Ny + 1 + 2 * ABC)
	alloc1d(float, dezx, Nx + 1 + 2 * ABC)
	alloc1d(float, dezy, Ny + 1 + 2 * ABC)
	alloc1d(float, fhxy, Ny + 0 + 2 * ABC)
	alloc1d(float, fhxz, Nz + 0 + 2 * ABC)
	alloc1d(float, dhxy, Ny + 0 + 2 * ABC)
	alloc1d(float, dhxz, Nz + 0 + 2 * ABC)
	alloc1d(float, fhyz, Nz + 0 + 2 * ABC)
	alloc1d(float, fhyx, Nx + 0 + 2 * ABC)
	alloc1d(float, dhyz, Nz + 0 + 2 * ABC)
	alloc1d(float, dhyx, Nx + 0 + 2 * ABC)
	alloc1d(float, fhzx, Nx + 0 + 2 * ABC)
	alloc1d(float, fhzy, Ny + 0 + 2 * ABC)
	alloc1d(float, dhzx, Nx + 0 + 2 * ABC)
	alloc1d(float, dhzy, Ny + 0 + 2 * ABC)
	if (ABC) {
		pmlfactor(ABC, fpml);
		setupPmlEx(ABC, Ny, Nz, Dt, Eps0, Yn, Zn, dYn, dZn, fpml, fexy, fexz, dexy, dexz);
		setupPmlEy(ABC, Nz, Nx, Dt, Eps0, Zn, Xn, dZn, dXn, fpml, feyz, feyx, deyz, deyx);
		setupPmlEz(ABC, Nx, Ny, Dt, Eps0, Xn, Yn, dXn, dYn, fpml, fezx, fezy, dezx, dezy);
		setupPmlHx(ABC, Ny, Nz, Dt, Eps0, Yn, Zn, dYc, dZc, fpml, fhxy, fhxz, dhxy, dhxz);
		setupPmlHy(ABC, Nz, Nx, Dt, Eps0, Zn, Xn, dZc, dXc, fpml, fhyz, fhyx, dhyz, dhyx);
		setupPmlHz(ABC, Nx, Ny, Dt, Eps0, Xn, Yn, dXc, dYc, fpml, fhzx, fhzy, dhzx, dhzy);
	}

	// 電磁界初期化
	initfield(ABC, Nx, Ny, Nz, NRx, NFreq,
		Ex, Ey, Ez, Hx, Hy, Hz,
		Exmury, Exmurz, Eymurz, Eymurx, Ezmurx, Ezmury,
		Exy, Exz, Eyz, Eyx, Ezx, Ezy,
		Hxy, Hxz, Hyz, Hyx, Hzx, Hzy,
		etx_r, etx_i, erx_r, erx_i,
		iout, Ex_r, Ex_i, Ey_r, Ey_i, Ez_r, Ez_i);

	// FDTD反復計算
	float t = 0;
	for (int iter = 0; iter <= Niter; iter++) {
		// E更新
		updateEx(ABC, Nx, Ny, Nz, Ex, Hy, Hz, dYn, dZn, C1Ex, C2Ex);
		updateEy(ABC, Nx, Ny, Nz, Ey, Hz, Hx, dZn, dXn, C1Ey, C2Ey);
		updateEz(ABC, Nx, Ny, Nz, Ez, Hx, Hy, dXn, dYn, C1Ez, C2Ez);

		// 送信電界
		const float freq = (Freq[0] + Freq[NFreq - 1]) / 2;
		const float vf = vfeed(t, freq);
		float etx = 0;
		if      (mTx[itx] == 0) {
			const float dx = Xn[iTx[itx] + 1] - Xn[iTx[itx]];
			etx = Ex[iTx[itx] + ABC][jTx[itx] + ABC][kTx[itx] + ABC] = vf / dx;
		}
		else if (mTx[itx] == 1) {
			const float dy = Yn[jTx[itx] + 1] - Yn[jTx[itx]];
			etx = Ey[iTx[itx] + ABC][jTx[itx] + ABC][kTx[itx] + ABC] = vf / dy;
		}
		else if (mTx[itx] == 2) {
			const float dz = Zn[kTx[itx] + 1] - Zn[kTx[itx]];
			etx = Ez[iTx[itx] + ABC][jTx[itx] + ABC][kTx[itx] + ABC] = vf / dz;
		}

		// ABC
		if (ABC) {
			pmlEx(ABC, Nx, Ny, Nz, Ex, Hy, Hz, Exy, Exz, fexy, fexz, dexy, dexz);
			pmlEy(ABC, Nx, Ny, Nz, Ey, Hz, Hx, Eyz, Eyx, feyz, feyx, deyz, deyx);
			pmlEz(ABC, Nx, Ny, Nz, Ez, Hx, Hy, Ezx, Ezy, fezx, fezy, dezx, dezy);
		}
		else {
			murEx(Nx, Ny, Nz, Dt, Yn, Zn, Ex, Exmury, Exmurz, Eps0);
			murEy(Nx, Ny, Nz, Dt, Zn, Xn, Ey, Eymurz, Eymurx, Eps0);
			murEz(Nx, Ny, Nz, Dt, Xn, Yn, Ez, Ezmurx, Ezmury, Eps0);
		}

		// H更新
		updateHx(ABC, Nx, Ny, Nz, Hx, Ey, Ez, dYc, dZc);
		updateHy(ABC, Nx, Ny, Nz, Hy, Ez, Ex, dZc, dXc);
		updateHz(ABC, Nx, Ny, Nz, Hz, Ex, Ey, dXc, dYc);

		// ABC
		if (ABC) {
			pmlHx(ABC, Nx, Ny, Nz, Hx, Ey, Ez, Hxy, Hxz, fhxy, fhxz, dhxy, dhxz);
			pmlHy(ABC, Nx, Ny, Nz, Hy, Ez, Ex, Hyz, Hyx, fhyz, fhyx, dhyz, dhyx);
			pmlHz(ABC, Nx, Ny, Nz, Hz, Ex, Ey, Hzx, Hzy, fhzx, fhzy, dhzx, dhzy);
		}

		// DFT
		dft1(iter, ABC, NFreq, etx, NRx, iRx, jRx, kRx, mRx, Ex, Ey, Ez, fcos, fsin, etx_r, etx_i, erx_r, erx_i);
		if (iout) {
			const int ifreq = 0;  // 第1周波数のみ
			dft2(ABC, Nx, Ny, Nz, fcos[ifreq][iter], fsin[ifreq][iter], Ex, Ey, Ez, Ex_r, Ex_i, Ey_r, Ey_i, Ez_r, Ez_i);
		}
	
		// 時間ステップ
		t += Dt;

		// debug, 平均電磁界出力
		if (ilog && (iter % 50 == 0)) {
			float fsum[2];
			average(ABC, Nx, Ny, Nz, Ex, Ey, Ez, Hx, Hy, Hz, fsum);
			printf("%5d %.6f %.6f\n", iter, fsum[0], fsum[1]);
		}
	}

	// S行列 = Erx / Etx
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		for (int irx = 0; irx < NRx; irx++) {
			 cdiv(erx_r[ifreq][irx], erx_i[ifreq][irx], etx_r[ifreq], etx_i[ifreq], &s_r[ifreq][itx][irx], &s_i[ifreq][itx][irx]);
		}
	}

	// debug, 電界分布, セル中心, 第1周波数のみ, データ0=プロセス0=通信不要
	if (iout) {
		efield(Nx, Ny, Nz, Ex_r, Ex_i, Ey_r, Ey_i, Ez_r, Ez_i, Ea);
	}

	// free
	free(fpml);
	free(fexy); free(fexz); free(deyz); free(deyx); free(fezx); free(fezy);
	free(dexy); free(dexz); free(feyz); free(feyx); free(dezx); free(dezy);
	free(fhxy); free(fhxz); free(dhyz); free(dhyx); free(fhzx); free(fhzy);
	free(dhxy); free(dhxz); free(fhyz); free(fhyx); free(dhzx); free(dhzy);

	// free
	free(fcos);
	free(fsin);
	free(etx_r);
	free(etx_i);
	free2d(erx_r, NFreq)
	free2d(erx_i, NFreq)
}


// 電磁界初期化
static void initfield(int ABC, int Nx, int Ny, int Nz, int NRx, int NFreq,
	float ***Ex, float ***Ey, float ***Ez, float ***Hx, float ***Hy, float ***Hz,
	float ***Exmury, float ***Exmurz, float ***Eymurz, float ***Eymurx, float ***Ezmurx, float ***Ezmury,
	float ***Exy, float ***Exz, float ***Eyz, float ***Eyx, float ***Ezx, float ***Ezy,
	float ***Hxy, float ***Hxz, float ***Hyz, float ***Hyx, float ***Hzx, float ***Hzy,
	float *etx_r, float *etx_i, float **erx_r, float **erx_i,
	int iout, float ***Ex_r, float ***Ex_i, float ***Ey_r, float ***Ey_i, float ***Ez_r, float ***Ez_i)
{
	assert((Nx > 0) && (Ny > 0) && (Nz > 0) && (ABC >= 0) && (NRx > 0) && (NFreq > 0));

	// E
	for (int i = - ABC; i < Nx + 0 + ABC; i++) {
	for (int j = - ABC; j < Ny + 1 + ABC; j++) {
		memset(Ex[i + ABC][j + ABC], 0, (Nz + 1 + 2 * ABC) * sizeof(float));
	}
	}
	for (int i = - ABC; i < Nx + 1 + ABC; i++) {
	for (int j = - ABC; j < Ny + 0 + ABC; j++) {
		memset(Ey[i + ABC][j + ABC], 0, (Nz + 1 + 2 * ABC) * sizeof(float));
	}
	}
	for (int i = - ABC; i < Nx + 1 + ABC; i++) {
	for (int j = - ABC; j < Ny + 1 + ABC; j++) {
		memset(Ez[i + ABC][j + ABC], 0, (Nz + 0 + 2 * ABC) * sizeof(float));
	}
	}
	// H
	for (int i = - ABC; i < Nx + 1 + ABC; i++) {
	for (int j = - ABC; j < Ny + 0 + ABC; j++) {
		memset(Hx[i + ABC][j + ABC], 0, (Nz + 0 + 2 * ABC) * sizeof(float));
	}
	}
	for (int i = - ABC; i < Nx + 0 + ABC; i++) {
	for (int j = - ABC; j < Ny + 1 + ABC; j++) {
		memset(Hy[i + ABC][j + ABC], 0, (Nz + 0 + 2 * ABC) * sizeof(float));
	}
	}
	for (int i = - ABC; i < Nx + 0 + ABC; i++) {
	for (int j = - ABC; j < Ny + 0 + ABC; j++) {
		memset(Hz[i + ABC][j + ABC], 0, (Nz + 1 + 2 * ABC) * sizeof(float));
	}
	}

	// ABC
	if (ABC == 0) {
		for (int m = 0; m < 2; m++) {
			for (int k = 0; k < Nz + 1; k++) {
				memset(Exmury[m][k], 0, (Nx + 0) * sizeof(float));
			}
			for (int j = 0; j < Ny + 1; j++) {
				memset(Exmurz[m][j], 0, (Nx + 0) * sizeof(float));
			}
			for (int i = 0; i < Nx + 1; i++) {
				memset(Eymurz[m][i], 0, (Ny + 0) * sizeof(float));
			}
			for (int k = 0; k < Nz + 1; k++) {
				memset(Eymurx[m][k], 0, (Ny + 0) * sizeof(float));
			}
			for (int j = 0; j < Ny + 1; j++) {
				memset(Ezmurx[m][j], 0, (Nz + 0) * sizeof(float));
			}
			for (int i = 0; i < Nx + 1; i++) {
				memset(Ezmury[m][i], 0, (Nz + 0) * sizeof(float));
			}
		}
	}
	else {
		// E
		for (int i = - ABC; i < Nx + 0 + ABC; i++) {
		for (int j = - ABC; j < Ny + 1 + ABC; j++) {
			memset(Exy[i + ABC][j + ABC], 0, (Nz + 1 + 2 * ABC) * sizeof(float));
			memset(Exz[i + ABC][j + ABC], 0, (Nz + 1 + 2 * ABC) * sizeof(float));
		}
		}
		for (int i = - ABC; i < Nx + 1 + ABC; i++) {
		for (int j = - ABC; j < Ny + 0 + ABC; j++) {
			memset(Eyz[i + ABC][j + ABC], 0, (Nz + 1 + 2 * ABC) * sizeof(float));
			memset(Eyx[i + ABC][j + ABC], 0, (Nz + 1 + 2 * ABC) * sizeof(float));
		}
		}
		for (int i = - ABC; i < Nx + 1 + ABC; i++) {
		for (int j = - ABC; j < Ny + 1 + ABC; j++) {
			memset(Ezx[i + ABC][j + ABC], 0, (Nz + 0 + 2 * ABC) * sizeof(float));
			memset(Ezy[i + ABC][j + ABC], 0, (Nz + 0 + 2 * ABC) * sizeof(float));
		}
		}
		// H
		for (int i = - ABC; i < Nx + 1 + ABC; i++) {
		for (int j = - ABC; j < Ny + 0 + ABC; j++) {
			memset(Hxy[i + ABC][j + ABC], 0, (Nz + 0 + 2 * ABC) * sizeof(float));
			memset(Hxz[i + ABC][j + ABC], 0, (Nz + 0 + 2 * ABC) * sizeof(float));
		}
		}
		for (int i = - ABC; i < Nx + 0 + ABC; i++) {
		for (int j = - ABC; j < Ny + 1 + ABC; j++) {
			memset(Hyz[i + ABC][j + ABC], 0, (Nz + 0 + 2 * ABC) * sizeof(float));
			memset(Hyx[i + ABC][j + ABC], 0, (Nz + 0 + 2 * ABC) * sizeof(float));
		}
		}
		for (int i = - ABC; i < Nx + 0 + ABC; i++) {
		for (int j = - ABC; j < Ny + 0 + ABC; j++) {
			memset(Hzx[i + ABC][j + ABC], 0, (Nz + 1 + 2 * ABC) * sizeof(float));
			memset(Hzy[i + ABC][j + ABC], 0, (Nz + 1 + 2 * ABC) * sizeof(float));
		}
		}
	}

	// DFT
	memset(etx_r, 0, NFreq * sizeof(float));
	memset(etx_i, 0, NFreq * sizeof(float));
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		memset(erx_r[ifreq], 0, NRx * sizeof(float));
		memset(erx_i[ifreq], 0, NRx * sizeof(float));
	}

	// debug, E
	if (iout) {
		for (int i = 0; i < Nx + 0; i++) {
		for (int j = 0; j < Ny + 1; j++) {
			memset(Ex_r[i][j], 0, (Nz + 1) * sizeof(float));
			memset(Ex_i[i][j], 0, (Nz + 1) * sizeof(float));
		}
		}
		for (int i = 0; i < Nx + 1; i++) {
		for (int j = 0; j < Ny + 0; j++) {
			memset(Ey_r[i][j], 0, (Nz + 1) * sizeof(float));
			memset(Ey_i[i][j], 0, (Nz + 1) * sizeof(float));
		}
		}
		for (int i = 0; i < Nx + 1; i++) {
		for (int j = 0; j < Ny + 1; j++) {
			memset(Ez_r[i][j], 0, (Nz + 0) * sizeof(float));
			memset(Ez_i[i][j], 0, (Nz + 0) * sizeof(float));
		}
		}
	}
}


// FDTD係数
static void setupFactor(int Nx, int Ny, int Nz, float Dt,
	float ***epsr, float ***sigm, float Eps0, float Sig0,
	float ***C1Ex, float ***C2Ex, float ***C1Ey, float ***C2Ey, float ***C1Ez, float ***C2Ez)
{
	const float pi = (float)(4 * atan(1));
	const float eta = C * 4 * pi * 1e-7f;

	float eps, sig;

	// Ex
	for (int i = 0; i < Nx + 0; i++) {
	for (int j = 0; j < Ny + 1; j++) {
	for (int k = 0; k < Nz + 1; k++) {
		if ((j == 0) || (j == Ny) || (k == 0) || (k == Nz)) {
			// 境界
			eps = Eps0;
			sig = Sig0;
		}
		else {
			// 内部:4点の平均
			eps = (epsr[i][j][k] + epsr[i][j - 1][k] + epsr[i][j][k - 1] + epsr[i][j - 1][k - 1]) / 4;
			sig = (sigm[i][j][k] + sigm[i][j - 1][k] + sigm[i][j][k - 1] + sigm[i][j - 1][k - 1]) / 4;
		}
		C2Ex[i][j][k] = 1 / (eps + (sig * eta * C * Dt));
		C1Ex[i][j][k] = C2Ex[i][j][k] * eps;
	}
	}
	}

	// Ey
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 0; j++) {
	for (int k = 0; k < Nz + 1; k++) {
		if ((k == 0) || (k == Nz) || (i == 0) || (i == Nx)) {
			// 境界
			eps = Eps0;
			sig = Sig0;
		}
		else {
			// 内部:4点の平均
			eps = (epsr[i][j][k] + epsr[i][j][k - 1] + epsr[i - 1][j][k] + epsr[i - 1][j][k - 1]) / 4;
			sig = (sigm[i][j][k] + sigm[i][j][k - 1] + sigm[i - 1][j][k] + sigm[i - 1][j][k - 1]) / 4;
		}
		C2Ey[i][j][k] = 1 / (eps + (sig * eta * C * Dt));
		C1Ey[i][j][k] = C2Ey[i][j][k] * eps;
	}
	}
	}

	// Ez
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 1; j++) {
	for (int k = 0; k < Nz + 0; k++) {
		if ((i == 0) || (i == Nx) || (j == 0) || (j == Ny)) {
			// 境界
			eps = Eps0;
			sig = Sig0;
		}
		else {
			// 内部:4点の平均
			eps = (epsr[i][j][k] + epsr[i - 1][j][k] + epsr[i][j - 1][k] + epsr[i - 1][j - 1][k]) / 4;
			sig = (sigm[i][j][k] + sigm[i - 1][j][k] + sigm[i][j - 1][k] + sigm[i - 1][j - 1][k]) / 4;
		}
		C2Ez[i][j][k] = 1 / (eps + (sig * eta * C * Dt));
		C1Ez[i][j][k] = C2Ez[i][j][k] * eps;
	}
	}
	}
}


// 平均電磁界
static void average(int ABC, int Nx, int Ny, int Nz,
	float ***Ex, float ***Ey, float ***Ez, float ***Hx, float ***Hy, float ***Hz, float fsum[])
{
	// <E>
	float sume = 0;
	int nume = 0;
	for (int i = 0; i < Nx + 0; i++) {
	for (int j = 0; j < Ny + 1; j++) {
	for (int k = 0; k < Nz + 1; k++) {
		sume += (float)fabs(Ex[i + ABC][j + ABC][k + ABC]);
		nume++;
	}
	}
	}
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 0; j++) {
	for (int k = 0; k < Nz + 1; k++) {
		sume += (float)fabs(Ey[i + ABC][j + ABC][k + ABC]);
		nume++;
	}
	}
	}
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 1; j++) {
	for (int k = 0; k < Nz + 0; k++) {
		sume += (float)fabs(Ez[i + ABC][j + ABC][k + ABC]);
		nume++;
	}
	}
	}
	fsum[0] = sume / nume;

	// <H>
	float sumh = 0;
	int numh = 0;
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 0; j++) {
	for (int k = 0; k < Nz + 0; k++) {
		sumh += (float)fabs(Hx[i + ABC][j + ABC][k + ABC]);
		numh++;
	}
	}
	}
	for (int i = 0; i < Nx + 0; i++) {
	for (int j = 0; j < Ny + 1; j++) {
	for (int k = 0; k < Nz + 0; k++) {
		sumh += (float)fabs(Hy[i + ABC][j + ABC][k + ABC]);
		numh++;
	}
	}
	}
	for (int i = 0; i < Nx + 0; i++) {
	for (int j = 0; j < Ny + 0; j++) {
	for (int k = 0; k < Nz + 1; k++) {
		sumh += (float)fabs(Hz[i + ABC][j + ABC][k + ABC]);
		numh++;
	}
	}
	}
	fsum[1] = sumh / numh;
}


// DFT加算, 送受信電界
static void dft1(int iter, int ABC, int NFreq,
	float etx, int NRx, int *iRx, int *jRx, int *kRx, int *mRx,
	float ***Ex, float ***Ey, float ***Ez, float **fcos, float **fsin,
	float *etx_r, float *etx_i, float **erx_r, float **erx_i)
{
	for (int ifreq = 0; ifreq < NFreq; ifreq++) {
		const float gcos = fcos[ifreq][iter];
		const float gsin = fsin[ifreq][iter];
		// 送信電界
		etx_r[ifreq] += gcos * etx;
		etx_i[ifreq] -= gsin * etx;
		// 受信電界
		for (int nrx = 0; nrx < NRx; nrx++) {
			float erx = 0;
			if      (mRx[nrx] == 0) {
				erx = Ex[iRx[nrx] + ABC][jRx[nrx] + ABC][kRx[nrx] + ABC];
			}
			else if (mRx[nrx] == 1) {
				erx = Ey[iRx[nrx] + ABC][jRx[nrx] + ABC][kRx[nrx] + ABC];
			}
			else if (mRx[nrx] == 2) {
				erx = Ez[iRx[nrx] + ABC][jRx[nrx] + ABC][kRx[nrx] + ABC];
			}
			erx_r[ifreq][nrx] += gcos * erx;
			erx_i[ifreq][nrx] -= gsin * erx;
		}
	}
}


// debug, 全領域調和界, 計算時間が増えるので必要なときのみ
static void dft2(int ABC, int Nx, int Ny, int Nz, float fcos, float fsin,
	float ***Ex, float ***Ey, float ***Ez,
	float ***Ex_r, float ***Ex_i, float ***Ey_r, float ***Ey_i, float ***Ez_r, float ***Ez_i)
{
	// Ex
	for (int i = 0; i < Nx + 0; i++) {
	for (int j = 0; j < Ny + 1; j++) {
	for (int k = 0; k < Nz + 1; k++) {
		Ex_r[i][j][k] += fcos * Ex[i + ABC][j + ABC][k + ABC];
		Ex_i[i][j][k] -= fsin * Ex[i + ABC][j + ABC][k + ABC];
	}
	}
	}

	// Ey
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 0; j++) {
	for (int k = 0; k < Nz + 1; k++) {
		Ey_r[i][j][k] += fcos * Ey[i + ABC][j + ABC][k + ABC];
		Ey_i[i][j][k] -= fsin * Ey[i + ABC][j + ABC][k + ABC];
	}
	}
	}

	// Ez
	for (int i = 0; i < Nx + 1; i++) {
	for (int j = 0; j < Ny + 1; j++) {
	for (int k = 0; k < Nz + 0; k++) {
		Ez_r[i][j][k] += fcos * Ez[i + ABC][j + ABC][k + ABC];
		Ez_i[i][j][k] -= fsin * Ez[i + ABC][j + ABC][k + ABC];
	}
	}
	}
}


// debug, 電界分布（調和界）
static void efield(int Nx, int Ny, int Nz,
	float ***Ex_r, float ***Ex_i, float ***Ey_r, float ***Ey_i, float ***Ez_r, float ***Ez_i, float ***Ea)
{
	for (int i = 0; i < Nx; i++) {
	for (int j = 0; j < Ny; j++) {
	for (int k = 0; k < Nz; k++) {
		// セル平均(ZXYに注意)
		const float ex_r = (Ex_r[i][j][k] + Ex_r[i    ][j + 1][k    ] + Ex_r[i    ][j    ][k + 1] + Ex_r[i    ][j + 1][k + 1]) / 4;
		const float ex_i = (Ex_i[i][j][k] + Ex_i[i    ][j + 1][k    ] + Ex_i[i    ][j    ][k + 1] + Ex_i[i    ][j + 1][k + 1]) / 4;
		const float ey_r = (Ey_r[i][j][k] + Ey_r[i    ][j    ][k + 1] + Ey_r[i + 1][j    ][k    ] + Ey_r[i + 1][j    ][k + 1]) / 4;
		const float ey_i = (Ey_i[i][j][k] + Ey_i[i    ][j    ][k + 1] + Ey_i[i + 1][j    ][k    ] + Ey_i[i + 1][j    ][k + 1]) / 4;
		const float ez_r = (Ez_r[i][j][k] + Ez_r[i + 1][j    ][k    ] + Ez_r[i    ][j + 1][k    ] + Ez_r[i + 1][j + 1][k    ]) / 4;
		const float ez_i = (Ez_i[i][j][k] + Ez_i[i + 1][j    ][k    ] + Ez_i[i    ][j + 1][k    ] + Ez_i[i + 1][j + 1][k    ]) / 4;
		Ea[k][i][j] = (float)sqrt((ex_r * ex_r) + (ex_i * ex_i)
		                        + (ey_r * ey_r) + (ey_i * ey_i)
		                        + (ez_r * ez_r) + (ez_i * ez_i));
	}
	}
	}
}
